package across.util.skn;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.ListIterator;

import across.data.Batch;
import across.data.CooperationResult;
import across.data.ItemCoverage;
import across.data.Proposal;
import across.data.RequestList;
import across.data.TransportBatch;
import across.data.TransportCfp;
import across.data.TransportRequest;
import across.util.skn.util.membership.CrispMembership;
import across.util.skn.util.membership.Membership;
import aglobe.container.transport.Address;
import aglobe.container.transport.InvisibleContainerException;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;

/**
 * This is a leader specific Coalition class. It includes the whole information about the task.
 */
public class AcrossCoalitionLeaderKnowledge extends AcrossCoalitionKnowledge<TransportCfp>
{
    /** Contains the proposal sent as an answer to the client location */
    Proposal proposal;
    /** total count of goods to transport */
    long totalCount;
    /** count of goods whose transport has completed successfully */
    long completed = 0;
    /** count of goods whose transport has failed due to banditism and fraud */
    long lost = 0;
    /** contains requests of all individual member agents, including the leader. */
    HashMap<String, TransportRequest> requests = new HashMap<String, TransportRequest>();
    /** Reference to the self-agent knowledge */
    AgentKnowledge meLeader;


    /**
     * Appends a new memeber as a part of the coalition.
     * We do apend the member here only if the TransportRequest message was successfully sent.
     */
    public void appendMember(AgentKnowledge member, TransportRequest tr)
    {
        // we avoid leader duplicity if it acts as a teammember also
        if (!meLeader.equals(member))
        {
            new CrispMembership<AgentKnowledge>(member, this);
        }
        requests.put(member.getName(), tr);
    }

    /**
     * This method is called to notify that the cargo was successfully delivered.
      * @param batch delivered cargo identification
     */
    public void transportCompleted(Batch batch)
    {
        completed += Math.abs(batch.getCount());
        //System.out.println("Completed: " + requestObject.getRequestid() + " " + batch.toString() + "\n coalStat:(total, completed, stolen) " + totalCount + " "+completed + " " +lost);
        // if completed, calculate the trustfulness and inform the coalition memebrs about tyhe results
        if (totalCount <= completed + lost)
        {
            calculatePayoff();
            distributePayoffToMembers();
        }
    }

    /**
     * This method is called to notify that the cargo was successfully delivered.
      * @param req delivered cargo identification
     */
    public void transportCompleted(RequestList req) {
          for (Iterator batchIter = req.getBatch().iterator(); batchIter.hasNext(); ) {
              Batch batch = (Batch)batchIter.next();
              this.transportCompleted(batch);
          }
    }

    /**
     * This method is called to notify the loss of a cargo.
     * @param batch lost cargo identification
     */
    public void transportFailed(Batch batch)
    {
        lost += Math.abs(batch.getCount());
        //System.out.println("Failed: " + batch.toString() + "\n coalStat:(total, completed, stolen) " + totalCount + " "+completed + " " +lost);
        if (totalCount <= completed + lost)
        {
            calculatePayoff();
            distributePayoffToMembers();
        }
    };

    /**
     * This method is called to notify the loss of a cargo.
     * @param req lost cargo identification
     */

    public void transportFailed(RequestList req) {
      for (Iterator batchIter = req.getBatch().iterator(); batchIter.hasNext(); ) {
          Batch batch = (Batch)batchIter.next();
          this.transportFailed(batch);
      }
    }

    /**
     *  Distributes payoff information to other coalition members.
     *  If the success is not complete, all members revenues are diminished by the same percentage.
     */
    private void distributePayoffToMembers()
    {
        // we distribute the payement to members.
        // If the success is not complete, all members revenues are diminished by the same percentage.
        String requestid = requestObject.getRequestid();
        double gt = getTrustfulness();
        // add the leader to the members
        new CrispMembership<AgentKnowledge>(meLeader, this);
        //System.out.println( meLeader.getName() + " **##** Cooperation Result Local Update: " + gt + "Mem. count.: " + members.size());
        //register the payoff for itself
        updateMemberTrust();
        // send it to the others
        Collection<Address> allAgentsAddress = getAllAgentsAddress(null);
        for (Membership<AgentKnowledge> membership : members) {
            // we don't send message to ourselves
            if (!membership.getMember().equals(meLeader))
            {
                CooperationResult cr = new CooperationResult();
                cr.setRequestid(requestid);
                cr.setGlobalTrust(gt);
                cr.setPayement(Math.round(requests.get(membership.getMember().getName()).getProposal().getTotalPrice()*actualPayoff));
                cr.setTeamLeader(meLeader.address);
                cr.getTeamMember().addAll(allAgentsAddress);
                Address ad = membership.getMember().address;
                Message m = Message.newInstance(MessageConstants.INFORM_RESULT, meLeader.address, ad);
                m.setProtocol(MessageConstants.INFORM);
                m.setContent(cr);
                try {
//                    System.out.println( meLeader.getName() + " **** Cooperation Result Being sent to " + ad.getName() + "\n" + m.getContent() + "\n");
                    caller.sendMessage(m);
                } catch (InvisibleContainerException e) {}
            }
        }
        // destroy the coalition
        caller.removeCoalition(requestid);
        // notify that the coalition payoff was distributed
        caller.submitTopic(across.visio.VisualAgent.TOPIC_COALITION_FINISHED, requestid, null);
    }

    public double calculatePayoff()
    {
        if ( 0 < totalCount )
        {
            actualPayoff = ((double)completed)/totalCount;
        }
        else
        {
            actualPayoff = 1;
        }
        return actualPayoff;
    };

    public double getTrustfulness()
    {
        return actualPayoff*actualPayoff;
    }


    public AcrossCoalitionLeaderKnowledge(TransportCfp transportCfp, Proposal prop, CoalitionKnowledge.Strategy strategy, CommunityKnowledge caller)
    {
        super(transportCfp, strategy, caller);
        proposal = prop;
        meLeader = caller.getOwnerAgentKnowledge();
        if (null == meLeader)
        {
            throw new RuntimeException("Agent knowledge of the owner not available during coalition negotiation.");
        }
        // pass through the proposal and get the totalCount...
        totalCount = 0;
        // go through all batches of the cfp request and compute the coverage
        for (ListIterator i = requestObject.getTransportBatch().listIterator(); i.hasNext();)
        {
            TransportBatch tb = (TransportBatch) i.next();
            // find corresponding coverage in proposal
            for (Iterator it = prop.getItemCoverage().iterator(); it.hasNext();)
            {
                ItemCoverage ic = (ItemCoverage) it.next();
                if ( ic.getItemid() == tb.getBatchid())
                {
                    totalCount += Math.round(Math.floor(ic.getCoverage()*tb.getCount()));
                    break;
                }
            }
        }
    }
}
