package across.util.ixplanparser;

import iglobe.plan.AbstractActivity;
import iglobe.plan.OrderConstraint;
import iglobe.plan.PlanConstraint;
import iglobe.plan.RescueActivity;
import iglobe.plan.RescuePlan;
import iglobe.plan.TransportActivity;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import across.agents.emergency.centre.AcrossActivity;
import across.agents.emergency.centre.AcrossPlan;
import across.agents.emergency.repair.RepairVehicleActivities;

public class IxPlanParser {

	private ParserKnowledge knowledge;
	
	public IxPlanParser(ParserKnowledge pk) {
		knowledge = pk;
	}
	
	
	public AcrossPlan ixToAcrossPlan(RescuePlan rp) {
		AcrossPlan acrossPlan = new AcrossPlan();
		Map<String,AcrossActivity> actsToIds = new HashMap<String, AcrossActivity>(); 
		List<RescueActivity> actList = rp.getActivities();
		List<TransportActivity> allActivities = new LinkedList<TransportActivity>();		
		// strip the abstract activities
		for (Iterator<RescueActivity> iter = actList.iterator(); iter.hasNext();) {
			RescueActivity ra = iter.next();
			if(ra instanceof AbstractActivity) {
				iter.remove();
				// remove the ordering constraints as well
				for (Iterator<PlanConstraint> iterator = rp.getConstraints().iterator(); iterator.hasNext();) {
					PlanConstraint pc = iterator.next();
					if(pc instanceof OrderConstraint) {
						OrderConstraint oc = (OrderConstraint) pc;
						if(oc.getFromId().equals(ra.getId()) || oc.getToId().equals(ra.getId())) {
							iterator.remove();
						}
					}
				}
			} else {
				TransportActivity ta = (TransportActivity) ra;
				allActivities.add(ta);
			}
		}
		List<TransportActivity> vehicleActivities = new LinkedList<TransportActivity>();
		List<TransportActivity> materialPickActivities = new LinkedList<TransportActivity>();
		List<TransportActivity> materialDropActivities = new LinkedList<TransportActivity>();
		
		for (TransportActivity ta : allActivities) {
			if(knowledge.vehicles.contains(ta.getResource())) {
				vehicleActivities.add(ta);
			} else if (knowledge.pickupMaterials.contains(ta.getResource())) {
				materialPickActivities.add(ta);
			} else if (knowledge.dropMaterials.contains(ta.getResource())) {
				materialDropActivities.add(ta);
			}
		}
		
		List<OrderConstraint> ocList = new LinkedList<OrderConstraint>();
		for (PlanConstraint pc : rp.getConstraints()) {
			if(pc instanceof OrderConstraint) {
				OrderConstraint oc = (OrderConstraint) pc;
				ocList.add(oc);
			}
		}
		
		// first the no-transport activities - just go somewhere and use effect
		for (TransportActivity va : vehicleActivities) {
			// the vehicle cannot transport anything
			if(knowledge.whichVehicleTransportWhichMaterial.get(va.getResource())==null) {
				String dest = va.getDestination();
				
				AcrossActivity goA = newGoto(getTo(dest), getFrom(dest), getObjId(dest));
				actsToIds.put(va.getId(), goA);
				acrossPlan.getActivitiesHandlers().put(goA, va.getResource());
				
				AcrossActivity effA = newEff(getObjId(dest));
				String effId = ""+effA.hashCode();
				actsToIds.put(effId, effA);
				acrossPlan.getActivitiesHandlers().put(effA, va.getResource());
				
				putTwoActsInsteadOfOneFirstIx(ocList, va.getId(), goA, effId, effA);
				
				acrossPlan.getDrivers().put(va.getResource(), "Repair");
			}
			// remove the schedule transport activity from constraints
			else {
				for (Iterator<OrderConstraint> iter = ocList.iterator(); iter.hasNext();) {
					OrderConstraint constraint = iter.next();
					if(constraint.getFromId().equals(va.getId()) || constraint.getToId().equals(va.getId())) {
						iter.remove();
					}
				}
			}
		}
		
		String STORAGE = "STORAGE";
		
		String LOCATION = "LOCATION";
		
		int storCount = 0;
		
		// pick activities
		for (TransportActivity pa : materialPickActivities) {
			for (TransportActivity va : vehicleActivities) {
				// if there is a vehicle that can transport this material and goes to the right place
				if(knowledge.whichVehicleTransportWhichMaterial.get(va.getResource())!=null && knowledge.whichVehicleTransportWhichMaterial.get(va.getResource()).contains(pa.getResource())) {
					String dest = pa.getDestination();
					
					String thisStorage = STORAGE+storCount;
					String thisStorageLocation = thisStorage+LOCATION;
					storCount++;
					
					AcrossActivity goToPica = newGoto(thisStorageLocation,null,null);
					String goToPicaId = ""+goToPica.hashCode();
					actsToIds.put(goToPicaId, goToPica);
					acrossPlan.getActivitiesHandlers().put(goToPica, va.getResource());
					
					AcrossActivity picA = newPick(thisStorage, pa.getResource(), pa.getAmount());
					String picaId = ""+picA.hashCode();
					actsToIds.put(picaId, picA);
					acrossPlan.getActivitiesHandlers().put(picA, va.getResource());
					
					acrossPlan.getMaterial().put(thisStorage, STORAGE);
					
					OrderConstraint oc = new OrderConstraint();
					oc.setFromId(goToPicaId);
					oc.setToId(picaId);
					ocList.add(oc);
					
					AcrossActivity goA = newGoto(getTo(dest),getFrom(dest),getObjId(dest));
					actsToIds.put(pa.getId(), goA);
					acrossPlan.getActivitiesHandlers().put(goA, va.getResource());
					
					putTwoActsInsteadOfOneLastIx(ocList, picaId, picA, pa.getId(), goA);
					
					AcrossActivity effA = newEff(getObjId(dest));
					String effId = ""+effA.hashCode();
					actsToIds.put(effId, effA);
					acrossPlan.getActivitiesHandlers().put(effA, va.getResource());
					
					putTwoActsInsteadOfOneFirstIx(ocList, pa.getId(), goA, effId, effA);
					
					acrossPlan.getDrivers().put(va.getResource(), "Repair");
				}
			}
		}
		
		// drop activities
		for (TransportActivity da : materialDropActivities) {
			for (TransportActivity va : vehicleActivities) {
				// if there is a vehicle that can transport this material and goes to the right place
				if(knowledge.whichVehicleTransportWhichMaterial.get(va.getResource())!=null && knowledge.whichVehicleTransportWhichMaterial.get(va.getResource()).contains(da.getResource())) {
					String thisStorage = STORAGE+storCount;
					String thisStorageLocation = thisStorage+LOCATION;
					storCount++;
					
					AcrossActivity dropA = newDrop(thisStorage, da.getResource(), da.getAmount());
					String dropId = ""+dropA.hashCode();
					actsToIds.put(dropId, dropA);
					acrossPlan.getActivitiesHandlers().put(dropA, va.getResource());
					acrossPlan.getMaterial().put(thisStorage, "Storage");
					
					AcrossActivity goA = newGoto(thisStorageLocation,null,null);
					actsToIds.put(da.getId(), goA);
					acrossPlan.getActivitiesHandlers().put(goA, va.getResource());
					
					putTwoActsInsteadOfOneFirstIx(ocList, da.getId(), goA, dropId, dropA);
					
					acrossPlan.getDrivers().put(va.getResource(), "Repair");
				}
			}
		}
		for (OrderConstraint constraint : ocList) {
			AcrossActivity first = actsToIds.get(constraint.getFromId());
			AcrossActivity last = actsToIds.get(constraint.getToId());
			List<AcrossActivity> aaList = acrossPlan.getPrecedences().get(last);
			if(aaList==null) {
				aaList = new ArrayList<AcrossActivity>();
				acrossPlan.getPrecedences().put(last, aaList);
			}
			aaList.add(first);
		}
		
		return acrossPlan;
	}
	
	private void putTwoActsInsteadOfOneFirstIx(List<OrderConstraint> ocList, String firstId, AcrossActivity first, String lastId, AcrossActivity last) {
		for (OrderConstraint constraint : ocList) {
			if(constraint.getFromId().equals(firstId)) {
				constraint.setFromId(lastId);
			}
		}
		OrderConstraint flOc = new OrderConstraint();		
		flOc.setFromId(firstId);
		flOc.setToId(lastId);
		ocList.add(flOc);
	}
	
	private void putTwoActsInsteadOfOneLastIx(List<OrderConstraint> ocList, String firstId, AcrossActivity first, String lastId, AcrossActivity last) {
		for (OrderConstraint constraint : ocList) {
			if(constraint.getToId().equals(lastId)) {
				constraint.setToId(firstId);
			}
		}
		OrderConstraint flOc = new OrderConstraint();
		flOc.setFromId(firstId);
		flOc.setToId(lastId);
		ocList.add(flOc);
	}
	
	private AcrossActivity newGoto(String to, String from, String objId) {
		AcrossActivity aa = new AcrossActivity();
		aa.setActivityType(RepairVehicleActivities.ACTIVITY_GOTO);
		aa.getParams().put("TO", to);
		if(from!=null && objId!=null) {
			aa.getParams().put("FROM", from);
			aa.getParams().put("OBJECT", objId);
		}
		return aa;
	}
	
	private AcrossActivity newEff(String where){
		AcrossActivity aa = new AcrossActivity();
		aa.setActivityType(RepairVehicleActivities.ACTIVITY_EFFECT);
		aa.getParams().put("WHERE", where);
		return aa;
	}
	
	private AcrossActivity newPick(String where, String mat, int num){
		AcrossActivity aa = new AcrossActivity();
		aa.setActivityType(RepairVehicleActivities.ACTIVITY_PICKUP);
		aa.getParams().put("WHERE", where);
		aa.getParams().put("MATERIAL_ID", mat);
		aa.getParams().put("AMOUNT", num);
		return aa;
	}
	
	private AcrossActivity newDrop(String where, String mat, int num) {
		AcrossActivity aa = newPick(where,mat,num);
		aa.setActivityType(RepairVehicleActivities.ACTIVITY_DROP);
		return aa;
	}
	
	private String getTo(String name) {
		if(name.indexOf("@")==-1) {
			return name;
		} else {
			return name.substring(name.indexOf("@")+"@".length(),name.indexOf("<->"));
		}
	}
	
	private String getFrom(String name) {
		if(name.indexOf("@")==-1) {
			return null;
		} else {
			return name.substring(name.indexOf("<->")+"<->".length(),name.length());
		}
	}
	
	private String getObjId(String name) {
		if(name.indexOf("@")==-1) {
			return null;
		} else {
			return name.substring(0,name.indexOf("@"));
		}
	}
}
