package across.agents.transporter.task;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;

import across.agents.transporter.TransporterAgent;
import across.data.AdvTeamProposal;
import across.data.Batch;
import across.data.CooperationResult;
import across.data.Page;
import across.data.Pages;
import across.data.Proposal;
import across.data.RequestList;
import across.data.SemiPrivateParams;
import across.data.TeamProposal;
import across.data.TransportCfp;
import across.data.TransportRequest;
import across.simulation.constants.AcrossMessageConstants;
import across.util.skn.AcrossCoalitionKnowledge;
import across.util.skn.AcrossCoalitionLeaderKnowledge;
import across.util.skn.AcrossCoalitionMemberKnowledge;
import across.util.skn.AgentKnowledge;
import across.util.skn.AllianceKnowledge;
import across.visio.VisualAgent;
import aglobe.container.task.Task;
import aglobe.container.transport.Address;
import aglobe.container.transport.InvisibleContainerException;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobex.protocol.queryif.QueryIfParticipantTask;
import aglobex.protocol.subscribe.SubscribeParticipantTask;


/**
 * Idle task for TranporterAgent. Starts other tasks to handle new conversations and handles simple conversations.
 */

public class TransporterIdleTask extends Task {

  /** This task's owner */
  protected TransporterAgent myAgent;

  private HashSet<String> uniqueMessages = new HashSet<String>();

  public TransporterIdleTask(TransporterAgent _myAgent) {
    super(_myAgent);
    this.myAgent = _myAgent;
  }

  /**
   * Filtering of incomming message based on Performative.
   * Handles messages with these protocols:
   * <li> REQUEST </li>
   * <li> INFORM </li>
   * <li> QUERY </li>
   * <li> CONTRACT_NET </li>
   * <li> SUBSCRIBE </li>
   * @param msg incomming message
   */
  protected void handleIncomingMessage(Message msg)   {

      String protocol = msg.getProtocol();

      if (MessageConstants.REQUEST.equalsIgnoreCase(protocol))  {        
          handleRequestProtocol(msg);
      } else if (MessageConstants.INFORM.equalsIgnoreCase(protocol)) {   
          handleInformProtocol(msg);
      } else if (MessageConstants.QUERY.equalsIgnoreCase(protocol)) {    
          handleQueryProtocol(msg);
      } else if (MessageConstants.CONTRACT_NET.equalsIgnoreCase(protocol)) {    
          handleCNPProtocol(msg);
      } else if (MessageConstants.SUBSCRIBE.equalsIgnoreCase(protocol)) {    
          handleSubscribeProtocol(msg);
      } else {
             myAgent.getLogger().warning("!!!! unexpected message:" + msg);
             sendNotUnderstood(msg, "!!!! unexpected message:");
        }
  }

  // TODO - doc
  /**
   * Handles subscribe protocol.  There are these cases of subscribe protocol:
   * <li> Content = TransporterAgent.PUBLIC_INFO_REQUEST </li>
   * <li> Content instanceof SemiPrivateParams</li>
   * @param msg
   */
    private void handleSubscribeProtocol(Message msg) {
      // don't subscribe itself
        if (msg.getSender() == myAgent.getAddress()) {
        	return;
        }
        // Public info subscribe
        if (AcrossMessageConstants.PUBLIC_INFO_REQUEST.equals(msg.getContent()))  {
            SubscribeParticipantTask task = new SubscribeParticipantTask(myAgent, msg, false) {
                protected void processSubscribe(Message subscribeMessage) {
                    Page pg = new Page();
                    pg.setName(myAgent.getAddress().getName());
                    pg.setAddress(myAgent.getAddress());
                    pg.setPublicParams(myAgent.publicParams);
                    informResult(pg);
                }
            };
            task.start();
            myAgent.publicSubscribers.add(task);
        }
        // Semi-Private info subscribe
        else if (msg.getContent() instanceof SemiPrivateParams) {
            // receive semiprivate params of the new alliance member and send back my semiprivate params
            AllianceKnowledge ak = myAgent.community.getAlliance(myAgent.myAlliance);
            AgentKnowledge senkn = myAgent.community.getAgent(msg.getSender().getName());
            // check if the agent is known and is my alliance's member
            if (null != ak && null != senkn && ak.containsMember(senkn))  {
                SubscribeParticipantTask task = new SubscribeParticipantTask(myAgent,msg,false) {
                    protected void processSubscribe(Message subscribeMessage) {
//                   	 subscribe to agent's semi-provate info if not already done...
                    	AgentKnowledge senkn = myAgent.community.getAgent(subscribeMessage.getSender().getName());
                    	if (null == senkn.semiPrivateParams)
                    	{
                    		myAgent.subscribeAndSendSemiPrivateInfo(senkn.address);
                    	}
                    	// update the knowledge with received params
                    	myAgent.updateSemiPrivate(subscribeMessage.getSender(), (SemiPrivateParams) subscribeMessage.getContent());
                    	// send own spp as an answer
                        informResult(myAgent.semiPrivateParams);
                    }

                    /**Before sending the info, we check whether the recipient is in my alliance or not...
                     * @param result Semi-Private information about the owner, updated if possible...*/
                    public void informResult(Object result) {
                        AllianceKnowledge ak = myAgent.community.getAlliance(myAgent.myAlliance);
                        AgentKnowledge senkn = myAgent.community.getAgent(subscribeMessage.getSender().getName());
                        // check if the agent is known and is my alliance member - this may have changed since the subscribe...
                        if (null != ak && null != senkn && ak.containsMember(senkn))
                        {
                            super.informResult(result);
                        }
                    }
                };
                task.start();
                myAgent.semiPrivateSubscribers.add(task);
            }
        }
        else {
            handleGeneralSubscribe(msg);
        }
     }

  /**
   * Override this in inherited tasks
   *
   * @param msg Message
   */
  protected void handleGeneralSubscribe(Message msg)  {
        sendNotUnderstood(msg, "Wrong Performative or Content");
  }

  /**
   * Handles Contract Net Protocol
   * @param msg
   */
  public void handleCNPProtocol(Message msg) {
	  
	  if(!MessageConstants.CONTRACT_NET.equalsIgnoreCase(msg.getProtocol())) {
		  return;
	  }

      String performative = msg.getPerformative();
      Object content = msg.getContent();
      if (MessageConstants.CFP.equalsIgnoreCase(performative) && content instanceof TransportCfp) {
      	if (! myAgent.receivedCFP(msg,null)) {
        		new CNPTransportBidAnswer(myAgent, CNPTransportBidAnswer.TIMEOUT, msg);
        	}
      } else {
         handleGeneralCNP(msg);
      }
   }

  /**
   * Override this in inherited tasks
   *
   * @param msg Message
   */
   protected void handleGeneralCNP(Message msg)  {
        sendNotUnderstood(msg, "Wrong Performative or Content");
    }

   // TODO - doc
    /**
     * Handles request protocol.  These cases are handled:
     * <li> performative = INFORM_DONE && content instanceof RequestList </li>
     * <li> performative = INFORM_DONE && content instanceof Batch </li>
     * <li> performative = FAILURE && content instanceof RequestList </li>
     * <li> performative = AGREE</li>
     * <li> performative = REQUEST</li>
     * @param msg incomming message
     */
    private void handleRequestProtocol(Message msg) {
        Object content = msg.getContent();
        
        if (msg.getPerformative().equalsIgnoreCase(MessageConstants.INFORM_DONE) && msg.getContent() instanceof RequestList) {
            // Done received from sub-contractor when the proposal was accepted by village and subcontractors were notified
            RequestList rl = (RequestList) content;
            AcrossCoalitionKnowledge c = myAgent.community.getCoalition(rl.getRequestid());
            if (null != c) {
                c.transportCompleted(rl);
            } else {
                myAgent.getLogger().warning( myAgent.getName() + ": Task completion for unknown coalition by driver: " + msg.getSender() + " Task: " + rl);
            }
        } else if (msg.getPerformative().equalsIgnoreCase(MessageConstants.INFORM_DONE) && msg.getContent() instanceof Batch) {
            
            if (!uniqueMessages.contains(msg.getConversationID())) {
                uniqueMessages.add(msg.getConversationID());
            } else {
                return;
            }

          // Done received from sub-contractor when the proposal was accepted by village and subcontractors were notified
          Batch batch = (Batch) content;
          AcrossCoalitionKnowledge c = myAgent.community.getCoalition(msg.getReason());
          if (null != c) {
              c.transportCompleted(batch);
          }
          else {
              myAgent.getLogger().warning( myAgent.getName() + ": Task completion for unknown coalition by driver: " + msg.getSender() + " Task: " + batch);
          }

        } else if (msg.getPerformative().equalsIgnoreCase(MessageConstants.FAILURE) && content instanceof RequestList) {
            RequestList rl = (RequestList)content;
            // cover up fraud
            if (rl.getRequestTime() == -1) {
                rl.setDeliveryLocation(rl.getDeliveryAddress().getName());
               msg.setContent(rl);
            }

            Message fail = Message.newInstance(MessageConstants.FAILURE,myAgent.getAddress(),rl.getDeliveryAddress());
            fail.setProtocol(MessageConstants.REQUEST);
            fail.setContent(rl);
            //TODO !!!! dodelat resendovani v pripade ze transportAgent nesedi na Location, ted na sebe vzdy vidi...
            try {
                sendMessage(fail);
            }
            catch (InvisibleContainerException e) {}
            // we will now inform the community knowledge...
            AcrossCoalitionKnowledge c = myAgent.community.getCoalition(rl.getRequestid());
            if (null != c) {
                c.transportFailed(rl);
            }
            else {
                myAgent.getLogger().warning( myAgent.getName() + ": Task failure for unknown coalition by driver: " + msg.getSender() + " Task: " + rl);
            }
        } else if (msg.getPerformative().equalsIgnoreCase(MessageConstants.AGREE)) {
            // AGREE received from sub-contractor when the proposal was accepted by village and subcontractors were notified
            // answer to the request sent from CNPTransportBidAnswer proposalAccepted
        } else if ((msg.getPerformative().equalsIgnoreCase(MessageConstants.REQUEST))) {
            // Transport Requested
            if (content instanceof TransportRequest)  {
                new TransportAcceptedTask(myAgent, msg, false);
            } else if (content instanceof Page) {
                // join alliance
                RequstJoinAllianceParticipantTask task = new RequstJoinAllianceParticipantTask(myAgent, msg);
                task.start();
            } else if (content instanceof String) {
                // query for alliance merge - as a passive member of one alliance...nasty hack...
                AllianceKnowledge ak = myAgent.community.getAlliance((String) content);
                Message m = msg.getReply();
                if (myAgent.allianceAcceptable(ak)) {
                    m.setPerformative(MessageConstants.INFORM_DONE);
                } else {
                    m.setPerformative(MessageConstants.INFORM_RESULT);//bad, very bad
                }
                try {
                    sendMessage(m);
                } catch (InvisibleContainerException e){}
            }
            // request for merge from other alliance - act as a local intermediary...
            else if (content instanceof Pages)  {
                // will check with other agents in the alliance whether they agree with the fusion
                new RequestMergeAllianceTask(myAgent, msg);
            }
            // kick the agent in the content from the alliance - if you are kicked, remove yourself from visio too
            else if (content instanceof Address) {
              Address kickedAllianceMember = (Address) content;
              // kicked myself
              if (myAgent.getAddress().equals(kickedAllianceMember)) {
                AllianceKnowledge ak = myAgent.community.getAlliance(myAgent.myAlliance);
                ak.removeMember(myAgent.community.getOwnerAgentKnowledge());
                myAgent.community.getOwnerAgentKnowledge().alliance = null;
                myAgent.createNewAlliance();
                // send topic that the agent was kicked
                myAgent.gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.
                                             TOPIC_VISIO_ACTION,
                                             Byte.toString(across.visio.oldvisio.
                    VisioConnectionAgent.ACTION_POLL_EXPELLED), myAgent.getAddress().toString());

              }
              // someone else removed
              else {
                AgentKnowledge knowledgeAboutKickedAgent = myAgent.community.
                    getAgent(kickedAllianceMember.getName());
                myAgent.community.getOwnerAgentKnowledge().alliance.removeMember(
                    knowledgeAboutKickedAgent);
              }
            } else if (content instanceof AdvTeamProposal) {
            	myAgent.receivedAdvTeamRequest(msg);

            }  else {
              handleGeneralRequest(msg);
            }
          } else { // unknown performative
            handleGeneralRequest(msg);
          }
    }

  /**
   * Override this in inherited tasks
   *
   * @param msg Message
   */
  protected void handleGeneralRequest(Message msg) {
        sendNotUnderstood(msg, "Unexpected performative in request protocol");
  }
  
  // TODO - doc
  /**
   * Handles inform protocol.  These cases:
   * <li> performative = INFORM_RESULT && content instanceof CooperationResult </li>
   * <li> performative = INFORM_DONE && content instanceof Batch && reason!=null </li>
   * <li> performative = FAILURE && content instanceof Batch && reason!=null </li>
   * @param msg incomming message
   */
  private void handleInformProtocol(Message msg) {
	  
    if(!MessageConstants.INFORM.equalsIgnoreCase(msg.getProtocol())){
    	return;
    }

    String performative = msg.getPerformative();
    Object content = msg.getContent();
    // Coalition member has received the result of cooperation from leader
    if (MessageConstants.INFORM_RESULT.equalsIgnoreCase(performative) && content instanceof CooperationResult) {
        CooperationResult cooperationResult = (CooperationResult)content;
        AcrossCoalitionMemberKnowledge coalition = (AcrossCoalitionMemberKnowledge) myAgent.community.getCoalition(cooperationResult.getRequestid());
        if (null != coalition) {
            coalition.handleCoalitionTerminated(cooperationResult);
        } else {
            myAgent.getLogger().warning( myAgent.getName() + ": Task result for unknown coalition from leader: " + msg.getSender() + " Task: " + cooperationResult.getRequestid());
        }
    }
    // Coalition leader's update from member that a delivery of a batch was successfull
    else  if ((MessageConstants.INFORM_DONE.equalsIgnoreCase(performative)) && (content instanceof Batch) && (msg.getReason() != null))  {
        AcrossCoalitionLeaderKnowledge coalition = (AcrossCoalitionLeaderKnowledge) myAgent.community.getCoalition(msg.getReason());
        if (null != coalition) {
            coalition.transportCompleted((Batch)content);
        } else {
            myAgent.getLogger().warning( myAgent.getName() + ": Task result (OK) for unknown coalition received from member: " + msg.getSender() + " Task: " + content);
        }
    }
    // Coalition leader's update from member that a batch was stolen
    else if (MessageConstants.FAILURE.equalsIgnoreCase(performative) && content instanceof Batch && msg.getReason() != null)  {
        AcrossCoalitionLeaderKnowledge coalition = (AcrossCoalitionLeaderKnowledge) myAgent.community.getCoalition(msg.getReason());
        if (null != coalition) {
            coalition.transportFailed((Batch)content);
        }
        else {
            myAgent.getLogger().warning( myAgent.getName() + ": Task result (Failure) for unknown coalition received from member: " + msg.getSender() + " Task: " + content);
        }
    } else {
        handleGeneralInform(msg);
    }
  }

  protected void handleGeneralInform(Message msg) {
        myAgent.getLogger().warning(myAgent.getName() + "INFORM message with unexpected performative or structure received.\n" + msg );
  }

  // TODO - doc
  /**
   * Handles query protocol.  These cases:
   * <li> performative = QUERY_IF && content instanceof Message </li>
   * <li> performative = QUERY_IF && content instanceof Address </li>
   * <li> performative = QUERY_REF && content instanceof TeamProposal </li>
   * <li> performative = QUERY_REF && content instanceof AdvTeamProposal </li>
   */
   protected void handleQueryProtocol(Message msg) {
	   String performative = msg.getPerformative();
       Object content = msg.getContent();
       // queryIf accept new alliance member
       if (MessageConstants.QUERY_IF.equalsIgnoreCase(performative) && content instanceof Message) {
           if (myAgent.acceptAllianceMember( (Message) content, msg.getSender())) {
               informTrue(msg);
           } else {
               informFalse(msg);
           }
       } else if (MessageConstants.QUERY_IF.equalsIgnoreCase(performative) && content instanceof Address) {
          QueryIfParticipantTask qiit = new QueryIfParticipantTask(this.myAgent, msg, false) {
		        protected void processQuery(Message queryMessage) {
		
		          Address candidate  = (Address)queryMessage.getContent();
		          AgentKnowledge konwledgeAboutCandidate = myAgent.community.getAgent(candidate.getName());
		
		          if (konwledgeAboutCandidate.isTrusted() == AgentKnowledge.TrustDecision.DISTRUSTED) {
		            // i don't want him in my alliance
		            informTrue();
		            TransporterIdleTask.this.myAgent.gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_ACTION, Byte.toString(across.visio.oldvisio.VisioConnectionAgent.ACTION_POLL_YES) ,myAgent.getAddress().toString());
		          } else {
		            informFalse();
		            TransporterIdleTask.this.myAgent.gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_ACTION, Byte.toString(across.visio.oldvisio.VisioConnectionAgent.ACTION_POLL_NO) ,myAgent.getAddress().toString());
		          }
		
		        }

          };
          qiit.start();
        } else if (MessageConstants.QUERY_REF.equalsIgnoreCase(performative) && content instanceof TeamProposal) {
            aglobex.protocol.queryref.QueryRefParticipantTask partTask = new aglobex.protocol.queryref.QueryRefParticipantTask(myAgent, msg, false) {
                protected void processQuery(Message queryMessage) {
                    if (! (queryMessage.getContent() instanceof TeamProposal)) {
                        refuse();
                        return;
                    }
                    Proposal proposal = myAgent.createTeamMemberProposal((TeamProposal) queryMessage.getContent());
                    // do GUI and reporting
                    List data = new LinkedList();
                    data.add(proposal);
                    data.add(queryMessage.getContent());
                    data.add(myAgent.getName());
                    myAgent.gisShell.submitTopic(VisualAgent.TOPIC_TEAM_MEMBER_PROPOSAL, data);
                    myAgent.gui.addMyProposal(proposal);
                    if (null != proposal) {
                      informResult(proposal);
                    }
                    else {
                      refuse();
                    }
                    cancelTask();
                }
            };
            partTask.start();
        } else if (MessageConstants.QUERY_REF.equalsIgnoreCase(performative) && content instanceof AdvTeamProposal) {
        	// this is used in planning-enabled transporters (class PlanMemTransporterAgent)
        	myAgent.receivedAdvTeamProposal(msg);
        } else {
            handleGeneralQuery(msg);
            return;
        }
  }

    protected void handleGeneralQuery(Message msg) {
        sendNotUnderstood(msg,"bad performative or content, QUERY_REF or QUERY_IF expected");
        return;
    }

  /**
   * use to send inform Result message
   *
   * @param queryMessage Message
   * @param result Object
   */
  protected void informResult(Message queryMessage, Object result) {
    Message re = queryMessage.getReply();
    re.setPerformative(MessageConstants.INFORM_RESULT);
    re.setContent(result);
    try {
      sendMessage(re);
    } catch (InvisibleContainerException ex) {
    }
  }

  /**
   * use to send positive informTF message
   *
   * @param queryMessage Message
   */
  protected void informTrue(Message queryMessage) {
    Message re = queryMessage.getReply();
    re.setPerformative(MessageConstants.INFORM_TF);
    re.setContent(aglobex.protocol.queryif.QueryIfInitiatorTask.TRUE);
    try {
      sendMessage(re);
    } catch (InvisibleContainerException ex) {
    }
  }

  /**
   * use to send negative informTF message
   *
   * @param queryMessage Message
   */
  protected void informFalse(Message queryMessage) {
    Message re = queryMessage.getReply();
    re.setPerformative(MessageConstants.INFORM_TF);
    re.setContent(aglobex.protocol.queryif.QueryIfInitiatorTask.FALSE);
    try {
      sendMessage(re);
    } catch (InvisibleContainerException ex) {
    }
  }
}
