package across.agents.transporter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import across.agents.driver.DriverAgent;
import across.agents.location.LocationAgent;
import across.agents.transporter.gui.TransporterAgentGUI;
import across.agents.transporter.task.TransporterIdleTask;
import across.agents.transporter.util.checker.DebugRestrictionsChecker;
import across.agents.transporter.util.checker.RestrictionsChecker;
import across.data.AcrossMapNodes;
import across.data.ItemCoverage;
import across.data.Page;
import across.data.Param;
import across.data.PrivateParams;
import across.data.Proposal;
import across.data.PublicParams;
import across.data.Resource;
import across.data.SemiPrivateParams;
import across.data.TeamProposal;
import across.data.TransportBatch;
import across.data.TransportCfp;
import across.simulation.constants.AcrossDataConstants;
import across.simulation.constants.AcrossMessageConstants;
import across.util.GoodsConstants;
import across.util.InterLinguaTools;
import across.util.skn.AgentKnowledge;
import across.util.skn.AllianceKnowledge;
import across.util.skn.CommunityKnowledge;
import across.util.skn.ServiceKnowledge;
import across.util.skn.listeners.NewAgentRegisteredListener;
import across.util.skn.update.UpdateListener;
import across.visio.oldvisio.VisioConnectionAgent;
import aglobe.container.agent.CMAgent;
import aglobe.container.sysservice.directory.DirectoryException;
import aglobe.container.sysservice.directory.DirectoryService;
import aglobe.container.transport.Address;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.Message;
import aglobe.service.gis.client.GISClientService;
import aglobe.service.gis.client.GISTopicListener;
import aglobex.protocol.request.RequestInitiatorTask;
import aglobex.protocol.subscribe.SubscribeInitiatorTask;
import aglobex.protocol.subscribe.SubscribeParticipantTask;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.entity.EntityDescriptor;
import aglobex.simulation.protocol.cnp.CNPParticipantTask;
import aglobex.simulation.protocol.cnp.CNPTaskOwner;

/**
 * Created by IntelliJ IDEA.
 * User: rehakm1
 * Date: 20.5.2004
 * Time: 13:39:21
 * 
 */
public class TransporterAgent
    extends CMAgent
    implements CNPTaskOwner, GISTopicListener, NewAgentRegisteredListener,ClientServerTopicConstants {

  /** Agent type. */
  public final static String TYPE = "Transporter";
  
  /** Checks the restrictions on building alliances. */
  public RestrictionsChecker restrictionsChecker;
  
  /** Probably says whether this TransporterAgent is a renegade - semsch??? */
  public boolean isRenegade = false;
  
  /** Probably the probablity with which this agent is likely to steal goods - semsch??? */
  public double fraudProbability = 0;

  /** Agent's public parameters */
  public PublicParams publicParams;
  
  /** Agent's semi-private parameters */
  public SemiPrivateParams semiPrivateParams;
  
  /** Agent's private parameters */
  public PrivateParams privateParams;

  /** Name of my alliance. */
  public String myAlliance = null;
  
  /** Name of the fake alliance in the social knowledge used to represent my drivers. */
  public String myDriversAlliance;

  /** GIS service shell - used for communication with the GIS client service of the local container. */
  public GISClientService.Shell gisShell;
  
  /** Directory service shell - for communication with the DirectoryService. */
  protected DirectoryService.Shell dsShell;

  /** Contains the time ticks as recieved from time server */
  protected long currentTime = 0;

  /** Stores the knowledge about other agents and entities in the community */
  public CommunityKnowledge community = new CommunityKnowledge(this);

  /** GUI of the agent */
  public TransporterAgentGUI gui;

  /** agent subscribed to receive public info updates - contains SubscribeParticipantTask objects*/
  public Collection<SubscribeParticipantTask> publicSubscribers = new LinkedList<SubscribeParticipantTask>();
  
  /** agents subscribed to receive Semi-private info updates - contains SubscribeParticipantTask objects*/
  public Collection<SubscribeParticipantTask> semiPrivateSubscribers = new ArrayList<SubscribeParticipantTask>();
  
  /** Agents subscribed to receive the results of cooperations from coalition leaders.*/
  public Map<String,SubscribeParticipantTask> reputationObservationSubscribers = new LinkedHashMap<String,SubscribeParticipantTask>();
  
  /** Agents subscribed to receive processed trustfulness values.*/
  public Map<String,SubscribeParticipantTask> reputationValuesSubscribers = new LinkedHashMap<String,SubscribeParticipantTask>();

  /** Nodes of the map */
  public AcrossMapNodes acrossMapNodes = null;

  /** for each transported whether is alloeed to cover given batch from other alliances */
  public boolean subcontractAllowed = false;

  // TODO - remove?
  /** Specifies how many times are we going to try to change the alliance - heuristic rule */
  private long changeCount = 4;
  
  /** Descriptor (configuration) of this entity */
  protected EntityDescriptor entityDescriptor;
  
  /** Node specifying the location where the TransporterAgent is situated */
  protected String location;

  /**
   * This method initializes the agent in these steps:
   * <li> Sets up gis and directory service shells </li>
   * <li> Subscribes topics at the GIS. (also topics for the simulation) </li>
   * <li> Sets up gui </li>
   * <li> Sets idle task. </li>
   */
  public void init(final AgentInfo ai, int initState) {
	 
	gisShell = (GISClientService.Shell) getContainer().getServiceManager().getService(this,GISClientService.SERVICENAME);
	if(gisShell==null) {
		logSevere("GISSevice unavailiable!");
		this.stop();
		return;
	}
    dsShell = (DirectoryService.Shell) getContainer().getServiceManager().getService(this,DirectoryService.SERVICENAME);
    if(gisShell==null) {
		logSevere("DirectorySevice unavailiable!");
		this.stop();
		return;
	}
    
	gisShell.subscribeTopic(TOPIC_ENTITY_CONNECTION, this);
	gisShell.subscribeTopic(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND, this);
	
	myDriversAlliance = getName() + "_DRIVERS";

	// XXX - gui
    gui = new TransporterAgentGUI(this, community);
    gui.setTitle(ai.getReadableName());
    gui.setBounds(0, 0, 400, 300);
    
    setIdleTask(new TransporterIdleTask(this));

  }

  /**
   *  Use this to override default checker as a first thing in derived classes 
   */
   protected void setupRestrictionChecker() {
     restrictionsChecker = new DebugRestrictionsChecker(this);
   }
   
   /**
    * Registers with the DirectoryService and subscribes for some of its topics.
    */
   protected void directoryRegisterAndSubscribe() {
	   
    community.subscribeNewAgentRegisteredListener(this);
    
    UpdateListener ulist = new UpdateListener(community, this);

    if (null != dsShell && null != gisShell) {
        Collection<String> transporters = new ArrayList<String>();
        transporters.add(TransporterAgent.TYPE);
        try {
            dsShell.register(this, transporters);
            dsShell.subscribe(ulist, TransporterAgent.TYPE);
            dsShell.subscribe(ulist, LocationAgent.TYPE);
            dsShell.subscribe(ulist, DriverAgent.TYPE);
        }
        catch (DirectoryException e) {
            logger.severe("Directory Service exception during registration.: " + e + e.getMessage()+ e.getStackTrace());
        }
    }
    else {
        logger.severe("Directory Service not found on container: " + getContainer().getContainerName());
    }
   }
   
   /** Sends updated public info to all subscribers. Updates the gui and visio as well. */
   private void updateMyPublicInfo() {
	   
   	  if(publicParams==null) {
   		 logSevere("Null public parameters at: "+ getName());
   		 return;
   	   }
   	  
       gui.setPublic(publicParams);

       Page page = new Page();
       page.setName(getAddress().getName());
       page.setAddress(getAddress());
       page.setPublicParams(publicParams);
       page.getPublicParams().setType(TYPE);
       
       // update information about itself
       community.agentPageUpdated(getAddress(), publicParams, getContainer().getContainerName());

       restrictionsChecker.agentPageUpdated(page);

       for (SubscribeParticipantTask spt : semiPrivateSubscribers) {
           spt.informResult(page);
       }
       gisShell.submitTopic(across.visio.oldvisio.VisioConnectionAgent.TOPIC_PAGE_UPDATE, page);
       gisShell.submitTopic(VisioConnectionAgent.TOPIC_VISIO_INFO, "<c00FF00>" + getContainer().getContainerName(),null);
   }
    
    /**
     * Handles the incoming configuration
     *
     */
    private void handleConfiguration() {
    	
  	  if(entityDescriptor==null) {
  		  logSevere("Cannot configure transporter");
  		  throw new RuntimeException("Cannot configure transporter");
  	  }
  	  
  	  setupRestrictionChecker();
  	  
  	  privateParams = (PrivateParams) entityDescriptor.confObjects.get(AcrossDataConstants.TRANSPORTER_PRIVATE_PARAMS);
  	  publicParams = (PublicParams) entityDescriptor.confObjects.get(AcrossDataConstants.TRANSPORTER_PUBLIC_PARAMS);
  	  location = entityDescriptor.confParamsString.get(AcrossDataConstants.ENTITY_START_NODE);
  	  
 	  setupSemiPrivateFromPrivate();
  	  
  	  // XXX - gui
  	  gui.setPublic(publicParams);

      // Get list of node names
  	  acrossMapNodes = (AcrossMapNodes) entityDescriptor.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_NODES);

  	  for (Param param : (List<Param>) publicParams.getParam() ) {
  		  if (AcrossDataConstants.TRANSPORTER_SUBCONTRACT_ALLOWED.equalsIgnoreCase(param.getName())) {
  			  this.subcontractAllowed = Boolean.valueOf(param.getValue());
  	      } 
  	  }
  	  
  	  community.agentPageUpdated(getAddress(), publicParams, getContainer().getContainerName());
  	  
  	  // put my semi private to the skn
  	  AgentKnowledge ak = community.getOwnerAgentKnowledge();
  	  if (null != ak) {
  	    ak.updateSemiPrivateParams(semiPrivateParams);
  	  }
  	  
  	  // transporter automatically create its own alliance
  	  createNewAlliance();
  	  
  	  // Task for changing alliances.
  	  this.scheduleEvent(new Runnable() {
		  
		  public void run() {
            startTaskChangeAlliance();        			  
		  }
		  
	  }, (long) (2000*Math.random()+2000));
  	    
  	  directoryRegisterAndSubscribe();
    }
    

    /**
     * Handles these topics:
     * <li> TOPIC_ENTITY_CONNECTION - handles configuration. </li>
     * <li> TOPIC_SIMULATION_TIME_UPDATE_1_SECOND - handles simulation time. </li>
     * @param topic String
     * @param content Object
     * @param reason String
     */
    public void handleTopic(String topic, Object content, String reason) {
  	  if(topic.equalsIgnoreCase(TOPIC_ENTITY_CONNECTION)) {
  		  if(content instanceof EntityDescriptor) {
  			  this.entityDescriptor = (EntityDescriptor) content;
  			  handleConfiguration();
  		  }
  	  } else if(topic.equalsIgnoreCase(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND)) {
  		  updateMyPublicInfo();
  	      currentTime = Long.parseLong( (String) content);
  	  } else {
    	  logSevere("Unexpected topic: "+topic);
      }
    }

    /**
     * This method is called by social knowledge to notify its owner when new
     * agent (identified by unique name.) becomes known to the agent for the
     * first time.
     * 
     * This method is mainly used for hooking up with the driver agents.  It also tries to 
     * get public info from newly registered TransporterAgents.
     *
     * @param agent AgentKnowledge
     */
    public void handleNewAgentRegistered(AgentKnowledge agent) {  
      for (ServiceKnowledge serviceKnowledge : agent.servicesProvided) {
          if (DriverAgent.TYPE.equalsIgnoreCase(serviceKnowledge.serviceName)) {
              //driver agent found... start the subscribe if it is my agent
        	  for (Resource resource : privateParams.getResource()) {
                  if (agent.address.getName().equalsIgnoreCase(resource.getResid()) ) {
                	  
                      SubscribeInitiatorTask task = new SubscribeInitiatorTask(this, agent.address, location, false) {
                    	  
                          boolean agreed = false;
                          AgentKnowledge ak = null;
                          
                          /** 
                           * Receives "I'am available message from drivers. Marks them as available in resources." 
                           */
                          protected void subscribeInformResult(Object object) {
                              // to make sure that info is ready...
                              if (!agreed) {
                                subscribeAgreed();
                              }
                              if (null != ak && null != ak.alliance && ak.alliance.getName().equalsIgnoreCase(myDriversAlliance))  {
                                  Resource car = null;
                                  // find the driver in the private info
                                  for (Iterator rit = privateParams.getResource().iterator(); rit.hasNext();) {
                                      Resource carRes = (Resource) rit.next();
                                      if (carRes.getResid().equalsIgnoreCase(participant.getName())) {
                                          car = carRes;
                                          break;
                                      }
                                  }
                                  if (null != car) {
                                      car.setAvailabilityTime(currentTime);
                                      car.setLastLocation((String) object);
                                      setupSemiPrivateFromPrivate();
                                      broadcastSemiPrivateToAlliance();
                                  }
                              } else {
                                  logger.severe(" ++++ Received information from driver, unused. Transporter: " + getAddress().getName() + " driver: " + participant + " loc: " + object);
                              }
                          }
                          
                          protected void subscribeAgreed() {
                              AgentKnowledge driver = community.agents.get(participant.getName());
                              if (null != driver) {
                                  driver.enterAlliance( myDriversAlliance );
                                  ak = community.getAgent(participant.getName());
                                  agreed = true;
                              }
                          }

                          protected void subscribeRefused() {
                              logger.severe("Driver " + subscribeContent + " has refused my subscribe.");
                          }
                      };
                      task.start();
                  }
              }
          } else if (TransporterAgent.TYPE.equalsIgnoreCase(serviceKnowledge.serviceName) && !getAddress().equals(agent.address)) {
              SubscribeInitiatorTask task = new SubscribeInitiatorTask(this, agent.address, AcrossMessageConstants.PUBLIC_INFO_REQUEST, false) {
            	  
                  protected void subscribeInformResult(Object object) {
                      Page page = (Page)object;
                      community.agentPageUpdated(page.getAddress(), page.getPublicParams());
                      restrictionsChecker.agentPageUpdated(page);
                  }

                  protected void subscribeAgreed() {
                  }

                  protected void subscribeRefused() {
                      logger.severe("Transporter " + subscribeContent + " has refused my subscribe.");
                  }
              };
              task.start();
          }
      }
    }

    /**
     * This function generates new semiPrivateParams from private params. It shall be called upon each
     * update of privateParams that shall be reflected in semi-private (alliance accessible) info.
     * Besoides this, it also registres newly detected or updated resource changes with the planner.
     */
    public void setupSemiPrivateFromPrivate() {
    	semiPrivateParams = new SemiPrivateParams();
        // obtain map with all types Resources...
        Map<String,Resource> resbytype = GoodsConstants.getCargoTypesResourceMap();
        // parse all trucks and sum capacity by type
        for (Resource res : privateParams.getResource()) {
            Collection<String> availableActions = restrictionsChecker.getAvailableActions(res);
        	if (availableActions != null && availableActions.size() > 0) {
        		res.getAvailableAction().addAll(availableActions);
			}
            Resource newRes = (Resource)resbytype.get(res.getType());
            if(null != newRes) {
                if (res.getAvailabilityTime() <= currentTime) {
                    newRes.setCapacity(newRes.getCapacity() + res.getCapacity());
                }
            } else {
                logger.warning("Unknown resource type in private params: " + res.getType());
            }
        }
        // prepare list of non-zero types
        for (Resource resource : resbytype.values()) {
            if (0 < resource.getCapacity())
            {
                // name the resource before sending ...
            	resource.setResid(getName() + resource.getType());
            	Collection<String> availableActions = restrictionsChecker.getAvailableActions(resource);
            	if (availableActions != null && availableActions.size() > 0)
				{
            		resource.getAvailableAction().addAll(availableActions);
				}
            	semiPrivateParams.getResource().add(resource);
            }
        }
        // XXX - gui
        gui.setSemi(semiPrivateParams);
    }

  /*****************************************
   * Methods used to maintain alliances
   ****************************************/

  /**
   * set the alliance of this agent. It also send actualized information to
   * other agents
   *
   * @param alliance String
   */
  protected void setAlliance(String alliance) {
    logger.fine(getName() + " new alliance ->"+alliance);
    myAlliance = alliance;
    boolean changed = false;
      for (Param param : publicParams.getParam()) {
          if (AcrossDataConstants.TRANSPORTER_PARAM_ALLIANCE_NAME.equalsIgnoreCase(param.getName())) {
              param.setValue(alliance);
              changed = true;
          }
      }
      if (!changed) {
          Param newParam = new Param();
          newParam.setName(AcrossDataConstants.TRANSPORTER_PARAM_ALLIANCE_NAME);
          newParam.setValue(alliance);
          publicParams.getParam().add(newParam);
      }
    community.setAlliance(alliance);
    updateMyPublicInfo();
  }

  /**
   * returns the name of alliance the agent with page is member of. returns
   * <code>null</code> if page does not contain the name of alliance
   *
   * @param page Page
   * @return String param value, null if not found
   */
  public static String getAlliance(Page page) {
    return getPageValue(page, AcrossDataConstants.TRANSPORTER_PARAM_ALLIANCE_NAME);
  }

  /**
   * Gets the parameter paramName from the page.
   * @param page
   * @param paramName
   * @return
   */
  public static String getPageValue(Page page, String paramName) {
    return  InterLinguaTools.getParamValue(page.getPublicParams().getParam(), paramName);
  }
  
    /**
     * Says whether this TransporterAgent is an alliance leader - it determines it from the name
     * of the alliance.
     * @return
     */
    protected boolean isAllianceLeader() {
        return myAlliance.startsWith(getAddress().getName());
    }

    /**
     * finds possible alliances and tries to join them using subscribe protocol.
     * Whenever Refuse reply comes for its subscribe message, next possible alliance is
     * subscribed. If no alliance agree, new alliance is created.
     */
    protected void startTaskChangeAlliance() {
        // we limit the number of times we try to join the alliance
        if ( changeCount <= 0) {
            return;
        } else {
            changeCount--;
            AllianceKnowledge ak = community.getAlliance(myAlliance);
            // make sure that we don't break an existing alliance
            if (null == ak || ak.getAllMembersCount() == 1) {
                // public info was received as a result of subscribes sent upon uncovery
                LinkedList possibleAlliances = findAlliances();
                //logger.warning("possible: "+possibleAlliances.size()+"   "+this.getName());
                tryToJoinAlliances(possibleAlliances);
            }
        }
    }

  /**
   * Tries to join alliances by conntacting suitable alliance leaders.
   * @param possibleAlliances
   */
  protected void tryToJoinAlliances(final List possibleAlliances) {
      Object content = createJoinAllianceContent();
      if (possibleAlliances.size() > 0) {
          Address a = (Address) possibleAlliances.remove(0);
          if (!myAlliance.equalsIgnoreCase(community.getAgent(a.getName()).alliance.getName())) {
            SeqRequestInitiatorTask task = new SeqRequestInitiatorTask(this, a , content, false,possibleAlliances);
            task.start();
          } else {
        	  this.scheduleEvent(new Runnable() {
        		  
        		  public void run() {
                      startTaskChangeAlliance();        			  
        		  }
        		  
        	  }, (long) (2000*Math.random()+2000));
          }
      }
   }
  
  /**
   * creates new alliance.
   * It should also inform yellow pages about change of public information.
   */
  public void createNewAlliance() {
    String newAllianceName = getAddress().getName() + "`s alliance";
    setAlliance(newAllianceName);
  }

  /**
   * returns the content of send subscribe to alliances
   *
   * @return java.lang.Object
   */
  protected Page createJoinAllianceContent() {
        Page pg = new Page();
        pg.setName(getAddress().getName());
        pg.setAddress(getAddress());
        pg.setPublicParams(publicParams);
        return pg;
    }

  /**
   * returns list of addresses of agents in alliances (one agent per alliance)
   * accepted by this agent's private and public knowledge. LinkedList is
   * returned because it supports removeFirst method.
   *
   * @return java.util.LinkedList
   */
  protected LinkedList<Address> findAlliances() {

      TreeMap<String, Address> acceptableAlliances = new TreeMap<String, Address>();
      for (AllianceKnowledge ak : community.alliances.values()) {
      	if (!ak.getName().equalsIgnoreCase(this.myDriversAlliance)) {
            if (allianceAcceptable(ak)) {
                acceptableAlliances.put(ak.getName(),(ak.pickRandomMember()).address);
            }
          }
      }
      return new LinkedList<Address>(acceptableAlliances.values());
  }

  /**
   * RestrictionChcecker checks whether this alliance is suitable.
   * @param ak
   * @return
   */
  public boolean allianceAcceptable(AllianceKnowledge ak)  {
      if (null != ak) {
          for (AgentKnowledge member : ak.getAllMembers()) {
              if (!restrictionsChecker.acceptAllianceMember(member)) {
                  return false;
              }
          }
          return true;
      }
      return false;
  }

  /**
   * Determines whether accept or not a new alliance member (base on the restrictionsChecker).
   * @param queryMessage
   * @param intermediary
   * @return
   */
  public boolean acceptAllianceMember(Message queryMessage, Address intermediary) {

	Page content = (Page)queryMessage.getContent();
	community.agentPageUpdated(content.getAddress(), content.getPublicParams());
    AgentKnowledge ak = community.getAgent(queryMessage.getSender().getName());
    if(null != ak)
        return restrictionsChecker.acceptAllianceMember(ak);
    return false;
  }


  /**
   * Updates this TA semi-private knowledge
   * @param agent
   * @param params
   */
  public void updateSemiPrivate(Address agent, SemiPrivateParams params) {
      // first time contact - return the semi-private knowledge
      AgentKnowledge ak = community.getAgent(agent.getName());
      if (null != ak ) {
          ak.updateSemiPrivateParams(params);
      }
  }


  /**
   * Gets all the members of alliance.
   * @return
   */
  public Collection<Address> getAllianceMembers() {
      AllianceKnowledge ak = community.alliances.get(myAlliance);
      if (null != ak) {
          return ak.getAllAgentsAddress(getAddress());
      }
      return new LinkedList<Address>(); 
  }

    /** 
     * Broadcasts the semi-private information to all members of current alliance
     */
    public void broadcastSemiPrivateToAlliance() {
    	for (SubscribeParticipantTask spt : semiPrivateSubscribers) {
            spt.informResult(semiPrivateParams);
        }
    }
    /** Sends the semi-private information to the agent address and request its semiprivate params.
     *  @param address Receiver of the semi-private info
     **/
    public void subscribeAndSendSemiPrivateInfo(Address address) {
    	if (address != getAddress()) {
    		SubscribeInitiatorTask task = new SubscribeInitiatorTask(this, address,
    				semiPrivateParams, false) {
    			/**
    			 * informResult - receives information from subscribed agent
    			 * @param result Object
    			 */
    			protected void subscribeInformResult(Object result) {
    				if (result instanceof SemiPrivateParams) {
    					updateSemiPrivate(this.participant,
    							(SemiPrivateParams) result);
    				} else {
    					logger
    					.warning("Bad response type for semiPrivate subscribe: "
    							+ result);
    				}
    			}
    		};
    		task.start();
    	}
    }

  /*****************************************
   * Methods used to maintain coalition
   ****************************************/

  /**
   * Accepts or rejects this Proposal based upond the restrictionsChecker (the restcitions 
   * on alliances)
   * @param teamProposal TeamProposal
   * @return boolean
   */
  public boolean acceptTeam(TeamProposal teamProposal) {
      List<Address> teamMembers = teamProposal.getTeamMember();
      boolean aLeader = restrictionsChecker.acceptTeamLeader(teamProposal.getTeamLeader());
      boolean aProp = restrictionsChecker.acceptRequestedServices(teamProposal.getRequestedServices());
      if (aLeader && aProp ) {
    	  for (Address member : teamMembers) {
              if (!restrictionsChecker.acceptTeamMember(member)) {
                  return false;
              }
          }
          return true; // all members and batches were accepted
      }
      return false; //leader refused
  }

  /**
   * getCoalitionMemberQuality
   *
   * quality is taken as average coverage of all requested resources
   *
   * @param cfp TransportCfp
   * @param requestedResources List
   * @param member Address
   * @return int, above zero according to the ratio of the batches this agent
   *   can handle, 0 if there is no possible contribution or if there are no
   *   requested resources
   */
  public double getCoalitionMemberQuality(TransportCfp cfp, Collection<Resource> requestedResources, Address member) {
        long sumReq = 0;
        double quality = 0;
        if (0 < requestedResources.size()) {
            for (Resource resource : requestedResources) {
                long requestedCapacity = resource.getCapacity();
                if (requestedCapacity > 0) {
                    sumReq += requestedCapacity;
                    long agentCapacity = getAgentResource(member, resource.getType());
                    // if agent's capacity is bigger then requested => quality is 1
                    quality += Math.min(agentCapacity,  requestedCapacity);
                }
            } 
            if (0 < sumReq) {
                quality /= sumReq;
            }
        }
        return quality;
    }

  /**
   * getAgentResource - returns agents resource - in accordance to semi-private
   * knowledge
   *
   * @param agent Address
   * @param resourceType String
   * @return long
   */
  public long getAgentResource(Address agent, String resourceType) {
      SemiPrivateParams semiP;
      if (getAddress() == agent) {
          semiP = semiPrivateParams;
      } else {
          AgentKnowledge ak = community.getAgent(agent.getName());
          semiP = ak.semiPrivateParams;
      }
      // check if we do have semi-p info about this agent...
      if (null != semiP) {
    	  for (Resource resource : semiP.getResource()) {
              if (resource.getType().equals(resourceType)) {
                  return resource.getCapacity();
              }
          }
      }
      // agent has no capacity of this type
      return 0;
  }

    /**
     * createProposal - simple first version
     * if does not accept team returns null otherwise propose as much resources as possible
     *
     * @param teamProposal TeamProposal
     * @return Proposal
     */
    public Proposal createTeamMemberProposal(TeamProposal teamProposal) {
       if (! acceptTeam(teamProposal)) {
          return null;
        }

        TransportCfp requestedResources = teamProposal.getRequestedServices();

        Proposal proposal = new Proposal();

        proposal.setProposalTime(currentTime);
        proposal.setRequestid(requestedResources.getRequestid());
        // price determined later!
        proposal.setTotalPrice(0);
        List itemCoverage = proposal.getItemCoverage();
        // buffer currently available resources...
        Map agentResources = GoodsConstants.getCargoTypesResourceMap();// contains Resource objects
        for (Resource res : semiPrivateParams.getResource()) {
            ((Resource)agentResources.get(res.getType())).setCapacity(getAvailableResource(res.getType(), currentTime));
        }
        for (TransportBatch item : requestedResources.getTransportBatch()) {
            if (0 < item.getCount()) {
                ItemCoverage coverage = new ItemCoverage();
                coverage.setItemid(item.getBatchid());
                long capacity = ((Resource)agentResources.get(GoodsConstants.resolveType(item.getComodityName()))).getCapacity();
                long usedcap = Math.min(capacity, item.getCount());
                if (0 < usedcap)  {
                    coverage.setCoverage(((double)item.getCount())/usedcap);
                    itemCoverage.add(coverage);
                    long price = computePrice(item, coverage);
                    proposal.setTotalPrice(proposal.getTotalPrice() + price);
                    // remove the capacity used for this batch from available resource list
                    ((Resource)agentResources.get(GoodsConstants.resolveType(item.getComodityName()))).setCapacity(capacity - usedcap);
                }
            }
        }
        if (0 < proposal.getTotalPrice()) {
            return proposal;
        }
        else {
            return null; // refuse, no contribution possible
        }
    }



  /**
   * Returns the price at which we are able to provide coverage of given item
   * transport.
   *
   * @param item what is to be transported, from where and where, + times
   * @param coverage ItemCoverage
   * @return long
   */
  protected long computePrice(TransportBatch item, ItemCoverage coverage)   {
        return Math.round(item.getCount()*coverage.getCoverage()*2 + 10); //TODO - call the map or query the drivers to obtain the price... /**@todo*/
    }

    /**
     * Returns aggregated cappacity of all resources of given type available at strartTime.
     * @param cargoType type of the cargo/resource to transport
     * @param startTime time to which resource availability is compared
     * @return sum of all resources available at time startTime
     */
    protected long getAvailableResource(String cargoType, long startTime) {
        long result = 0;
        for (Resource res : privateParams.getResource()) {
            if (res.getType().equalsIgnoreCase(cargoType) && res.getAvailabilityTime() <= startTime) {
                result += res.getCapacity();
            }
        }
        return result;
    }
 
    /**
     * Called by Idle task to allow CFP treatment modification.
     * @param msg received message with cfp
     * @return true if the CFP is handled by the method, false to continue normal processing by dedicated task
     */
    public boolean receivedCFP(Message msg, final CNPParticipantTask goodsRequestTask) {
    	return false;
    }
    
    /**
     * Gets the current simulation time.
     * @return
     */
    public long getTime() {
      return currentTime;
    }

    /**
     * Returns the CommunityKnowledge of this TA.
     * @return
     */
    public CommunityKnowledge getCommunity() {
      return community;
    }
    /**
     * To be overloaded in derived types.
     * @param msg
     */
    public void receivedAdvTeamProposal(Message msg) {
		sendNotUnderstood(msg, "Capability not present AdvTeamProposal content not understood.");
	}

    /**
	 * To be overloaded in derived types.
	 * @param msg
	 */
	public void receivedAdvTeamRequest(Message msg) {
		sendNotUnderstood(msg, "Capability not present AdvTeamProposal content not understood.");
	};

	/**
	 * RequestInitiatorTask for joining an alliance.  It tries all the possible alliance leaders.
	 * @author Eduard Semsch
	 *
	 */
	class SeqRequestInitiatorTask extends RequestInitiatorTask {
        List addressListforTheNext;
        public SeqRequestInitiatorTask(CMAgent owner, Address participant, Object content, boolean start, List addressListforTheNext)  {
            super(owner, participant, content, false);
            this.addressListforTheNext = addressListforTheNext;
            logger.fine(getName() + " alliance entry request sent to: " + participant.getName());
            if (start) {
              start();
            }
        }

        /**
         * This TransporterAgent has been accepted in the alliance.
         */
        protected void informDone() {
            AgentKnowledge particip = community.getAgent(participant.getName());
            AllianceKnowledge allik = particip.alliance;
            setAlliance(allik.getName());

            // send semi-private info to all members of new alliance
            //TODO - subscribe and send semiprivate to all alliance...

            AllianceKnowledge alk = community.getAlliance(myAlliance);
            if (null != alk) {
                for (AgentKnowledge ak : alk.getAllMembers()) {
                    subscribeAndSendSemiPrivateInfo(ak.address);
                }
            }
            cancelTask();
        }
        
        /**
         * This TransporterAgent has been refused from the alliance - try another possible alliance leader.
         */
        protected void informResult(Object result) {
            // if the result is different from done, repeat
            cancelTask();
            if (addressListforTheNext.size() > 0) {
                Address a = (Address) addressListforTheNext.remove(0);
                new SeqRequestInitiatorTask(TransporterAgent.this, a, content, true,this.addressListforTheNext);
            } else {
            	TransporterAgent.this.scheduleEvent(new Runnable() {
          		  
          		  public void run() {
                        startTaskChangeAlliance();        			  
          		  }
          		  
          	  }, (long) (2000*Math.random()+2000));
                logger.info(getName() + " rescheduling alliance entry, none found.");
            }
        }
    }
}
