package across.agents.terrorists.driver;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import across.agents.driver.data.RoadInfo;
import across.agents.driver.data.Route;
import across.agents.driver.data.RouteInfo;
import across.data.AcrossMapArcs;
import across.data.AcrossMapDistances;
import across.data.AcrossMapNodes;
import across.data.Arc;
import across.data.Node;
import across.data.Waypoint;
import across.simulation.constants.AcrossControlConstants;
import across.simulation.constants.AcrossDataConstants;
import across.util.pathfinding.AStarModPathfinding;
import across.util.pathfinding.DijkstraPathfinding;
import across.util.pathfinding.Pathfinding;
import across.util.pathfinding.RoadEvaluator;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.entity.EntityDescriptor;

public class TerDriverMovement {

	/** Owner of this class - the agent. */
	protected TerroristDriverAgent owner;
	
	/** Algorithm for pathfinding */
	protected Pathfinding pathfinding;
	
	/** Arcs of the map */
	protected AcrossMapArcs arcsOfMap;
	
	/** Nodes of the map (map). */
	protected LinkedList<String> locNames = new LinkedList<String>(); 
	  
	/** Location where the driver currently is. */
	protected String actualLoc;

	/** The node where the driver is currently heading to. */
	protected String headingTo;
	
	/** The set of waypoints this agent is going to go through. */
	protected LinkedList<Waypoint> waypoints = new LinkedList<Waypoint>(); 
	
	/** Current position on the map */
	protected double[] position;
	
	public TerDriverMovement(TerroristDriverAgent owner) {
		this.owner = owner;
	}
	/**
	 * Handles all topics concerning movement.
	 * @param content
	 * @param reason
	 */
	public void handleMovementTopic(Object content, String reason) {
	  if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_BAD_NODE)) {
  		  owner.logSevere("Bad node chosen!");
	  } else if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_FAILURE)) {
		  owner.logSevere("Movement failure!");
	  } else if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_STOPPED) && content instanceof String) {
		  
		  	// get the actual waypoint, check the requests to be loaded/unloaded
		  	actualLoc = (String) content;
		  	
		  	// if there are no waypoints scheduled.
			if(waypoints.size()==0) {
			  return;
			}
		  	
			// get next waypoint
		  	Waypoint wp = waypoints.getFirst();
		  	if(wp.getLocationName().equals(actualLoc)) {
		  		waypoints.removeFirst();
		  	}
		  	
		  	// proceed moving
			move();
      	} else if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_STOPPED) && content instanceof double[]) {
      		position = (double[]) content;
      	}
	}
	
	/**
	 * Handles configuration of the vehicle.
	 * @param ed
	 * @param velocity
	 * @param consumption
	 */
	public void handleConfiguration(EntityDescriptor ed, final double velocity, final double consumption) {
		this.actualLoc = ed.confParamsString.get(AcrossDataConstants.ENTITY_START_NODE);
		  
		AcrossMapNodes nodesOfMap = (AcrossMapNodes) ed.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_NODES);
		arcsOfMap = (AcrossMapArcs) ed.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_ARCS);
		AcrossMapDistances mapDistances = (AcrossMapDistances) ed.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_DISTANCES);
		
	  	// Get list of node names
	    locNames.clear();
	    for (Node node : (List<Node>) nodesOfMap.getNode()) {
	    	locNames.add(node.getName());
	    }
	      
	    RoadEvaluator re = new RoadEvaluator() {

	    	public double getPrice(Arc arc) {
				return (1/arc.getQuality()) * arc.getLength() * consumption/100;
			}
			
			public double getTime(Arc arc) {
				return 10*arc.getLength()/velocity;
			}
					  
	     };
	      
		 pathfinding = new AStarModPathfinding(nodesOfMap.getNode(), arcsOfMap.getArc(), mapDistances.getDistance(), re, new LinkedList<Arc>());
	}
	
	/**
	 * Says whether the agent is moving or not.
	 * @return
	 */
	public boolean isOnRoute() {
		return waypoints.size()!=0;
	}
	
	/**
	 * Gets the current position.
	 * @return
	 */
	public double[] getPosition() {
		return position;
	}
	
	/**
	 * Carries out a route if it is possible.
	 * @param to
	 * @param byPrice - specifies what to take in consideration when planning the route
	 * @return
	 */
	public boolean goToRandomLocation() {
		Random rand = new Random(System.currentTimeMillis());
		int locNum = locNames.size();
		String randLoc = locNames.get(rand.nextInt(locNum));
		Waypoint wp = new Waypoint(randLoc);
		Route route = planRoute(new LinkedList<Waypoint>(Arrays.asList(new Waypoint[] {wp})), true);
		if(route!=null) {
			waypoints.addAll(route.getWaypoints());
			move();
			return true;
		} else {
			return false;
		}
	}
	

    /**
     * Plans a route through a set of waypoints and calculates price and time.
     * @param wpList
     * @param byPrice
     * @return 
     */
    protected Route planRoute(List<Waypoint> wpList, boolean byPrice){
      String startPoint;
      Route route = new Route();
      
      if (waypoints.size() == 0) {
      	startPoint = new String(actualLoc);
      } else {
      	startPoint = new String(((Waypoint)waypoints.getLast()).getLocationName());
      }

      double taskTime = 0;
      double taskPrice = 0;

      for (Waypoint wp : wpList) {
        Route oneRoute;
        if (byPrice) {
      	  oneRoute = pathfinding.planRoute(startPoint, wp.getLocationName(), 0, 1);
        } else {
      	  oneRoute = pathfinding.planRoute(startPoint, wp.getLocationName(), 1, 0);
        }
        if (oneRoute == null) {
        	continue;
        }
        taskTime += oneRoute.getRouteInfo().routeTime;
        taskPrice += oneRoute.getRouteInfo().routePrice;
        startPoint = wp.getLocationName();
        // add all intermediate waypoints (with empty request lists)
        route.addWaypoints(oneRoute.getWaypoints());
        route.addWaypoint(wp);  // add the actual waypoint
      }
      route.setRouteInfo(new RouteInfo(taskTime, taskPrice));
      return route;
    }
    
	
	 /**
	   * Move to the next waypoint (if there's any in the list)
	   */
    protected void move(){
    	if (!isOnRoute()) {
		    return;
		} else {
			
			Waypoint wp = waypoints.getFirst();
		    
		    String waypointName = wp.getLocationName();
		    
		    headingTo = waypointName;
		    
		    owner.gisShell.submitTopic(ClientServerTopicConstants.TOPIC_ENTITY_CONTROL, waypointName, AcrossControlConstants.MOVE_TO);
		 }
	 }
}
