package across.agents.location.task;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;

import across.agents.location.LocationAgent;
import across.agents.location.accounts.StoreRec;
import across.data.Batch;
import across.data.Page;
import across.data.RequestList;
import across.simulation.constants.AcrossMessageConstants;
import aglobe.container.task.Task;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobex.protocol.request.RequestParticipantTask;
import aglobex.protocol.subscribe.SubscribeParticipantTask;

/**
 * Created by IntelliJ IDEA.
 * User: rehakm1
 * Date: 14.5.2004
 * Time: 15:56:59
 * 
 * This class implements the main communication logic of the location agent - this handles all the newly
 * incomming messages to the LocationAgent and determines the way of handling them. 
 */
public class LocIdleTask extends Task
{
	/**
	 * Owner agent - the LocationAgent.
	 */
    protected LocationAgent owner;
    
    /**
     * Time out tasks for the batches. Probably not used at all - semsch??? REMOVE?
     */
    private HashMap<Long,LinkedHashMap<LocSubscribeParticipantTask,Batch>> timeouts;
    
    public static final String LOC_TIMEOUT = "Location timeout";

    /**
     * The constructor
     * @param owner - the owning agent.
     */
    public LocIdleTask(LocationAgent owner)  {
        super(owner);
        this.owner = owner;
    }
 
    /**
     * This method handles all the newly incomming messages (with the not previously known message IDs).
     * It handles these messages:
     * 
     * <li> Subscribe messages - handled by handleSubscribeProtocol function. </li>
     * <li> CFP - handles the CNP by answerCNP function or if in the disaster area refuses the CNP. In some 
     * odd cases the CNP may be handled by handleGeneralCFP function </li>
     * <li> REQUEST protocol - the content may be a Batch - in that case it means that either the goods are
     * to be loaded on the location or out of the location (depending on the count - negative count means that
     * the goods are comming into the location, positive that they are leaving the location.  For any other 
     * request there is HandleGeneralRequest function. </li>
     * <li> Also handles attack notification. </li>
     * 
     * @param m The message.
     */
    protected void handleIncomingMessage(Message m) {
        if (MessageConstants.SUBSCRIBE.equalsIgnoreCase(m.getProtocol())) { // Protocol = SUBSCRIBE
            handleSubscribeProtocol(m);
        } 
        if (m.getPerformative().equalsIgnoreCase(MessageConstants.CFP))  {
          if (m.getContent() instanceof RequestList) {
            // handle goods request list
            // start the new CFP to sell goods - CNPParticipant
            answerCNP(m);  
          } else {
        	  handleGeneralCFP(m);
          }
        } else if (m.getPerformative().equalsIgnoreCase(MessageConstants.REQUEST)) {
            // load or unload request...
            if (m.getContent() instanceof Batch) {
                
                // positive count = load to location agent
                final Batch b = (Batch) m.getContent();
                long batchCount = b.getCount();
                
                if (batchCount < 0) { // load goods on the location agent
                	owner.gisShell.submitTopic(across.visio.oldvisio.VisioConnectionAgent.TOPIC_BATCH_LOADED_ON_LOCATION, (Batch)m.getContent());
                	System.out.println(owner.getName()+" LOAD ON LOCATION: "+b);
                	RequestParticipantTask task = new RequestParticipantTask(owner,m,false) {
                		@Override
                		protected void processRequest(Message requestMessage) {
                			informDone();
                			cancelTask();
                		}
                	};
                	task.start();
                	b.setCount(-batchCount);
                	try {
                		owner.books.goodsReceived(b);
                	} catch (Exception e) {
                		e.printStackTrace();
                		owner.getLogger().warning(e.toString());
                	}
                } else {
	                if (batchCount > 0) { // unload of goods from the location agent
	                	  System.out.println(owner.getName()+" UNLOAD FROM LOCATION");
	                          
	                	RequestParticipantTask task = new RequestParticipantTask(owner,m,false) {
	                		
	                		/**
	                		 * Handles the moving of the commodities from the location to the driver.
	                		 * @todo Remove CMDriverAgent.requestedFill - 
	                		 */
	                		// TODO - conf (used to be:
	//                		double successRate = owner.books.goodsUnload(b, DriverAgent.requestedFill, false);
	//            			if (successRate >= CMDriverAgent.requestedFill) {    // enough goods in store - possible to unload from the location agent
	//            				owner.books.goodsUnload(b, CMDriverAgent.requestedFill, true);
	//                      // CMDriverAgent.requestedFill substituted with 1
	                		@Override
	                		protected void processRequest(Message requestMessage) {
	                			double successRate = owner.books.goodsUnload(b, 1, false);
	                			if (successRate >= 1) {    // enough goods in store - possible to unload from the location agent
	                				owner.books.goodsUnload(b, 1, true);
	                                owner.gisShell.submitTopic(across.visio.oldvisio.VisioConnectionAgent.TOPIC_BATCH_UNLOADED_FROM_LOCATION, b);                          
	       		        		  System.out.println(owner.getName()+" RECEIVE REQUEST - SENDING INFORM DONE"+b+"success rate" + successRate);
	                				informDone();
	                			} else {  // not enought goods in store - send failure to driver and do nothing
	       		        		  System.out.println(owner.getName()+" RECEIVE REQUEST - SENDING FAILURE "+b+"success rate" + successRate);
	                				failure(new Double(successRate));
	                			}
	                			cancelTask();
	                		}
	                	};
	                	task.start();
	                }
                }
            } else {
            	handleGeneralRequest(m);
            }
        } else if (m.getPerformative().equalsIgnoreCase(MessageConstants.FAILURE )&& m.getContent() instanceof RequestList)  { // stolen good notification
            handleAttackNotification(m);
        }
        // refresh gui
        owner.gui.setAll(owner.books);
    }

    /**
     * Method for handling the subscribe protocol.  There are more cases when a subscribe message
     * can receive:
     * <li> Cancels out the subscribes sent by itself. </li>
     * <li> On receiving subscribe message with content=TransporterAgent.PUBLIC_INFO_REQUEST 
     * returns its public information (the Page). </li>
     * 
     * @param msg
     */
    private void handleSubscribeProtocol(Message msg) {
        // don't subscriber itself
        if (msg.getSender() == owner.getAddress()) return;
        // Public info subscribe
        if (AcrossMessageConstants.PUBLIC_INFO_REQUEST.equals(msg.getContent()))  {
            SubscribeParticipantTask task = new SubscribeParticipantTask(owner, msg, false)  {
                protected void processSubscribe(Message subscribeMessage) {
                    Page pg = new Page();
                    pg.setName(owner.getAddress().getName());
                    pg.setAddress(owner.getAddress());
                    pg.setPublicParams(((LocationAgent)owner).page.getPublicParams());
                    informResult(pg);
                }
            };
            task.start();
        }
    }
    
    /**
     * This method adjusts the expectations of receiving the goods that were stolen.  It basically
     * decreases the expected amount of goods to be delivered by the amount that was stolen.
     * @param m
     */
    protected void handleAttackNotification(Message m) {
        RequestList r = (RequestList)m.getContent();
        for (Batch batch : r.getBatch()) {
            for (Iterator i = owner.books.getGoodsExpected().iterator(); i.hasNext();) {
                StoreRec sr = (StoreRec) i.next();
                if (sr.getCommodity().compareTo(batch.getComodityName()) == 0) {
                    sr.increase(batch); // this batch is unload -> batch.getCount() is negative
                }
            } // end - for i
        } // end - for batchIter
    }

    /**
     * Not implemented! To be overriden!.
     * @param m
     */
    private void handleGeneralCFP(Message m) {
      owner.logWarning(owner.getName()+" HANDLE GENERAL REQUEST");
    }
        
    /**
     * Not implemented! To be overriden.
     * @param m
     */
    protected void handleGeneralRequest(Message m) {
  	  owner.logWarning(owner.getName()+" HANDLE GENERAL REQUEST");
    }
    
    /**
     * Answers the CNP - after receving a RequestList from a location - semsch???
     * @param m
     */
    protected void answerCNP(Message m) {
        new LocProposeGoodsTask(owner, LocProposeGoodsTask.TIMEOUT, m);
    }
    
    /**
     * This method is called when the simulation time tick arrives.  It looks for the batches that have
     * timed out and executes the associated tasks. 
     */
    public void handleTimeout() {
		if (timeouts!=null) {
    		LinkedHashMap<LocSubscribeParticipantTask,Batch> list = timeouts.get(owner.getCurrentTime());
    		if (list!=null) {
        		for (java.util.Map.Entry<LocSubscribeParticipantTask, Batch> e : list.entrySet()) {
    				LocSubscribeParticipantTask task = e.getKey();
    				Batch batch = e.getValue();
    				task.timeout(batch);
    			}
    			timeouts.remove(owner.getCurrentTime());
    		}
    	}
    }
    
    /**
     * This registers a timely task with a batch.  The task is to be executed after the specified 
     * amount of simulation time.
     * @param batch
     * @param task
     * @return
     */
    protected boolean registerTimeout(Batch batch, LocSubscribeParticipantTask task) {
		if (owner.getName().equals("Bandung"))
      		System.out.println(owner.getName()+" Registering timeout: "+batch);
    	Long time = new Long(batch.getDeliveryTime());
    	if (time<=owner.getCurrentTime()) {
    		task.timeout(batch);
    		return false;
    	}
    	LinkedHashMap<LocSubscribeParticipantTask,Batch> list;
    	if (timeouts == null) {
    		timeouts = new HashMap<Long,LinkedHashMap<LocSubscribeParticipantTask,Batch>>();
    		if (owner.getName().equals("Bandung"))
          		System.out.println(owner.getName()+" Registering timeout - creating new timeouts ");
    	}
    	list = timeouts.get(batch.getDeliveryTime());
    	if (list == null) {
    		list = new LinkedHashMap<LocSubscribeParticipantTask,Batch>();
    		if (owner.getName().equals("Bandung"))
          		System.out.println(owner.getName()+" Registering timeout - creating new list ");
    	} else {
    		Batch b=list.get(task);
    		if (b!=null) {   // toto by se stat nemelo
    			owner.getLogger().warning("Location: more timeout registration error");
    		}
    	}
		list.put(task,batch);
		timeouts.put(time,list);
    	return true;
    }

    /**
     * Deregisters a timely task from a batch.
     * @param batch
     * @param task
     */
    protected void deregisterTimeout(Batch batch, LocSubscribeParticipantTask task) {
    	LinkedHashMap<LocSubscribeParticipantTask,Batch> list;
    	if (timeouts != null) {
    		list = timeouts.get(batch.getDeliveryTime());
    		if (list != null) {
    			list.remove(task);
    		}
    	}
    }
}
