package across.agents.location.task;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;

import across.agents.location.LocationAgent;
import across.agents.location.accounts.NegativeStockException;
import across.data.Batch;
import across.data.ItemCoverage;
import across.data.Proposal;
import across.data.RequestList;
import across.data.TransportBatch;
import across.data.TransportCfp;
import aglobe.agent.dataanalyzer.DataAnalyzerAgent;
import aglobe.container.task.ConversationUnit;
import aglobe.container.transport.Address;
import aglobe.container.transport.InvisibleContainerException;
import aglobe.ontology.AglobeParams;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobe.service.gis.client.GISClientService;
import aglobe.util.AglobeXMLtools;
import aglobex.simulation.protocol.cnp.CNPInitiatorTask;


/**
 * This task implements the CFP for transport of sold goods to the client.
 * It arranges the transport and notifies the client about the result by providing the list of the goods that will be transported.
 */
public class CNPTransportManagerTask extends CNPInitiatorTask {
	/** The owner agent. */
    LocationAgent owner;
    /** stores the cfp content*/
    TransportCfp cfp;
    /** Used to handle situations when transport is refused or impossible */
    RequestList goodsCfp;
    /** Used to handle situations when transport is refused or impossible */
    Proposal goodsProposal;

    LocProposeGoodsTask creatorLocProposeGoodsTask;
    LocIdleTask idleTask;

    // TODO - conf
    public static final int TIMEOUT = 17000;


    public CNPTransportManagerTask(LocationAgent owner, Collection participants, TransportCfp cfpContent, RequestList goods, Proposal goodsProposal,int timeout,LocIdleTask locIdleTask) {
        super(owner, participants, cfpContent, timeout, null, false);
        this.owner = owner;
        cfp =  cfpContent;
        cfp.setRequestid(cfp.getRequestid()+"-"+owner.getName()); // petr
        goodsCfp = goods;
        this.idleTask = locIdleTask;
        this.goodsProposal = goodsProposal;
        start();
    }


    public CNPTransportManagerTask(LocationAgent owner, Collection participants, TransportCfp cfpContent, RequestList goods, Proposal goodsProposal,int timeout,LocProposeGoodsTask creatorLocProposeGoodsTask)    {
        super(owner, participants, cfpContent, timeout, null, false);
        this.owner = owner;
        cfp =  cfpContent;
        cfp.setRequestid(cfp.getRequestid()+"-"+owner.getName()); // petr
        goodsCfp = goods;
        this.creatorLocProposeGoodsTask = creatorLocProposeGoodsTask;
        this.goodsProposal = goodsProposal;
        start();
    }

    /**
     * Evaluates answers from transporters. Picks only one best offer. Criteria used are coverage and price.
     * We allways pick the best coverage. if there are several offers with max. coverage, we pick the best price between them.
     * If the price is equal also, we pick the first answer.
     * @return List of addresses to whom we send the proposal. List with 0 elements refuses all and cancels the task.
     */
    protected List evaluateReplies()    {
        // to store the selected offer(s)
        List selected = new LinkedList();
        // to store the best offer so far
        Message bestOffer = null;
        double bestCoverage = 0;// coverage ratio
        long bestPrice = Long.MAX_VALUE;
        // pass through all propositions and check the criteria - price only in our case
        for (Iterator iterator = receivedOffers.values().iterator(); iterator.hasNext();)
        {
            Message m = (Message) iterator.next();
            if (m.getPerformative().equalsIgnoreCase(MessageConstants.PROPOSAL))
            {
                try
                {
                    Proposal prop = (Proposal) m.getContent();
                    if (null != prop)
                    {
                        long price = prop.getTotalPrice();
                        // coverage is computed as count-weighted coverage of all items
                        double coverage = 0;
                        long total = 0;
                        // go through all batches of the cfp request and compute the coverage
                        for (ListIterator i = cfp.getTransportBatch().listIterator(); i.hasNext();)
                        {
                            TransportBatch tb = (TransportBatch) i.next();
                            total += tb.getCount();
                            // find corresponding coverage in proposal
                            for (Iterator it = prop.getItemCoverage().iterator(); it.hasNext();)
                            {
                                ItemCoverage ic = (ItemCoverage) it.next();
                                if ( ic.getItemid() == tb.getBatchid())
                                {
                                    coverage += tb.getCount() * ic.getCoverage();
                                    break;
                                }
                            }
                        }

                        if (0 != total) {
                          coverage = coverage / total;
                        }
                        else {
                          coverage = 1; // nothing is allways covered
                        }

                        if ( coverage > bestCoverage || (coverage == bestCoverage && price < bestPrice))
                        {
                            bestOffer = m;
                            bestPrice = price;
                            bestCoverage = coverage;
                        }
                    }
                }
                catch (ClassCastException e)
                {
                    owner.getLogger().severe("Unexpected content type in cfp answer - proposal expected. Message:\n " + m);
                }
            }
        }
        // only one winner in our case - fills inthe list if possible
        if (null != bestOffer)
        {
            selected.add(bestOffer.getSender());
            // acceptation is assumed as proposal from participant is binding
            // send done with real coverage to the buyer

            GISClientService.Shell gisShell = (GISClientService.Shell) ((LocationAgent)owner).gisShell;
           // gisShell.submitTopic(VisualAgent.TOPIC_BEST_TRANSPORT, bestOffer);

            // register sent goods
            try
            {
                owner.books.reservedGoodsAccepted(goodsCfp, goodsProposal, (Proposal)bestOffer.getContent());
            }
            catch (NegativeStockException e)
            {
                owner.getLogger().severe("Reserved goods not available for change when proposal was accepted." + goodsCfp + " \n " +goodsProposal);
                // send failure to the client if goods are not available
                try
                {  if (creatorLocProposeGoodsTask != null) {
                    creatorLocProposeGoodsTask.taskFailure(bestOffer.getContent());
                  }
                }
                catch (InvisibleContainerException e1)
                {
                    owner.getLogger().severe("Error: FAILURE message not sent to the client as areply to: " + goodsProposal);
                }
            }
        } else
        {
            // no acceptable transport offer returned
            try {
              if (creatorLocProposeGoodsTask != null) {
                creatorLocProposeGoodsTask.taskFailure(null);
              }
            }
            catch (InvisibleContainerException ex) {
            }
            try
            {
                owner.books.reservedGoodsRefused(goodsCfp, goodsProposal);
            }
            catch (NegativeStockException e){}
            return selected;
        }
        // send message to the Location agent receiving the goods...
        try
        { if (creatorLocProposeGoodsTask != null) {
            creatorLocProposeGoodsTask.taskDone(bestOffer.getContent());
          }
        }
        catch (InvisibleContainerException e){
            /**
             * @todo handle invisible ?
             */
            }

            // send actual number of algorithms to the visio/data analyzer
            AglobeParams logParams = new AglobeParams();
            logParams.getAglobeParam().add(AglobeXMLtools.makeAglobeParam("name", owner.getName()));
//      logParams.getParam().add(XMLtools.makeParam("timestamp", Integer.toString(observations.size())));
            logParams.getAglobeParam().add(AglobeXMLtools.makeAglobeParam("CFP best coverage", Double.toString(bestCoverage)));

            owner.gisShell.submitTopicToServer(DataAnalyzerAgent.TOPIC_TEST_DATA, logParams);


        // returns empty list if there is no selectable proposition...
        return selected;
    }

    /**
     * This function is called for each message received as a reply to accept (e.g DONE or FAILURE, ...). Return true if you want to kep the task running,
     * or false to cancel the task.
     * @return true to keep the task running, false to cancel it (when job is finished)
     */
    protected boolean evaluateAcceptReply(Message m)
    {
        if (m.getPerformative().equalsIgnoreCase(MessageConstants.FAILURE) )
        {

        }
        else if(m.getPerformative().equalsIgnoreCase(MessageConstants.INFORM_DONE) )
        {

        }
        // terminate by returning true
        return true;
    }

    /**
     * This function is called when timeout for accept reply (DONE or FAILURE, ...) elapses.
     */
    protected void evaluateAcceptTimeout()
    {
    }

    /**
     * Prepares a default transport cfp for goods acquired. This function supports multiple origins for the goods.
     * This function also performs load/unload of locally available goods.
     * @param goodsCfp RequestList from the goods acquisition
     * @param acceptedGoodsProposal acceptation message recieved from proposing party. Used fro
     * @param time current time, used for requesttime
     * @param targetLocation location to which all goods are to be transported
     * @param owner agent to which the goods shall bedelivered, typically the owner of this task... ASSUMPTION...
     * @return assembled transport request, null if no transport is required
     */
    public static TransportCfp prepareInboundTransportCfp(RequestList goodsCfp, Message acceptedGoodsProposal, long time, String targetLocation, LocationAgent owner)    {
        TransportCfp cfp = new TransportCfp();
        cfp.setRequestid(goodsCfp.getRequestid());
        cfp.setRequestTime(time);
        for (Batch b : goodsCfp.getBatch()) {
            // check if it was accepted and get the details of acceptance
            for (ItemCoverage ic : ((Proposal)acceptedGoodsProposal.getContent()).getItemCoverage()) {
                if ( ic.getItemid() == b.getBatchid() && ic.getCoverage() > 0) {
                    // prepare the request for loading
                    // check if the transport is necessary for the batch...
                    if (!ic.getLocation().equalsIgnoreCase(targetLocation)) {
                        TransportBatch tb = new TransportBatch();
                        tb.setBatchid(b.getBatchid());
                        tb.setComodityName(b.getComodityName());
                        tb.setCount((long)Math.floor(b.getCount()*ic.getCoverage()));
                        tb.setStart(ic.getLocation());// current location of the goods
                        tb.setStartAddress(acceptedGoodsProposal.getSender());
                        tb.setTargetAddress(owner.getAddress());
                        tb.setTarget(targetLocation);// current location of the goods
                        tb.setStartTime(time);
                        tb.setTargetTime(time);
                        cfp.getTransportBatch().add(tb);
                    } else {
                        // no transport necessary, just load/unload locally
                        Message loadReq = Message.newInstance(MessageConstants.REQUEST, owner.getAddress(), acceptedGoodsProposal.getSender());
                        loadReq.setProtocol(MessageConstants.REQUEST);
                        Batch batch = new Batch();
                        batch.setBatchid(b.getBatchid());
                        batch.setComodityName(b.getComodityName());
                        batch.setCount((long) Math.floor(b.getCount()*ic.getCoverage()));
                        batch.setDeliveryTime(b.getDeliveryTime());
                        loadReq.setContent(batch);
                        try {
                            owner.sendMessage(loadReq);
                            owner.books.goodsReceived(b);
                        } catch (InvisibleContainerException e){
                        }
                    }
                    break;
                }
            }
        }
        // check if the transport is required or not...
        if (!cfp.getTransportBatch().isEmpty()) {
            return cfp;
        } else {
            return null;// all goods are in the same location as buyer
        }
    }
    /**
     * This function is used to contract transporters for the delivery of goods provided by owner agent to client designated by cfpmessage.
     * @param cfpMessage Message with RequestList from the goods acquisition
     * @param submitedProposal proposal sent by proposing party
     * @param currentTime current time, used for requesttime
     * @param targetLocation location to which all goods are to be transported
     * @param owner agent delivering the goods
     * @return assembled transport request, null if no transport is required
     */
    public static TransportCfp prepareOutboundTransportCfp(Message cfpMessage, Proposal submitedProposal, long currentTime, String targetLocation, ConversationUnit owner) {
        return prepareTransportCfp((RequestList)cfpMessage.getContent(), submitedProposal, currentTime, owner.getConversationManager().getElementaryConversationEntity().getContainer().getContainerName(), owner.getConversationManager().getElementaryConversationEntity().getAddress(), targetLocation, cfpMessage.getSender());
    }

    /**
     * Prepares transport cfp for batches between two agents in different locations.
     * @param goodsCfp RequestList from the goods acquisition
     * @param acceptedGoodsProposal submitedProposal proposal sent by proposing party
     * @param time current time, used for requesttime
     * @param startLocation location from which all goods are to be transported
     * @param startAddress address of agent that delivers the goods
     * @param targetLocation location to which all goods are to be transported
     * @param targetAddress address of agent that receives the goods
     * @return  assembled transport request, null if no transport is required
     */
    public static TransportCfp prepareTransportCfp(RequestList goodsCfp, Proposal acceptedGoodsProposal, long time, String startLocation, Address startAddress ,String targetLocation, Address targetAddress)  {
        TransportCfp cfp = new TransportCfp();
        cfp.setRequestid(goodsCfp.getRequestid());
        cfp.setRequestTime(time);
        for (Batch b : goodsCfp.getBatch()) {
            // check if it was accepted and get the details of acceptance
        	for (ItemCoverage ic : acceptedGoodsProposal.getItemCoverage()) {
                if ( ic.getItemid() == b.getBatchid() && ic.getCoverage() > 0) {
                    // prepare the request for loading
                    TransportBatch tb = new TransportBatch();
                    tb.setBatchid(b.getBatchid());
                    tb.setComodityName(b.getComodityName());
                    tb.setCount((long)Math.floor(b.getCount()*ic.getCoverage()));
                    tb.setStart(startLocation);
                    tb.setStartAddress(startAddress);
                    tb.setTargetAddress(targetAddress);
                    tb.setTarget(targetLocation);
                    tb.setStartTime(time);
                    tb.setTargetTime(time + 2);
                    cfp.getTransportBatch().add(tb);
                    break;
                }
            }
        }
        // check if the transport is required or not...
        if (!cfp.getTransportBatch().isEmpty()) {
            return cfp;
        }
        else {
            return null;// all goods are in the same location as buyer
        }
    }

    /**
     * This function ensures sending the cargo between two location agents in the same container (location).
     * One loading request is generated per covered batch.
      * @param owner Agent sending cargo.
     * @param goodsCfp Call for proposal as sent by targetAddress
     * @param acceptedGoodsProposal proposal by owner accepted by target determining the coverage of request
     * @param targetAddress address of the requesting agent
     */
    public static void requestLocalBatchLoading(LocationAgent owner, RequestList goodsCfp, Proposal acceptedGoodsProposal, Address targetAddress) {
    	for (Batch  b : goodsCfp.getBatch()) {
            // check if it was accepted and get the details of acceptance
    		for (ItemCoverage ic : acceptedGoodsProposal.getItemCoverage()) {
                if ( ic.getItemid() == b.getBatchid() && ic.getCoverage() > 0)                {
                    // no transport necessary, just load/unload locally
                    Message loadReq = Message.newInstance(MessageConstants.REQUEST, owner.getAddress(), targetAddress);
                    loadReq.setProtocol(MessageConstants.REQUEST);
                    Batch batch = new Batch();
                    batch.setBatchid(b.getBatchid());
                    batch.setComodityName(b.getComodityName());
                    batch.setCount((long)Math.floor(b.getCount()*ic.getCoverage()));
                    batch.setDeliveryTime(b.getDeliveryTime());
                    loadReq.setContent(batch);
                    try {
                        owner.sendMessage(loadReq);
                        owner.books.goodsReceived(b);
                    } catch (InvisibleContainerException e){
                    }
                }
            }
        }
    }
}
