package across.agents.location.generator;

import across.agents.location.SimulationObjectsEffects;
import across.agents.location.accounts.StockOfferDemand;
import across.util.GoodsConstants;

/**
 * Created by IntelliJ IDEA.
 * User: rehakm1
 * Date: 26.5.2004
 * Time: 16:45:25
 * To change this template use Options | File Templates.
 */

/** This is a rather simple generator, generating the demand of goods by general population.*/
public class PopulationGenerator extends Generator
{
    public static final String NAME = "population";

    /** size of the population */
    long popSize = 1000;

    /** Field used to hold last 10 values of resource satisfaction. Values are used every 10'th generation cycle to calculate mood*/
    private double[] reqSatisfaction = new double[REQ_SAT_SIZE];
    /** Current position of the writing cursor in the request satisfaction field.*/
    private int reqSatisfactionCurrentPos = 0;
    /** Size of the request satisfaction field.*/
    private static final int REQ_SAT_SIZE = 3;

    /** Mood of the population. Increases if the request coverage is good, decreases, if it is bad. If it gets too low
     * (under DEADLY_MOOD), population decreases.
      */
    public long currentMood = 100;
    /** Maximum posible value for the mood of the population*/
    public static final long MAX_MOOD = 100;
    /** Minimum posible value for the mood of the population*/
    public static final long MIN_MOOD = 0;
    /** If the mood of the population falls under this value, it starts dying out (decreases population and consequently future requests).*/
    public static final long DEADLY_MOOD = 50;
    /** If the mood of the population falls under this value, it starts growing up */
    public static final long SUPERIOR_MOOD = 90;
    /** is False if the generator must make a new command to receive goods - first time or after population change*/
    private boolean goodsCommandMade = false;

    /**
     * Creates population demand generator.
     * @param popSize Number of people - demand depends on this number - grows with it lin.
     */
    public PopulationGenerator(long popSize)  {
        this.popSize = popSize;
    }

    /**
     * This method is called by location to demand resource generation or consumption.
     * Current stock/offer/demand/... available on the location is passes in the books variable.
     * Current weather is passed in as weather parameter.
     * @param weather Current weather
     * @param books Object with current goods/requests/supply/demand status
     */
    public long generationStep(Object weather, StockOfferDemand books, SimulationObjectsEffects effects) {
        if (!goodsCommandMade)  {
            // create demand for the goods...
            books.command(GoodsConstants.BREAD,1*popSize);
            books.command(GoodsConstants.WATER,2*popSize);
            books.command(GoodsConstants.FISH, (long)(0.25*popSize));
            books.command(GoodsConstants.SHEEP, (long)(0.25*popSize));
            goodsCommandMade = true;
        }

        // require the goods from the store
        long breadDemand = 1*popSize;
        long waterDemand = 2*popSize;
        long fishDemand = (long) (0.25*popSize);
        long sheepDemand = (long) (0.25*popSize);
        
        double breadUnits = books.goodsRequired(GoodsConstants.BREAD,breadDemand);
        double waterUnits = books.goodsRequired(GoodsConstants.WATER,waterDemand);
        double fishUnits = books.goodsRequired(GoodsConstants.FISH, fishDemand);
        double sheepUnits = books.goodsRequired(GoodsConstants.SHEEP, sheepDemand);

        // calculate the satisfaction of the population...
        double stepCoverage = (breadUnits/breadDemand+waterUnits/waterDemand+fishUnits/fishDemand+sheepUnits/sheepDemand)/4;

        // update the satisfaction values
        reqSatisfaction[reqSatisfactionCurrentPos%REQ_SAT_SIZE] = stepCoverage;
        reqSatisfactionCurrentPos++;
        if ( reqSatisfactionCurrentPos >= (REQ_SAT_SIZE-1)) {
            double avg = 0;
            for (int i = 0; i < reqSatisfaction.length; i++) {
                avg += reqSatisfaction[i];
            }
            avg /= REQ_SAT_SIZE;
            modifyMood(avg, books);
        } 
        if(currentMood<DEADLY_MOOD) {
        	// TODO - heuristic
        	popSize = Math.max(1, (long) Math.floor(popSize*0.95-10));
        } else if (currentMood>SUPERIOR_MOOD) {
        	popSize = Math.min(20000, (long) Math.floor(popSize*1.05+10));
        }
        if(effects.getAggregateFire()>0) {
        	popSize /= 2;
        }
        return popSize;
    }

    /**
     * Updates the mood of the population (and count also !!!), based on the resource coverage in previous 10 days.
     * @param avg average resource coverage in previous 10 days
     */
    private void modifyMood(double avg, StockOfferDemand books) {
        if (avg > 0.9) {
            increaseMood(7);
        } else if (avg > 0.8) {
            increaseMood(4);
        } else if (avg > 0.6) {
            // no change...
        } else if (avg > 0.3) {
            decreaseMood(3);
        } else {
            decreaseMood(7);
        }
    }
    /**
     * Increases the mood of the population.
     * @param i mood increase in units
     */
    private void increaseMood(long i)  {
        currentMood = Math.min(currentMood + i, MAX_MOOD);
    }

    private void decreaseMood(long i) {
        currentMood = Math.max(currentMood - i, MIN_MOOD);
    }

    public long getPopSize() {
        return popSize;
    }

    public long getCurrentMood() {
        return currentMood;
    }
}
