package across.agents.location.accounts;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;

import across.agents.location.LocationAgent;
import across.agents.location.task.LocSubscribeParticipantTask;
import across.agents.location.util.LocationUtils;
import across.data.Batch;
import across.data.ItemCoverage;
import across.data.Proposal;
import across.data.RequestList;
import aglobe.container.transport.Address;
import aglobex.protocol.subscribe.SubscribeParticipantTask;

/**
 * Created by IntelliJ IDEA.
 * User: rehakm1
 * Date: 14.5.2004
 * Time: 15:23:51
 * To change this template use Options | File Templates.
 */

/**
 * Represents offer and demand for different goods by one location agent or its component.
 * Goods are listed by type, hashed by ID's.
 * Positive value means available goods, negative values represent demand.
 */
public class StockOfferDemand  {
	
    /** Demanded goods. List of goods to acquire from other locations.*/
    protected SimpleStore goodsDemand = new SimpleStore();
    
    /** Private stock, not offered on the public market. */
    protected SimpleStore privateStock = new SimpleStore();
    
    /** Stock for trading purposes. */
    protected SimpleStore availableStock = new SimpleStore();
    
    /** Reserved stock - offered to other agents, waiting for accept or refuse */
    protected DetailedStore reservedStock = new DetailedStore();
    
    /** Keeps an account of the goods sold to other agents, but not picked-up yet.*/
    protected DetailedStore waitingForPickUp = new DetailedStore();
    
    /** Goods ordered elswhere, but not delivered yet.*/
    protected DetailedStore expectedGoods = new DetailedStore();


    /** Virtual store used for statistics only - Contains all demand created by Location agent.
     * Sums all demands since the beginning.*/
    protected SimpleStore agregateDemand = new SimpleStore();
    /** Virtual store used for statistics only - Contains all demand coverage created by Location agent.
     * Sums all received goods since the beginning.*/
    protected SimpleStore agregateReceived = new SimpleStore();
    /** Virtual store used for statistics only - Contains all production created by Location agent.
     * Sums all received goods since the beginning.*/
    protected SimpleStore agregateProduced = new SimpleStore();
    /** Virtual store used for statistics only - Contains all sales created by Location agent.
     * Sums all received goods since the beginning.*/
    protected SimpleStore agregateSold = new SimpleStore();

    private long _id = System.currentTimeMillis();
    
    /** Batches that have a task assigned - this task is run when the batch is delivered */
    private HashMap<Long,LinkedHashMap<Address,LocSubscribeParticipantTask>> registeredBatches = null;
    
    /**
     * Minimum lot size to aquire for the location.
     */
    private int minLotSize; 
    
    /**
     * Sets the minimumLotSize.
     * @param minLotSize
     */
	public void setMinLotSize(int minLotSize) {
		this.minLotSize = minLotSize;
	}
	
	/**
     * Returns one-item list of Request lists to be used for CFP's in this turn. List with no elements means nothing to buy.
     * Returns an only request containing all requested commodities (multiple batches).
     * @param agent Owner agent
     * @return List of RequestLists. Each of them spuns a cfp.
     */
    synchronized public List<RequestList> prepareSingleListGoodsDemand(LocationAgent agent,boolean dontMultiply) {
        LinkedList<RequestList> requestLists = new LinkedList<RequestList>();
        RequestList mainRequest = new RequestList();
        mainRequest.setDeliveryLocation(agent.getContainer().getContainerName());
        mainRequest.setDeliveryAddress(agent.getAddress());
        mainRequest.setRequestid(Long.toString(System.currentTimeMillis()%(agent.getAddress().hashCode()*agent.getCurrentTime())));
        // petr -- possible dividing by zero
        mainRequest.setRequestTime(agent.getCurrentTime());
        for (StoreRec lr : goodsDemand.values()) {
            long count = calcCommodityDemand(lr,dontMultiply);
            if ( count >= minLotSize) {
                // start the cfp for the goods
                mainRequest.getBatch().add(LocationUtils.prepareGoodsCFPContent(agent,lr.getCommodity(), count));
            }
        }
        requestLists.add(mainRequest);

        return requestLists;
    }

    /**
     * Returns a list of Request lists to be used for CFP's in this turn. List with no elements means nothing to buy.
     * Returns one RequestList per commodity requested.
     * @param agent Owner agent
     * @return List of RequestLists. Each of them spuns a cfp.
     */
    synchronized public List prepareSingleBatchGoodsDemand(LocationAgent agent,boolean dontMultiply) {
        LinkedList requestLists = new LinkedList();
        for (Iterator i = goodsDemand.values().iterator();i.hasNext();)
        {
            StoreRec lr = (StoreRec) i.next();
            long count = calcCommodityDemand(lr,dontMultiply);
            if (count >= minLotSize) {
                RequestList req = new RequestList();
                req.setDeliveryLocation(agent.getContainer().getContainerName());
                req.setDeliveryAddress(agent.getAddress());
                req.setRequestid(Long.toString(System.currentTimeMillis()%(agent.getAddress().hashCode()*agent.getCurrentTime())+lr.hashCode()));
                req.setRequestTime(agent.getCurrentTime());
                req.getBatch().add(LocationUtils.prepareGoodsCFPContent(agent,lr.getCommodity(), count));
                requestLists.add(req);
            }
        }
        return requestLists;
    }

    /**
     * Determines how many pieces of given type shall we buy from other sources.
     * @param demandRec record from the demand store
     * @return number of units to order
     */
    private long calcCommodityDemand(StoreRec demandRec, boolean dontMultiply) {
       long count = demandRec.getCount();
        if (!dontMultiply) {
          if (count < minLotSize) {
            count *= 10;
          }
          else if (count < 10000) {
            count *= 3;
          }
          return count -= (getPrivateAndAvailableStock(demandRec.getCommodity())/2);
        }
        return count;
    }

    /**
     * Registers the goods acquired from other agents, but not delivered yet.
     * @param rl contains the list of acquired commodities
     * @param prop retained proposal, detailing coverage
     */
    synchronized  public void goodsAcquired(RequestList rl, Proposal prop)  {
    	for (Batch batch : rl.getBatch()) {
            // determine the quantity
            Batch b = getBatchCoverageFromProposal(batch, prop);
            if (null != b) {
                expectedGoods.increase(b);
            }
        }
    }

    /**
     * Registers the goods acquired from other agents, but not delivered yet.
     * @param rl contains the list of acquired commodities
     * @param prop1 retained proposal, detailing coverage
     * @param prop2 retained proposal, detailing coverage by transport, must be multiplied with the first prop...
     */
    synchronized public void goodsAcquired(RequestList rl, Proposal prop1, Proposal prop2) {
    	for (Batch batch : rl.getBatch()) {
            // determine the quantity
            Proposal cprop = integrateTwoProposals(prop1, prop2);
            Batch b = getBatchCoverageFromProposal(batch, cprop);
            if (null != b) {
                expectedGoods.increase(b);
            }
        }
    }

    /**
     * Called when acquisition already in books fails.
     * @param rl contains the list of acquired commodities - to remove from the list
     */
    synchronized public void acquisitionFailed(RequestList rl) {
    	for (Batch batch : rl.getBatch()) {
            expectedGoods.decrease(batch);
        }
    }

    /**
     * Called to register goods as proposed - proposal was sent, but not evaluated yet.
     * @param rl request list of the proposal
     * @param prop proposal, detailing coverage
     */
    synchronized public void registerProposedGoods(RequestList rl, Proposal prop) throws NegativeStockException {
    	for (Batch batch : rl.getBatch()) {
    		// determine the quantity
            Batch b = getBatchCoverageFromProposal(batch, prop);
            if (null != b) {
                availableStock.decreaseNonNegative(b);
                reservedStock.increase(b);
            }
        }
    }
    
    /**
     * Called to register goods - full coverage is assumed.
     * @param rl request list of the proposal
     * @param prop proposal, detailing coverage
     */
    synchronized public void registerProposedBatch(Batch b) throws NegativeStockException {
		availableStock.decreaseNonNegative(b);
		reservedStock.increase(b);
    }

    /**
     * Called when we have received acceptation of our bid - puts the goods to waiting for pickup storage
     * @param rl request list of the accepted proposal
     */
   synchronized public void reservedGoodsAccepted(RequestList rl, Proposal prop) throws NegativeStockException {
	   for (Batch batch : rl.getBatch()) {
            Batch b = getBatchCoverageFromProposal(batch, prop);
            if (null !=b) {
                reservedStock.decreaseNonNegative(b);
                waitingForPickUp.increase(b);
                agregateSold.increase(b);
            }
        }
    }

    /**
     * Called when we have received acceptation of our bid and the transport was arranged for
     * - puts the goods to waiting for pickup storage
     * @param rl request list of the accepted proposal
     * @param prop1 goods coverage by agent
     * @param prop2 transport coverage by transporters
     */
   synchronized public void reservedGoodsAccepted(RequestList rl, Proposal prop1, Proposal prop2) throws NegativeStockException {
	   for (Batch batch : rl.getBatch()) {
            Proposal cprop = integrateTwoProposals(prop1, prop2);
            Batch unreserve = getBatchCoverageFromProposal(batch, prop1);
            Batch b = getBatchCoverageFromProposal(batch, cprop);
            if (null !=b) {
                // dereserve the whole reserved quantity
                reservedStock.decreaseNonNegative(unreserve);
                // schedule the final traded quantity for transport
                waitingForPickUp.increase(b);
                agregateSold.increase(b);
                // put the remaining reserved quantity back to the available stock
                availableStock.increase(b.getComodityName(), unreserve.getCount() - b.getCount());
            }
        }
   }

   /**
    * Adds a batch to the list of batches waiting to be picked up.
    * @param b
    */
   synchronized public void increaseWaitingForPickUp(Batch b) {
     waitingForPickUp.increase(b);
   }

    /**
     * Called when the proposal was refused
     * @param rl refused proposal
     */
    synchronized public void reservedGoodsRefused(RequestList rl, Proposal prop) throws NegativeStockException {
    	for (Batch batch : rl.getBatch()) {
            Batch b = getBatchCoverageFromProposal(batch, prop);
            if (null !=b) {
                reservedStock.decreaseNonNegative(b);
                availableStock.increase(b);
            }
        }
    }

    /**
     * Registers the physical delivery of goods acquired from other agents. Goods are stored as available stock.
     * @param b received the batch
     */
    synchronized public void goodsReceived(Batch b)  {
        // first, decrease the expected quantity
        if (expectedGoods.decrease(b))  {
        	// goods are for local consumption
        	// now, increase the store
        	privateStock.increase( b);
            agregateReceived.increase(b);
        } else  {
        	// goods are stored for future transport to another destination
        	waitingForPickUp.increase(b);
        }
        callRegistered(b, getIncreasedID());
    }

    /**
     * Registers the physical take-over of goods by the transporter/driver agent.
     * Goods are removed from waitingForPuckUp.
     * @param b transported batch
     */
    synchronized public void goodsUnload(Batch b) {
        // first, decrease the expected quantity
        waitingForPickUp.decrease(b);
    }

    /**
     * Unloads the goods from the driver and passes it to the location.
     * @param b
     * @param wantedFill
     * @param reallyUnload - if false this method does not actually unload the batch, it only
     * calculates the successrate
     * @return the successrate - number saying how much of the wanted amount has been delivered.
     */
    synchronized public double goodsUnload(Batch b, double wantedFill, boolean reallyUnload) {
        double successRate = waitingForPickUp.decrease(b, wantedFill, reallyUnload);
        return successRate;
    }



    /**
     * Called by generators to increase the quantity of goods on stock.
     * @param comname type created
     * @param count quantity created
     */
    synchronized public void goodsCreated(String comname, long count) {
        // increase the store
        availableStock.increase( comname, count);
        agregateProduced.increase( comname, count);
    }

    /**
     * Called by generators to consume certain quantity of goods available on the stock.
     * @param comname type consumed
     * @param count quantity consumed
     * @return true if available, false if not
     */
    synchronized public boolean goodsConsumed(String comname, long count) {
        // decrease the store if possible
        if (count <= getPrivateAndAvailableStock(comname) ) {
            // stock available
            goodsRequired(comname, count);
            return true;
        }
        return false;
    }

    /**
     * Used by Factory generators to commands goods for the next turn.
     * @param commodity commodity to obtain
     * @param count quantity to get
     */
    synchronized public void command(String commodity, long count) {
        // request the stock required by generator from the others
        goodsDemand.increase( commodity, count);
        agregateDemand.increase( commodity, count);

    }

    /**
     * Cancels part of the demand for the given commodity.
     * @param comname Name of the commodity
     * @param countToCancel Count of demanded goods to cancel
     */
    public void cancelDemand(String comname, long countToCancel) {
        long count = Math.max(Math.min(countToCancel, goodsDemand.getCount(comname)),0);
        try {
            goodsDemand.decreaseNonNegative(comname, count);
        }
        catch (NegativeStockException e){}
    }

    /**
     * Called by generators to request certain quantity of goods.
     * Goods will be taken from store or commanded from other locations. Satisfies only the complete demand, not partial.
     * @param comname type consumed
     * @param count quantity consumed
     * @return true if available, false if commanded
     */
    synchronized public boolean goodsRequiredComplete(String comname, long count) {
        if (count <= getPrivateAndAvailableStock(comname) )  {
            // stock available
            goodsRequired(comname, count);
            return true;
        } else {
            // request the stock required by generator from the others
            return false;
        }
    }

    /**
     * Called by generators to request certain quantity of goods. Goods are taken from private, then from public stock.
     * If the satisfaction is only partial, will get at least what is available and order the rest.
     * @param comname type consumed
     * @param count quantity consumed
     * @return quantity available that was really consumed
     */
    synchronized public long goodsRequired(String comname, long count) {
        // return what is available on the stock
        long priv = privateStock.getCount(comname);
        long av = availableStock.getCount(comname);
        // try to satisfy from local stock first...
        long privtake = Math.min(count, priv);
        long avtake = Math.min(count - privtake, av);
        if (0 < privtake){ privateStock.decrease( comname, privtake);}
        if (0 < avtake ){ availableStock.decrease( comname, avtake);}
        return avtake + privtake;
    }


    /**
     * Returns the stock count available for proposals to others.
     * @param comodityname name of the commodity
     */
    synchronized public long getAvailableStock(String comodityname) {
        StoreRec sr = ((StoreRec)availableStock.get(comodityname));
        if (null != sr) {
            return sr.getCount();
        }
        return 0;
    }

    /**
     * Returns the stock count available for local consumers - cumulated private and available.
     * @param comodityname name of the commodity
     */
    synchronized public long getPrivateAndAvailableStock(String comodityname) {
        long ar = availableStock.getCount(comodityname);
        long pr = privateStock.getCount(comodityname);
        return ar + pr;
    }

    /**
     * Returns daily demand of the commodity
     * @param commodityname name of the commodity
     * @return long daily demand
     */
    synchronized public long getDemand(String commodityname) {
      return goodsDemand.getCount(commodityname);
    }

    synchronized public  Collection getGoodsAvailable() {
      return availableStock.values();
    }
    
    synchronized public  Collection getGoodsPrivate() {
      return privateStock.values();
    }
    
    synchronized public  Collection getGoodsDemand() {
      return goodsDemand.values();
    }

    synchronized public Collection getGoodsReserved() {
      return reservedStock.values();
    }

    synchronized public Collection getGoodsWaitingForPickUp() {
      return waitingForPickUp.values();
    }
    
    synchronized public Collection getGoodsExpected() {
      return expectedGoods.values();
    }
    
    synchronized public Collection getAgregateDemand() {
      return agregateDemand.values();
    }
    
    synchronized public Collection getAgregateReceived() {
        return agregateReceived.values();
    }
    
    synchronized public Collection getAgregateProduced() {
      return agregateProduced.values();
    }
    
    synchronized public Collection getAgregateSold() {
     return agregateSold.values();
    }

    synchronized public StoreRec getGoodsAvailableComodity(String comodity) {
    	return (StoreRec)(availableStock.get(comodity));
    }
    
    synchronized public StoreRec getGoodsPrivateComodity(String comodity) {
    	return (StoreRec)(privateStock.get(comodity));
    }
    
    synchronized public StoreRec getGoodsDemandComodity(String comodity) {
    	return (StoreRec)(goodsDemand.get(comodity));
    }
    
    synchronized public DetailedStoreRec getGoodsReservedComodity(String comodity) {
    	return (DetailedStoreRec)(reservedStock.get(comodity));
    }
    
    synchronized public DetailedStoreRec getGoodsWaitingForPickUpComodity(String comodity) {
    	return (DetailedStoreRec)(waitingForPickUp.get(comodity));
    }
    
    synchronized public DetailedStoreRec getGoodsExpectedComodity(String comodity) {
    	return (DetailedStoreRec)(expectedGoods.get(comodity));
    }
    
    synchronized public StoreRec getAgregateDemandComodity(String comodity) {
    	return (StoreRec)(agregateDemand.get(comodity));
    }
    
    synchronized public StoreRec getAgregateReceivedComodity(String comodity) {
    	return (StoreRec)(agregateReceived.get(comodity));
    }
    
    synchronized public StoreRec getAgregateProducedComodity(String comodity) {
    	return (StoreRec)(agregateProduced.get(comodity));
    }
    
    synchronized public StoreRec getAgregateSoldComodity(String comodity) {
    	return (StoreRec)(agregateSold.get(comodity));
    }

    /**
     * Prints out what is in the store.
     * @param caption
     * @return
     */
    public String toString(String caption) {
        StringBuffer sb = new StringBuffer(caption + "\n");
        sb.append("Available\n");
        for (Iterator iterator = availableStock.values().iterator(); iterator.hasNext();) {
            StoreRec storeRec = (StoreRec) iterator.next();
            sb.append(storeRec);
            sb.append("\n");
        }
        sb.append("Reserved\n");
        for (Iterator iterator = reservedStock.values().iterator(); iterator.hasNext();) {
            StoreRec storeRec = (StoreRec) iterator.next();
            sb.append(storeRec);
            sb.append("\n");
        }
        sb.append("Waiting for pickup\n");
        for (Iterator iterator = waitingForPickUp.values().iterator(); iterator.hasNext();) {
            StoreRec storeRec = (StoreRec) iterator.next();
            sb.append(storeRec);
            sb.append("\n");
        }
        return sb.toString();
    };

    /**
     * Returns a batch the parties have agreed on. For given batch from the requests, parses the proposal and determines actual batch size to store/destore.
     *
     * @param requestBatch Original batch from the request
     * @param prop Proposal with all coverages
     * @return null if not covered at all, new Batch with count = count*coverage if covered at least partially
     */
    public static Batch getBatchCoverageFromProposal(Batch requestBatch, Proposal prop) {
        for (Iterator it = prop.getItemCoverage().iterator(); it.hasNext();) {
            ItemCoverage ic = (ItemCoverage) it.next();
            if ( ic.getItemid() == requestBatch.getBatchid()) {
                // create batch with accepted value
                Batch b = new Batch();
                b.setComodityName(requestBatch.getComodityName());
                b.setBatchid(requestBatch.getBatchid());
                b.setCount((long)Math.floor(ic.getCoverage()*requestBatch.getCount()));
                return b;
            }
        }
        // no coverage
        return null;
    }

    /**
     * Multiplies the corresponding elements of two serial offers on single delivery.
     * Used to integrate coverage by selling location with transporter.
     * @param prop1 first proposal
     * @param prop2 second proposal
     * @return proposal with the same request id as a first one and items corresponding to the multiplied coverage of both proposals
     * , where we multiply and include by the same itemid
     */
    public static Proposal integrateTwoProposals(Proposal prop1, Proposal prop2) {
        Proposal res = new Proposal();
        res.setRequestid(prop1.getRequestid());
        res.setTotalPrice(0);
        for (Iterator it = prop1.getItemCoverage().iterator(); it.hasNext();) {
            ItemCoverage ic = (ItemCoverage) it.next();
            for (Iterator it2 = prop2.getItemCoverage().iterator(); it2.hasNext();) {
                ItemCoverage ic2 = (ItemCoverage) it2.next();
                if (ic2.getItemid() == ic.getItemid()) {
                    ItemCoverage nic = new ItemCoverage();
                    nic.setItemid(ic.getItemid());
                    nic.setCoverage(ic.getCoverage()*ic2.getCoverage());
                    res.getItemCoverage().add(nic);
                    break;
                }
            }
        }
        return res;
    }

    /**
     * Registers a listener agent with a batch.
     * When null, this method creates the field registeredBatches.
     * @param b - the Batch
     * @param listener - the listener agent
     * @param task - the task that should be done when the batch arrives (the task is used to notify
     * the listener agent that the batch arrived and how much per cents of the promised amount have
     * been delivered).
     * @return true if the listener or the task has not been previously registered. false otherwise.
     */
    public synchronized boolean registerBatch(Batch b, Address listener, LocSubscribeParticipantTask task) {
      if (registeredBatches == null) {
        registeredBatches = new HashMap<Long,LinkedHashMap<Address,LocSubscribeParticipantTask>>();
      }

      if ((registeredBatches.get(b.getBatchid()) != null) && (registeredBatches.get(b.getBatchid()) instanceof LinkedHashMap)) {
    	  LinkedHashMap<Address,LocSubscribeParticipantTask> batchList = registeredBatches.get(b.getBatchid());
      	if (batchList.get(listener)!=null) {
          return false;
        } else {
          batchList.put(listener,task);
          return true;
        }
      } else {
    	  LinkedHashMap<Address,LocSubscribeParticipantTask> tasks = new LinkedHashMap<Address,LocSubscribeParticipantTask>();
    	  tasks.put(listener,task);
    	  registeredBatches.put(b.getBatchid(), tasks);
          return true;
      }
    }
    
    /**
     * Deregisters the listener agent from the batch.
     * @param b
     * @param listener
     * @param task
     */
    public synchronized void deregisterBatch(Batch b, Address listener, SubscribeParticipantTask task) {
        if (registeredBatches != null)  {
        	if ((registeredBatches.get(b.getBatchid()) != null) && (registeredBatches.get(b.getBatchid()) instanceof LinkedHashMap)) {
        		LinkedHashMap<Address,LocSubscribeParticipantTask> ll = registeredBatches.get(b.getBatchid());
        		ll.remove(listener);
        		registeredBatches.put(b.getBatchid(), ll);
        	}
        } else {
          System.out.println("no registration to cancel error");
        }
      }

    /**
     * Calls the listener agents upon the arrival of the batch.  
     * @param b
     * @param id
     */
	private void callRegistered(Batch b, Long id) {
		if (registeredBatches != null) {
			LinkedHashMap<Address,LocSubscribeParticipantTask> ll = registeredBatches.get(b.getBatchid());
    		if ( ll != null ) {
				for (LocSubscribeParticipantTask task : ll.values()) {
    				if (task != null) {
    					task.goodsDelivered(b,id);
    				}
    			}
    		}
    	}
	}

	// TODO - remove?
	/**
	 * Returns unique id
	 * @return
	 */
	private long getIncreasedID() {
    	return _id++;
    }

	/**
	 * This class represents a store - it keeps track of all the commodities that are stored inside it.
	 * @author Eduard Semsch
	 *
	 */
    static abstract class OneStore extends HashMap<String,StoreRec> {
        /**
         * Creates a new record of the appropriate type.
         * @param comname name of the commodity
         * @param count initial capacity
         * @return created object
         */
        abstract StoreRec getNewStoreRec(String comname, long count);

        /** Creates a new record from the received Batch. Type of the record depends on the type of the store.
         *
         * @param b received batch
         * @return created StoreRec
         */
        abstract public StoreRec getNewStoreRec(Batch b);

        /**
         * Returns the actual store of given commodity, identified by the name.
         * @param comname Name of the commodity.
         * @return count, 0 if not present.
         */
        synchronized public long getCount(String comname) {
            StoreRec sr = ((StoreRec)get(comname));
            if (null != sr) {
                return sr.getCount();
            }
            return 0;
        }
        /**
         * Returns a reference to the commodity identified by the name.
         * @param comname
         * @return
         */
        synchronized public StoreRec getRecord(String comname) {
            return ((StoreRec)get(comname));
        }

        /** 
         * Increases the count on the record of one store
         */
        synchronized void increase( Batch b) {
        	StoreRec sr = (StoreRec) get(b.getComodityName());
        	if (sr != null)	{
        		sr.increase(b);
        	} else {
        		StoreRec nsr = getNewStoreRec(b);
        		put(b.getComodityName(), nsr);
        	}
        }

        /** Decreases the count on the record of one store. Throws an exception if the new value would be negative.*/
        synchronized void decreaseNonNegative( Batch b) throws NegativeStockException  {
            StoreRec sr = (StoreRec)get(b.getComodityName());
            if (null != sr ) {
                sr.decreaseNonNegative(b);
            } else {
                throw new NegativeStockException("Goods not found on stock. Commodity: " + b.getComodityName());
            }
        }

        /** Decreases the count on the record of one store. Throws an exception if the new value would be negative.*/
        synchronized void decreaseNonNegative( Batch b, String error) throws NegativeStockException {
            StoreRec sr = (StoreRec)get(b.getComodityName());
            if (null != sr ) {
                sr.decreaseNonNegative(b, error);
            } else {
                throw new NegativeStockException("Goods not found on stock.");
            }
        }

        /** Decreases the count on the record of one store. If the batch is not there, it is not removed.
         *  If the stock is too low, it becomes zero.
         *  @return true if decreased, false if the batch was not found*/
        synchronized boolean decrease( Batch b) {
            StoreRec sr = (StoreRec)get(b.getComodityName());
            if (null != sr ) {
                return sr.decrease(b);
            }
            return false;// batch not found
        }

        /**
         * Method returns percentage of requested goods unloaded from location. If this method returns number smaller than 1,
         * it means that no cargo was unloaded in fact. 
         */
        synchronized double decrease(Batch b, double requestedFill, boolean reallyUnload) {
            StoreRec sr = (StoreRec)get(b.getComodityName());
            if (null != sr ) {
                return sr.decrease(b, requestedFill, reallyUnload);
            }
            return 0; // good not found
        }
    }

    /** Class used to represents simple stores. */
    static class SimpleStore extends OneStore {

        public StoreRec getNewStoreRec(String comname, long count) {
            return new StoreRec(comname, count);
        }

        public StoreRec getNewStoreRec(Batch b)  {
            return new StoreRec(b.getComodityName(), b.getCount());
        }

        /** Increases the count of the record on the store.*/
        synchronized void increase( String comname, long count) {
            StoreRec sr = (StoreRec)get(comname);
            if (null != sr) {
                sr.increase(count);
            } else {
                StoreRec nsr = getNewStoreRec(comname, count);
                put(comname, nsr);
            }
        }

        /** Increases the count on the record of one store. Throws an exception if the new value would be negative.*/
        synchronized void decreaseNonNegative(String comname, long count) throws NegativeStockException {
            StoreRec sr = (StoreRec)get(comname);
            if (null != sr ) {
                sr.decreaseNonNegative(count);
            } else {
                throw new NegativeStockException("Goods not found on stock.");
            }
        }

        /** Decreases the count on the record of one store. If the batch is not there, it is not removed.
         *  If the stock is too low, it becomes zero.*/
        synchronized void decrease( String comname, long count) {
            StoreRec sr = getRecord(comname);
            if (null != sr ) {
                sr.decrease(count);
            }
        }
    }

    /** Class used to represents detailed stores.  */
    static class DetailedStore extends OneStore {
        /**
         * Creates a new record of the appropriate type.
         * @param comname name of the commodity
         * @param count initial capacity
         * @return created object
         */
        public StoreRec getNewStoreRec(String comname, long count) {
            throw new UnsupportedOperationException("Use different operation for this type of Store.");
        }

        /** Creates a new record from the received Batch. Type of the record depends on the type of the store.
         *
         * @param b received batch
         * @return created StoreRec
         */
        public StoreRec getNewStoreRec(Batch b) {
            return new DetailedStoreRec(b);
        };
    }
}
