package across.agents.location;

import java.util.HashMap;
import java.util.Map;

import across.data.simulation.SimulationObjectDescription;
import across.simulation.constants.AcrossEntityConstants;
import across.simulation.constants.AcrossSimulationConstants;
import across.simulation.constants.AcrossTopicConstants;
import aglobex.simulation.ontology.Sensor;
import aglobex.simulation.ontology.SensoryData;
import aglobex.simulation.ontology.SensoryDataRecord;

/**
 * This class keeps track of the effects of the simulation object in the vicinity
 * of the location on the location.
 * @author Eduard Semsch
 *
 */
public class SimulationObjectsEffects {

	protected double aggregateFire = 0;
	
	protected int actualFire;
	
	protected double aggregateInsurgency = 0;
	
	protected int actualInsurgency; 
	
	protected double aggregateConstructionDamage = 0;
	
	protected int actualConstructionDamage;
	
	protected int numberOfWounded = 0;
	
	protected long startTime;
	
	/** The last time the data was updated */
	private long lastTimeStamp;
	
	/** LocationAgent that uses this class */
	private LocationAgent owner;
	
	/** Simulation data of the owner location agent */
	private LocationAgentSimulationData ownerData;
	
	/** Objects active in the vicinity of the location */
	private Map<String,SimulationObjectDescription> activeObjects = new HashMap<String,SimulationObjectDescription>();

	public SimulationObjectsEffects(LocationAgent owner, LocationAgentSimulationData ownerData) {
		this.owner = owner;
		this.ownerData = ownerData;
	}
	
	/**
	 * Parses data incoming from the location sensors.
	 * @param sd
	 */
	public void parseSensoryData(SensoryData sd) {
		calculateAggregate(sd.simulationTimestamp-lastTimeStamp);
		lastTimeStamp = sd.simulationTimestamp;
		for (Sensor sensor : sd.sensor) {
			for (SensoryDataRecord sdr : sensor.sensoryDataRecord) {
				if(sdr.targetEntityType.equals(AcrossEntityConstants.SIMULATION_OBJECT_TYPE)) {
					if(!activeObjects.containsKey(sdr.targetContainerName) && isInVincinity(sdr)) {
						activeObjects.put(sdr.targetContainerName, null);
					}
				}
			}
		}
		askObjectsDescription();
	}
	
	/**
	 * Process simulation object description.
	 * @param sod
	 */
	public void objectDescription(SimulationObjectDescription sod) {
		activeObjects.put(sod.getName(), sod);
	}
	
	public void startAggregation(long startTime) {
		this.startTime = startTime;
		aggregateConstructionDamage = 0;
		aggregateFire = 0;
		aggregateInsurgency = 0;
	}
	
	/**
	 * Calculates the aggregate effects of the disasters for the period.
	 *
	 */
	private void calculateAggregate(long effectPeriod) {
		numberOfWounded = 0;
		for (SimulationObjectDescription sod : activeObjects.values()) {
			if(sod!=null) {
				aggregateConstructionDamage += effectPeriod*sod.getConstructionDamage();
				aggregateFire += effectPeriod*sod.getExtentOfFire();
				aggregateInsurgency += effectPeriod*sod.getInsurgencyLevel();
				numberOfWounded += sod.getNumberOfWounded();
			}
		}
	}
	
	/**
	 * Asks for description of all the active simulation objects.
	 *
	 */
	private void askObjectsDescription() {
		for (String objName : activeObjects.keySet()) {
			owner.gisShell.submitTopicToServer(AcrossTopicConstants.TOPIC_SIMULATION_OBJECT, objName, AcrossSimulationConstants.OBJECT_DESCRIPTION);
		}
	}
	
	/**
	 * Is the simulation object in vicinity of this location?
	 * @param sdr
	 * @return
	 */
	private boolean isInVincinity(SensoryDataRecord sdr) {
		double diffX = sdr.ownerX-sdr.targetX;
		double diffY = sdr.ownerY-sdr.targetY;
		double diffZ = sdr.ownerZ-sdr.targetZ; 
		double distance = (diffX*diffX+diffY*diffY);
		distance = (distance*distance+diffZ*diffZ);
		if(distance<=ownerData.locVicinity) {
			return true;
		} else {
			return false;
		}
	}

	public double getAggregateConstructionDamage() {
		return aggregateConstructionDamage;
	}

	public double getAggregateFire() {
		return aggregateFire;
	}

	public double getAggregateInsurgency() {
		return aggregateInsurgency;
	}
}
