package across.agents.emergency.storage;

import java.util.Arrays;
import java.util.LinkedList;

import across.data.Batch;
import across.simulation.constants.AcrossDataConstants;
import across.simulation.constants.AcrossTopicConstants;
import aglobe.container.agent.CMAgent;
import aglobe.container.sysservice.directory.DirectoryException;
import aglobe.container.sysservice.directory.DirectoryService;
import aglobe.container.task.ConversationUnit;
import aglobe.container.task.Task;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobe.service.gis.client.GISClientService;
import aglobe.service.gis.client.GISTopicListener;
import aglobex.protocol.request.RequestParticipantTask;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.entity.EntityDescriptor;
import aglobex.simulation.protocol.cnp.CNPParticipantTask;
import aglobex.simulation.sensor.UniversalSensorAgent;

/**
 * This agent represents a storage with emergency material or a hospital.
 * @author Eduard Semsch
 *
 */
public class EmergencyStorageAgent extends CMAgent implements ClientServerTopicConstants, GISTopicListener {

	/** Type of this agent by which it is known at the directory service. */
	public static final String TYPE = "EmergencyStorageAgent";
	
	/** GIS shell. */
	protected GISClientService.Shell gisShell;
	
	/** Directory service shell. */
	protected DirectoryService.Shell dirShell;
	
	/** Location of this agent. */
	protected String location;
	
	@Override
	public void init(AgentInfo ai, int initState) {
		gisShell = (GISClientService.Shell) this.getContainer().getServiceManager().getService(this, GISClientService.SERVICENAME);
		dirShell = (DirectoryService.Shell) this.getContainer().getServiceManager().getService(this, DirectoryService.SERVICENAME);
		if(gisShell!=null && dirShell!=null) {
			gisShell.subscribeTopic(TOPIC_ENTITY_CONNECTION, this);
			gisShell.subscribeTopic(TOPIC_ENTITY_CONTROL, this);
			gisShell.subscribeTopic(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND, this);
			gisShell.subscribeTopic(UniversalSensorAgent.TOPIC_SENSORY_DATA,this);
			gisShell.subscribeTopic(AcrossTopicConstants.TOPIC_SIMULATION_OBJECT, this);
		} else {
			logSevere("Unable to locate GIS!");
			this.stop();
		}
		try {
			dirShell.register(this, new LinkedList<String>(Arrays.asList(new String[]{TYPE})));
		} catch (DirectoryException e) {
			e.printStackTrace();
		}
		this.setIdleTask(new EmergencyStorageAgentIdleTask(this));
	}
	
	/**
	 * Agents idle task - handles new incoming messages that 
	 * does not have known conversation ids.
	 * @author Eduard Semsch
	 *
	 */
	protected class EmergencyStorageAgentIdleTask extends Task implements MessageConstants {

		public EmergencyStorageAgentIdleTask(ConversationUnit cu) {
			super(cu);
		}

		/**
		 * This method handles all the incoming messages that does not have
		 * known conversation ids. 
		 */
		@Override
		protected void handleIncomingMessage(Message m) {
			if(m.getPerformative().equals(REQUEST)) {
				handleStorageRequest(m);
			} else if(m.getProtocol().equals(CONTRACT_NET)) {
				handleStorageCNP(m);
			}
		}
	}
	
	/** Handles the request to put something in the storage or to take something out. */
	protected void handleStorageRequest(Message m) {
		new RequestParticipantTask(this, m) {

			@Override
			protected void processRequest(Message requestMessage) {
				Batch b = (Batch) requestMessage.getContent();
				informDone();
			}
			
		};
	}
	
	/** Handles CNP for material. */
	protected void handleStorageCNP(Message m) {
		new CNPParticipantTask(this,10000,m) {

			// Resource always availiable so choose randomly.
			@Override
			protected void prepareProposal() {
				sendProposal(location);
			}

			/** Not implemented. */
			@Override
			protected void proposalAccepted(Message acceptMessage) {}

			/** Not implemented. */
			@Override
			protected void proposalRefused(Message refuseMessage) {}
			
			@Override
			protected void timeout(){}
			
		};
	}

	/**
	 * This method handles the incoming topics from the simulation.
	 */
	public void handleTopic(String topic, Object content, String reason) {
		if(topic.equalsIgnoreCase(TOPIC_ENTITY_CONNECTION)) {
			if(content instanceof EntityDescriptor) {
				handleConfiguration((EntityDescriptor) content);
			}
		}
	}
	
	/**
	 * This method handles configuration.
	 * @param ed
	 */
	private void handleConfiguration(EntityDescriptor ed) {
		
		location = ed.confParamsString.get(AcrossDataConstants.ENTITY_START_NODE);
		
		 // display visio information
        String visioInfo = "<b>" + this.getName();
        gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_INFO, visioInfo);
	}
	
}
