package across.agents.emergency.repair;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import across.agents.driver.DriverBehaviour;
import across.agents.driver.data.RoadInfo;
import across.agents.driver.data.Route;
import across.agents.driver.data.RouteInfo;
import across.data.AcrossMapArcs;
import across.data.AcrossMapDistances;
import across.data.AcrossMapNodes;
import across.data.Arc;
import across.data.Node;
import across.data.Waypoint;
import across.simulation.constants.AcrossControlConstants;
import across.simulation.constants.AcrossDataConstants;
import across.util.pathfinding.AStarModPathfinding;
import across.util.pathfinding.Pathfinding;
import across.util.pathfinding.RoadEvaluator;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.entity.EntityDescriptor;

/**
 * This class is responsible for performing movement actions of the RepairVehicleAgent.
 * This simulation class controlled by this class is {@link DriverBehaviour}.  This class
 * enables these actions:
 * <ol>
 * 	<li> Go from one node on map to another. </li>
 * 	<li> Stop movement at any position. </li>
 *  <li> Evaluate the cost of movement. </li>
 * </ol>
 * @author Eduard Semsch
 *
 */
public class RepairVehicleMovement implements ClientServerTopicConstants {
	
	/** Owner of this class - the repair vehicle agent. */
	protected RepairVehicleAgent owner;
	
	/** Algorithm for pathfinding */
	protected Pathfinding pathfinding;
	
	/** Nodes of the map (map). */
	protected LinkedList<String> locNames = new LinkedList<String>(); 
	  
	/** The distances between the nodes. */
	protected RoadInfo roads[][];
	
	/** 
	 * This is either the last node through which the vehicle drove or
	 * the node where the vehicle is standing.  See headingTo as well.  
	 */
	protected String actualLoc;
	
	/**
	 * This is the location where the vehicle is currently heading to.  If
	 * null this vehicle is standing still.
	 */
	protected String headingTo;
	
	/** The set of waypoints this agent is going to go through. */
	protected LinkedList<Waypoint> waypoints = new LinkedList<Waypoint>(); 
	
	/** Current position on the map.  These are the precise x,y,z coordinates of the vehicle. */
	protected double[] position;
	
	/** Constructor - the class needs callback to the owner agent. */
	public RepairVehicleMovement(RepairVehicleAgent owner) {
		this.owner = owner;
	}
	/**
	 * Handles all topics concerning movement.  The owner agent should pass all the topics
	 * concerning movement to this class.
	 * @param content
	 * @param reason
	 */
	public void handleMovementTopic(Object content, String reason) {
	  if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_BAD_NODE)) {
  		  owner.logSevere("Bad node chosen!");
	  } else if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_FAILURE)) {
		  owner.logSevere("Movement failure!");
	  } else if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_STOPPED) && content instanceof String) {
		  // The actual location (the last node the driver drove through)
		  actualLoc = (String) content;
		  
		  // if there are no waypoints scheduled.
		  if(waypoints.size()==0) {
			  return;
		  }
		  
		  // Remove the first waypoint in the list since it should equal to the node where the vehicle is now.
		  Waypoint wp = waypoints.getFirst();
		  if(wp.getLocationName().equals(actualLoc)) {
		  	waypoints.removeFirst();
		  }
		  	
		  // Go on moving.
		  move();
      	} else if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_STOPPED) && content instanceof double[]) {
      		// if the vehicle stops in between the nodes it receives its precise coordinates.
      		position = (double[]) content;
      	}
	}
	
	/**
	 * Handles configuration of the vehicle movement.  This method should be run before any movement of 
	 * the vehicle starts. 
	 * @param ed
	 * @param velocity
	 * @param consumption
	 */
	public void handleConfiguration(EntityDescriptor ed, final double velocity, final double consumption) {
		this.actualLoc = ed.confParamsString.get(AcrossDataConstants.ENTITY_START_NODE);
		  
		AcrossMapNodes amn = (AcrossMapNodes) ed.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_NODES);
		AcrossMapArcs ama = (AcrossMapArcs) ed.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_ARCS);
		AcrossMapDistances amd = (AcrossMapDistances) ed.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_DISTANCES);
		  
	  	// Get list of node names
	    locNames.clear();
	    for (Node node : (List<Node>) amn.getNode()) {
	    	locNames.add(node.getName());
	    }
	    
	    RoadEvaluator re = new RoadEvaluator() {

	    	public double getPrice(Arc arc) {
				return (1/arc.getQuality()) * arc.getLength() * consumption/100;
			}
			
			public double getTime(Arc arc) {
				return 10*arc.getLength()/velocity;
			}
					  
	    };
	     
	    pathfinding = new AStarModPathfinding(amn.getNode(),ama.getArc(),amd.getDistance(),re, new LinkedList<Arc>());
	}
	
	/**
	 * Says whether the agent is moving or not.
	 * @return
	 */
	public boolean isOnRoute() {
		return waypoints.size()!=0;
	}
	
	/**
	 * Gets the current position.
	 * @return
	 */
	public double[] getPosition() {
		return position;
	}
	
	/**
	 * Drives the vehicle to a random location.
	 * @return
	 */
	public boolean goToRandomLocation() {
		Random rand = new Random(System.currentTimeMillis());
		int locNum = locNames.size();
		String randLoc = locNames.get(rand.nextInt(locNum));
		Waypoint wp = new Waypoint(randLoc);
		Route route = planRoute(new LinkedList<Waypoint>(Arrays.asList(new Waypoint[] {wp})), true);
		return moveAlongRoute(route);
	}
	
	/**
	 * Drives the vehicle to the specified location.
	 * @param location
	 * @return
	 */
	public boolean goToLocation(String location) {
		Waypoint locWay = new Waypoint(location);
		Route route = planRoute(new LinkedList<Waypoint>(Arrays.asList(new Waypoint[]{locWay})),true);
		return moveAlongRoute(route);
	}
	
	/**
	 * Drives the vehicle through the road between specified locations.
	 * @param startLoc
	 * @param endLoc
	 * @return
	 */
	public boolean driveThroughRoad(String startLoc, String endLoc) {
		Waypoint startWaypoint = new Waypoint(startLoc);
		Waypoint endWaypoint = new Waypoint(endLoc);
		Route route = planRoute(new LinkedList<Waypoint>(Arrays.asList(new Waypoint[]{startWaypoint})),true);
		route.getWaypoints().add(endWaypoint);
		return moveAlongRoute(route);
	}
	
	/** Stops movement immediately at the current location. */
	public void stopMovement() {
		controlStop();
	}
	
	/** Stops the vehicle and cancles all planned waypoints. */
	public void cancelMovement() {
		stopMovement();
		waypoints.clear();
	}
	
	/**
	 * Tries to plan a route and returns information about it.
	 * @param to
	 * @param byPrice
	 * @return
	 */
	public Route getRoute(List<Waypoint> wpList, boolean byPrice) {
		Route route = planRoute(wpList, byPrice);
		if(route!=null  && route.getWaypoints().size()!=0) {
			return route;
		} else {
			return null;
		}
	}
	
	/**
	 * Moves the vehicle along the specified route
	 * @param route
	 * @return - true if movement began
	 */
	protected boolean moveAlongRoute(Route route) {
		if(route!=null) {
			waypoints.addAll(route.getWaypoints());
			Waypoint actWay = new Waypoint(actualLoc);
			waypoints.addFirst(actWay);
			move();
			return true;
		} else {
			return false;
		}
	}
	

    /**
     * Plans a route through a set of waypoints and calculates price and time.
     * @param wpList
     * @param byPrice
     * @return 
     */
    protected Route planRoute(List<Waypoint> wpList, boolean byPrice){
      String startPoint;
      Route route = new Route();
      
      if (waypoints.size() == 0) {
      	startPoint = new String(actualLoc);
      } else {
      	startPoint = new String(((Waypoint)waypoints.getLast()).getLocationName());
      }

      double taskTime = 0;
      double taskPrice = 0;

      for (Waypoint wp : wpList) {
        Route oneRoute;
        if (byPrice) {
      	  oneRoute = pathfinding.planRoute(startPoint, wp.getLocationName(), 0, 1);
        } else {
      	  oneRoute = pathfinding.planRoute(startPoint, wp.getLocationName(), 1, 0);
        }
        if (oneRoute == null) {
        	continue;
        }
        taskTime += oneRoute.getRouteInfo().routeTime;
        taskPrice += oneRoute.getRouteInfo().routePrice;
        startPoint = wp.getLocationName();
        // add all intermediate waypoints (with empty request lists)
        route.addWaypoints(oneRoute.getWaypoints());
        route.addWaypoint(wp);  // add the actual waypoint
      }
      route.setRouteInfo(new RouteInfo(taskTime, taskPrice));
      return route;
    }
    
	
	/** Move to the next waypoint (if there's any in the list) */
    protected void move(){
    	if (!isOnRoute()) {
    		headingTo = null;
		    return;
		} else {
			
			Waypoint wp = waypoints.getFirst();
		    
		    String waypointName = wp.getLocationName();
		    
		    headingTo = waypointName;
		
		    controlMoveTo(waypointName);
		    
		 }
	 }
    
    /** Sends a control topic that moves the vehicle to the simulation server. */ 
    protected void controlMoveTo(String waypointName) {
    	owner.gisShell.submitTopic(TOPIC_ENTITY_CONTROL, waypointName, AcrossControlConstants.MOVE_TO);
    }
    
    /** Sends a control topic that stops the vehicle to the simulation server. */
    protected void controlStop() {
    	owner.gisShell.submitTopicToServer(TOPIC_ENTITY_CONTROL, null, AcrossControlConstants.STOP);	
    }
}
