package across.agents.emergency.repair;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import across.agents.driver.data.Route;
import across.agents.emergency.centre.AcrossActivity;
import across.agents.emergency.centre.EmergencyAgent;
import across.data.Waypoint;
import across.data.simulation.EffectOnSimulationObject;
import across.data.simulation.SimulationObjectDescription;
import across.simulation.constants.AcrossControlConstants;
import across.simulation.constants.AcrossDataConstants;
import across.simulation.constants.AcrossMessageConstants;
import across.simulation.constants.AcrossTopicConstants;
import across.visio.oldvisio.VisioConnectionAgent;
import aglobe.container.agent.CMAgent;
import aglobe.container.sysservice.directory.DirectoryException;
import aglobe.container.sysservice.directory.DirectoryListener;
import aglobe.container.sysservice.directory.DirectoryRecord;
import aglobe.container.sysservice.directory.DirectoryService;
import aglobe.container.task.ConversationUnit;
import aglobe.container.task.Task;
import aglobe.container.transport.Address;
import aglobe.container.transport.InvisibleContainerException;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobe.service.gis.client.GISClientService;
import aglobe.service.gis.client.GISTopicListener;
import aglobex.protocol.request.RequestParticipantTask;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.entity.EntityDescriptor;
import aglobex.simulation.protocol.cnp.CNPParticipantTask;
import aglobex.simulation.sensor.UniversalSensorAgent;

/**
 * <p>
 * This vehicle is capable of moving close to an accident/obstacle/disaster (object) and altering it.  
 * The removal (repair) takes some amount of time based on the obstacle characteristics and 
 * on resources of the vehicle.
 * </p>
 * <p>
 * The vehicle has two characteristics concerning the obstacle handling - the effect and the effect speed.
 * The effect describes how much the vehicle can alter the object and the speed how fast it can 
 * apply the effect.   
 * </p>
 * <p>
 * The vehicle has a specified set of activities that it can perform.  These are descibed in a specific language
 * see {@link RepairVehicleActivities} and are contents of {@link AcrossActivity}.  The activities implemented are:
 * <ol>
 * 	<li> RepairVehicleActivities.ACTIVITY_GOTO - the driver goes somewhere. </li>
 * 	<li> RepairVehicleActivities.ACTIVITY_EFFECT - the driver applies its effect somewhere. </li>
 * </ol>
 * @author Eduard Semsch
 *
 */
public class RepairVehicleAgent extends CMAgent implements GISTopicListener, ClientServerTopicConstants, DirectoryListener {

	/** Type of the agent - for registering with directory service */
	public static final String TYPE = "Repair";

	/** Period of simulation */
	protected int simulationTimePeriod = 50;
	
	/** Consumption of this vehicle */
	protected double consumption;
	
	/** Velocity of this vehicle */
	protected double velocity;

	/** How does this vehicle effect the simulation objects */
	protected EffectOnSimulationObject myEffect;
	
	/** Speed of the effect on an object. */
	protected int effectSpeed;

	/** GIS shell */
	protected GISClientService.Shell gisShell;
	
	/** DirectoryService Shell */
	protected DirectoryService.Shell dirShell;
	
	/** Class for handling movement of vehicle */
	protected RepairVehicleMovement myMovement = new RepairVehicleMovement(this);
	
	/** Class for handling activities of the vehicle. */
	protected RepairVehicleActivityHandler myActivityHandler = new RepairVehicleActivityHandler(this);
	
	/** Class for handling information from the sensors. */
	protected RepairVehicleSensors mySensors = new RepairVehicleSensors(this);
	
	/** This variable signifies whether the vehicle is currently fulfilling a mission for an emergency centre */
	protected boolean onMission = false;
	
	/** Address of the emergency agent. */
	protected List<Address> emergencyAgents = new ArrayList<Address>();

	/**
	 * Initializes the agent in these steps:
	 * <li> GIS shell. Subscription of simulation topics. </li>
	 * <li> Dir shell. Registration. </li>
	 * <li> Sets the idle task. </li>
	 */
	@Override
	public void init(AgentInfo ai, int initState) {
		gisShell = (GISClientService.Shell) this.getContainer().getServiceManager().getService(this, GISClientService.SERVICENAME);
		dirShell = (DirectoryService.Shell) this.getContainer().getServiceManager().getService(this, DirectoryService.SERVICENAME);
		if(gisShell!=null && dirShell!=null) {
			gisShell.subscribeTopic(TOPIC_ENTITY_CONNECTION, this);
			gisShell.subscribeTopic(TOPIC_ENTITY_CONTROL, this);
			gisShell.subscribeTopic(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND, this);
			gisShell.subscribeTopic(UniversalSensorAgent.TOPIC_SENSORY_DATA,this);
		} else {
			logSevere("Unable to locate GIS or Directory Service!");
			this.stop();
		}
		try {
			dirShell.register(this, new LinkedList<String>(Arrays.asList(new String[]{TYPE})));
		} catch (DirectoryException e) {
			e.printStackTrace();
		}
		dirShell.subscribe(this, EmergencyAgent.TYPE);
		this.setIdleTask(new RepairVehicleIdleTask(this));
	}
	
	/**
	 * This agent's idle task.  Handles the messages with unknown conversation id.
	 * @author Eduard Semsch
	 *
	 */
	public class RepairVehicleIdleTask extends Task implements MessageConstants{
	
		public RepairVehicleIdleTask(ConversationUnit cu) {
			super(cu);
		}

		/**
		 * Handles these messages:
		 * <li> Performative REQUEST + content AcrossActivity - activity to carry out. </li>
		 * <li> Performative REQUEST + content String (AcrossMessageConstants.CANCEL_DRIVER_SCHEDULE) -
		 * the EmergencyCentre frees the driver from the schedule. </li>
		 * <li> CNP, content String (node near which something happened that the driver should go to) - 
		 * contract net for accident handling sent by the EmergencyCentre. </li>
		 */
		@Override
		public void handleIncomingMessage(final Message mes) {
			// request for performing an activity
			if(mes.getPerformative().equalsIgnoreCase(REQUEST)) {
				Object content = mes.getContent();
				if(content instanceof AcrossActivity) {
					newActivityRequestObtained(mes);
				}
				// freeing the driver from the schedule
				else if (content instanceof String) {
					freeDriverRequestObtained(mes);
				}
			}
			// contract net for scheduling the driver to handle an accident.
			else if (mes.getProtocol().equalsIgnoreCase(MessageConstants.CONTRACT_NET)) {
				cnpForAccidentHandlingObtained(mes);
			} 
		}
	}
	
	/** Obtained a CNP for handling an accident. */
	private void cnpForAccidentHandlingObtained(final Message mes) {
		new CNPParticipantTask(this, 1000, mes) {
			
			/** In proposal the driver sends back the cost of driving to the accident. */
			@Override
			protected void prepareProposal() {
				if(!onMission) {
					String location = (String) mes.getContent();
					Waypoint locWay = new Waypoint(location);
					List<Waypoint> wayList = new ArrayList<Waypoint>();
					wayList.add(locWay);
					Route r = myMovement.planRoute(wayList, true);
					Double routePrice = r.getRouteInfo().routePrice;
					sendProposal(routePrice);
				} else {
					sendProposal(null);
				}
			}

			/** Proposal was accepted - the driver is waiting for orders. */
			@Override
			protected void proposalAccepted(Message acceptMessage) {
				displayVisioAction(VisioConnectionAgent.ACTION_CONVOY_STARTED);
				onMission = true;
				// we are on mission - stop moving and cancel all the scheduled waypoints.
				myMovement.cancelMovement();
				// send back accepting message
				cancelTask();
			}
			
			/** Proposal was refused - nothing happens. */
			@Override
			protected void proposalRefused(Message refuseMessage) {
				cancelTask();
			}
			
			/** If the task times out nothing happens - we have not obtained the contract. */
			@Override
			protected void timeout() {}
			
		};
	}
	
	/** Obtained a request that frees this vehicle from emergency centre schedule. */
	private void freeDriverRequestObtained(final Message mes) {
		displayVisioAction(VisioConnectionAgent.ACTION_POLL_YES);
		new RequestParticipantTask(RepairVehicleAgent.this,mes) {

			@Override
			protected void processRequest(final Message requestMessage) {
				String what = (String) requestMessage.getContent();
				if(what.equalsIgnoreCase(AcrossMessageConstants.CANCEL_DRIVER_SCHEDULE)) {
					onMission = false;
					// cancel the rest of the scheduled waypoints - since we are free again!
					myMovement.cancelMovement();
					displayNameInVisio();
				}
			}
			
		};
	}

	/** Obtained a request to perform an activity from the emergency centre. */
	private void newActivityRequestObtained(final Message mes) {
		displayVisioAction(VisioConnectionAgent.ACTION_POLL_NO);
		new RepairRequestParticipantTask(RepairVehicleAgent.this,mes) {

			@Override
			protected void processRequest(final Message requestMessage) {
				AcrossActivity act = (AcrossActivity) requestMessage.getContent();
				myActivityHandler.handleActivity(act, this);
				displayActivityInVisio(act);
			}
			
		};
	}
	
	/**
	 * Handles these simulation topics:
	 * <li> TOPIC_ENTITY_CONNECTION - configuration </li>
	 * <li> TOPIC_ENTITY_CONTROL - movement </li>
	 * <li> TOPIC_SIMULATION_TIME_UPDATE_1_SECOND - simulation time </li> 
	 * <li> TOPIC_SENSORY_DATA - Scans for simulation objects (accidents) that it has commited 
	 * to handle.  If it is currently handling an accident and it disappears from the sensor that
	 * means that the accident has been handled. </li>
	 * <li> TOPIC_SIMULATION_OBJECT - Incoming description of a simulation object. </li>
	 * @param topic
	 * @param content
	 * @param reason
	 */
	public void handleTopic(final String topic, final Object content, final String reason) {
		if(topic.equalsIgnoreCase(TOPIC_ENTITY_CONNECTION)) {
			if(content instanceof EntityDescriptor) {
				handleConfiguration((EntityDescriptor) content);
			}
		} else if (topic.equalsIgnoreCase(TOPIC_ENTITY_CONTROL)) {
			myMovement.handleMovementTopic(content, reason);
		} else if (topic.equalsIgnoreCase(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND)) {
			if(!onMission && !myMovement.isOnRoute()) {
				// if this is a police team, go patrol
				if(amICop()) {
					myMovement.goToRandomLocation();
				}
			} else {
				myActivityHandler.run();
			}
		} else if (topic.equalsIgnoreCase(UniversalSensorAgent.TOPIC_SENSORY_DATA)) {
			myActivityHandler.run();
			mySensors.processSensorTopic(content);
			List<String> accidentsInSight = mySensors.getObjectsInSight();
			if(accidentsInSight.size()>0) {
				reportAccidents(accidentsInSight);
			}
			// if I am a cop and I am not currently on mission look whether there are terrorist in sight
			// and destroy them - with some probability.
			if(!onMission && amICop()) {
				if(mySensors.isEntityTypeInSight("TERRORIST_DRIVER_ENTITY_TYPE")) {
					// destroy the terrorist with some probability
					List<String> terroristsInSight = new ArrayList<String>(mySensors.getEntitiesOfTypeInSight("TERRORIST_DRIVER_ENTITY_TYPE"));
					Random r = new Random(System.currentTimeMillis());
					if(r.nextInt(100) < 100) {
						for (String string : terroristsInSight) {
							if(!string.equals("Terrorist")) {
								destroyEntity(string);
								break;
							}
						}
					}
				}
			}
		} 
	}
	
	/** Tells whether this vehicle is a police team. */
	private boolean amICop() {
		return getName().indexOf("police-team")!=-1;
	}
	
	/**
	 * This method reports the accident found to the emergency agent.
	 * @param sol
	 */
	protected void reportAccidents(List<String> listOfAccidents) {
		for (Address adr : emergencyAgents) {
			for (String objId : listOfAccidents) {
				SimulationObjectDescription sod = mySensors.getObjectsDescription(objId);
				if(sod!=null) {
					Message mes = Message.newInstance();
					mes.setPerformative(MessageConstants.INFORM);
					mes.setContent(sod);
					mes.setReceiver(adr);
					mes.setSender(this.getAddress());
					try {
						sendMessage(mes);
					} catch (InvisibleContainerException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
				}
			}
		}
	}
	
  /**
   * This method preprocesses the incoming configuration from the simulation server.
   * @param ed
   */
	public void handleConfiguration(EntityDescriptor ed) {
		consumption = Double.parseDouble(ed.confParamsString.get(AcrossDataConstants.VEHICLE_CONSUMPTION));
		velocity = Double.parseDouble(ed.confParamsString.get(AcrossDataConstants.VEHICLE_VELOCITY));
		effectSpeed = Integer.parseInt(ed.confParamsString.get(AcrossDataConstants.VEHICLE_EFFECT_SPEED));
		myEffect = (EffectOnSimulationObject) ed.confObjects.get(AcrossDataConstants.VEHICLE_EFFECT);
		
		myMovement.handleConfiguration(ed, velocity, consumption);
		mySensors.handleConfiguration(ed);
		
        // display visio information
		displayNameInVisio();
	}

	/* --------------------- VISIO -----------------------*/
	/** Displays agent's name in visio. */
	protected void displayNameInVisio() {
        String visioInfo = "<b>" + this.getName();
        gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_INFO, visioInfo);		
	}
	
	/** Displays agent's name and current activity in visio. */
	protected void displayActivityInVisio(AcrossActivity aa) {
		if(aa==null) {
			displayNameInVisio();
		} else {
			String visioInfo = "<b>" + this.getName()+" <c00FF00>"+aa;
			gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_INFO, visioInfo);
		}
	}
	
	/** Displays an action in visio. */
	protected void displayVisioAction(byte action) {
		gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_VISIO_ACTION, ""+action, getAddress().toString());
	}

	/* --------------------- DIRECTORY LISTENER ----------------- */
	public void handleDeregister(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}

	public void handleInvisible(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}

	public void handleNewRegister(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}
	
	/** The agents that were subscribed become visible. */
	public void handleVisible(String containerName, DirectoryRecord[] records, String matchingFilter) {
		if(matchingFilter.equals(EmergencyAgent.TYPE)) {
			for (int i = 0; i < records.length; i++) {
				DirectoryRecord record = records[i];
				emergencyAgents.add(record.address);
			}
		}
	}

	/* ---------------------------- SIMULATION CONTROL ---------------------------- */
	/** Removes specified entity from the simulation. */
	public void destroyEntity(String contName) {
		displayVisioAction(VisioConnectionAgent.ACTION_STANDIN_KILLED);
		gisShell.submitTopicToServer(AcrossTopicConstants.TOPIC_AGENT, contName, AcrossControlConstants.REMOVE_AGENT);
	}
	
}
