package across.agents.emergency.repair;

import across.agents.emergency.centre.AcrossActivity;
import across.data.simulation.EffectOnSimulationObject;
import across.data.simulation.SimulationObjectDescription;
import across.simulation.constants.AcrossControlConstants;
import across.simulation.constants.AcrossTopicConstants;
import across.visio.oldvisio.VisioConnectionAgent;
import aglobe.ontology.Message;
import aglobex.simulation.utils.timer.SimulationTimeoutTask;

/**
 * Class in charge of handling activities of a repair vehicle
 * agent.
 * @author Eduard Semsch
 *
 */
public class RepairVehicleActivityHandler implements RepairVehicleActivities {

	/** Says whether some activity is running currently. */
	private boolean running = false;
	
	/** The owner agent using this class. */
	private RepairVehicleAgent owner;
	
	/** Activity the agent is currently handling. */
	private AcrossActivity currentActivity;
	
	/** End condition of the current activity. */
	private ActivityEndCondition currentActivityEndCondition;
	
	/** Task to report the outcome of the activity. */
	private RepairRequestParticipantTask reportTask;
	
	public RepairVehicleActivityHandler(RepairVehicleAgent owner) {
		this.owner = owner;
	}

	/* --------------------- INTERFACE ------------------- */
	/**
	 * This method carries out the specified activity and reports the outcome
	 * using the specified task.
	 * @param act
	 * @param rpt
	 */
	public void handleActivity(AcrossActivity act, RepairRequestParticipantTask rpt) {
		this.currentActivity = act;
		this.reportTask = rpt;
		running = true;
		startActivity();
	}
	
	// TODO - implement failure
	/**
	 * This method checks for the fulfillment of the end condition of 
	 * the current activity.  If the activity is done or failed it
	 * reports it back through the specified report task.
	 *
	 */
	public void run() {
		if(running) {
			if(currentActivityEndCondition.isDone()) {
				System.out.println(owner.getName()+"- Activity done: "+currentActivity.getActivityType());
				currentActivityEndCondition.finishActivity();
				reportTask.sendInformResult(new Boolean(!currentActivityEndCondition.isFailed()));
				running = false;
			}
		}
	}
	
	/* -------------------------- ACTIVITY EXECUTION ------------------- */
	/**
	 * This function starts the activity handling and sets up the endconditions
	 * for the activity.
	 *
	 */
	private void startActivity() {
		String activityType = currentActivity.getActivityType();
		if(activityType.equalsIgnoreCase(ACTIVITY_GOTO)) {
			System.out.println("Starting activity: "+activityType+" to "+currentActivity.getParams().get("TO")+ " for object: "+currentActivity.getParams().get("OBJECT"));			
			handleGOTOActivity();
		} else if (activityType.equalsIgnoreCase(ACTIVITY_PICKUP)) {
			System.out.println("Starting activity: "+activityType);
			handlePickupActivity();
		} else if (activityType.equalsIgnoreCase(ACTIVITY_DROP)) {
			System.out.println("Starting activity: "+activityType);
			handleDropActivity();
		} else if (activityType.equalsIgnoreCase(ACTIVITY_EFFECT)) {
			System.out.println("Starting activity: "+activityType);
			handleEffectActivity();
		} else {
			owner.logSevere("Unknown activity specified");
		}
	}
	
	/** This function handles the drop activity - the activity tells the vehicle to drop something. */
	private void handleDropActivity() {
		currentActivityEndCondition = new ActivityEndCondition() {

			public void finishActivity() {
				owner.displayVisioAction(VisioConnectionAgent.ACTION_LOAD);
			}

			public boolean isDone() {
				// TODO Auto-generated method stub
				return true;
			}

			public boolean isFailed() {
				// TODO Auto-generated method stub
				return false;
			}
			
		};
	}

	/** This function handles the pickup activity - the activity tells the vehicle to pick up something. */
	private void handlePickupActivity() {
		currentActivityEndCondition = new ActivityEndCondition() {

			public void finishActivity() {
				owner.displayVisioAction(VisioConnectionAgent.ACTION_UNLOAD);
			}

			public boolean isDone() {
				// TODO Auto-generated method stub
				return true;
			}

			public boolean isFailed() {
				// TODO Auto-generated method stub
				return false;
			}
			
		};
	}

	/**
	 * This function handles the effect activity - this activity is for using this vehicle effect
	 * on the simulation object.
	 *
	 */
	private void handleEffectActivity() {
		final String where = (String) currentActivity.getParams().get("WHERE");
		if(where!=null) {
			owner.myEffect = EffectOnSimulationObject.getEffectForObjectAndDefaultEffect(where, owner.myEffect);
			
			// start the effect - periodically use the effect on object with the cadency owner.effectSpeed
			final SimulationTimeoutTask stt = new SimulationTimeoutTask(owner,owner.effectSpeed) {

				@Override
				protected void timeout() {
					if(owner.mySensors.isObjectInSight(where)) {
						owner.gisShell.submitTopicToServer(AcrossTopicConstants.TOPIC_SIMULATION_OBJECT,owner.myEffect,AcrossControlConstants.CHANGE_SIMULATION_OBJECT);
					}
					rescheduleTimer(owner.effectSpeed);
				}

				/** Not implemented. */
				@Override
				protected void handleIncomingMessage(Message m) {}
				
			};
			
			// end condition - if the effect is fully applied the activity ends.
			currentActivityEndCondition = new ActivityEndCondition() {

				public void finishActivity() {
					stt.cancelTask();
				}

				/** Checks whether the effect has been fully applied. */
				public boolean isDone() {
					// if the object is no more in sight (it is removed)
					if(!owner.mySensors.isObjectInSight(where)) {
						return true;
					}
					SimulationObjectDescription sod = owner.mySensors.getObjectsDescription(where);
					// the description is not availiable but the object still exists
					if(sod==null) {
						return false;
					}
					// was my effect fully applied?
					if(owner.myEffect.constructionDamageDiff<0) {
						if(sod.getConstructionDamage()>0) {
							return false;
						}
					}
					if(owner.myEffect.fireDiff<0) {
						if(sod.getExtentOfFire()>0) {
							return false;
						}
					}
					if(owner.myEffect.insurgencyDiff<0) {
						if(sod.getInsurgencyLevel()>0) {
							return false;
						}
					}
					if(owner.myEffect.woundedDiff<0) {
						if(sod.getNumberOfWounded()>0) {
							return false;
						} 
					}
					return true;
				}
				
				// TODO - unrealistic presumption - we suppose this activity never fails.
				public boolean isFailed() {
					return false;
				}
				
			};
		} else {
			owner.logSevere("Target object not specified");
		}
	}

	/**
	 * This activity means that the driver should go somewhere.
	 *
	 */
	private void handleGOTOActivity() {
		final String to = (String) currentActivity.getParams().get("TO");
		String from = null;
		if(currentActivity.getParams().containsKey("FROM")) {
			from = (String) currentActivity.getParams().get("FROM");
		}
		final String objectId = (String) currentActivity.getParams().get("OBJECT"); 
		currentActivityEndCondition = new ActivityEndCondition() {

			/**
			 * The activity is done when the actualLocation is the same as the specified 
			 * TO location or the vehicle reached the specified object.
			 */
			public boolean isDone() {
				if(objectId!=null) {
					// either at the object or at the end destination
					return (owner.mySensors.isObjectInSight(objectId) || (!owner.myMovement.isOnRoute() && owner.myMovement.actualLoc.equals(to)));
				}
				return owner.myMovement.actualLoc.equals(to);						
			}
			
			// TODO - unrealistic presumption!
			/** We presume that this activity never fails. */
			public boolean isFailed() {
				return false;
			}
			
			/** Stop the vehicle once it reached the specified location. */
			public void finishActivity() {
				owner.myMovement.stopMovement();
			}
			
		};
		if(from==null) {
			owner.myMovement.goToLocation(to);
		} else {
			owner.myMovement.driveThroughRoad(from, to);
		}
	}
	
	/**
	 * Interface for specifying an end condition for an activity. 
	 * @author Eduard Semsch
	 *
	 */
	public interface ActivityEndCondition {
		
		/** Says whether the activity is done or not. */
		public boolean isDone();
		
		/** Says whether the activity has been done successfully or not. */
		public boolean isFailed();
		
		/** Call this method once the activity is finished. */
		public void finishActivity();
	}
	
}
