package across.agents.emergency.centre;

import iglobe.plan.RescuePlan;
import iglobe.plan.RescueTask;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;

import across.agents.emergency.centre.AcrossActivity.ActivityState;
import across.data.RequestList;
import across.simulation.constants.AcrossMessageConstants;
import aglobe.container.transport.Address;
import aglobe.ontology.Message;
import aglobex.protocol.request.RequestInitiatorTask;
import aglobex.simulation.protocol.cnp.CNPInitiatorTask;

/**
 * <p>
 * This class is in charge of handling one Rescue/Accident/Disaster event
 * in Across simulation scenario.  The class handles the whole cycle of
 * planning, scheduling and execution of the emergency task.  The transaction
 * has to be periodically woken through its run method to be able to proceed.
 * </p>
 * <p>
 * This class is supposed to be instantiated in the EmergencyAgent.
 * </p>
 * <p>
 * Presumptions made are these:
 * <ol>
 * <li> All the resources are fully scheduled before the execution starts. </li>
 * <li> All the resources are fully commited to performing this transaction - this
 * means that e.g. the drivers are performing only this transaction and if they are
 * not they will always find someone else that will. </li>
 * <li> The actors performing each action are specified in the plan obtained from the
 * planner - this means that the planner puts id of the actor to every activity. </li>  
 * </ol>
 * </p>
 * <p>
 * The whole transaction takes these steps:
 * <ol>
 * 	<li> Ask the planner for plan
 * 	<li> Try to schedule the plan 
 * 	<li> If plan scheduled ok, go to 4., if not
 * go back to 1. and pass the planner the constraints as obtained from scheduling (the 
 * unavailiable resources).
 * 	<li> Start execution of the ready tasks.
 * 	<li> If all tasks done, finish the transaction.
 * </ol> 
 * </p>
 * @author Eduard Semsch
 *
 */
public class EventHandlingTransaction {

	// TRANSACTION GLOBAL
	/** Knowledge of the scheduler (this one is shared among the all the transactions). */
	protected SchedulingKnowledge schedulingKnowledge;
	
	/** Owner agent - the EmergencyAgent. */
	protected EmergencyAgent owner;
	
	/** State of the transaction - see {@link TransactionState}. */
	protected TransactionState state;
	
	// TASK DEFINITION
	/** Event that is being handled by this transaction. */ 
	protected RescueTask rescueTask;
	
	// PLAN
	/** Plan as received from the planning agent. */
	protected RescuePlan ixPlan;
	
	/** Plan schedulable in Across scenario. */
	protected AcrossPlan acrossPlan;
	
	// SCHEDULE
	/** Drivers performing this transaction. These drivers are fully commited to this transaction*/
	protected Map<String,Address> scheduledDrivers = new HashMap<String,Address>();
	
	/** List of drivers IDs (plan IDs) that could not have been scheduled. */ 
	protected List<String> driversFailedToSchedule = new ArrayList<String>();
	
	/** Material in this transaction. */
	protected Map<String,Address> scheduledMaterial = new HashMap<String, Address>();
	
	/** Location of scheduled material. */
	protected Map<String,String> scheduledMaterialLocation = new HashMap<String, String>();

	/** List of material IDs (plan IDs) that could not have been scheduled. */
	protected List<String> materialFailedToSchedule = new ArrayList<String>();
	
	// EXECUTION
	/** Map activity <-> state - execution monitor. For description of activity states see {@link AcrossActivity.ActivityState}*/
	protected Map<AcrossActivity,ActivityState> activityStates = new HashMap<AcrossActivity, ActivityState>();
	
	/**
	 * Constructor of the Transaction - it receives all the necessary knowledge.  The transaction
	 * is suited to be the only event handler for a certain event - if there were two or more 
	 * transactions handling the same event they may show unpredictable behaviour.
	 * @param event - the handled event (disaster/accident)
	 * @param owner - the owner EmergencyAgent
	 * @param schedulingKnowledge - the knowledge needed for scheduling shared among the transactions
	 */
	public EventHandlingTransaction(RescueTask event, EmergencyAgent owner, SchedulingKnowledge schedulingKnowledge) {
		this.rescueTask = event;
		this.owner = owner;
		this.schedulingKnowledge = schedulingKnowledge;
		state = TransactionState.NOT_PLANNED;
	}

	
	/* ---------------------------- INTERFACE ------------------------- */	
	
	/**
	 * This method performs all currently availiable actions in this transaction.
	 * The method should be run periodically.
	 *
	 */
	public void run() {
		switch(state) {
			// Not yet planned (either the transaction has been run just now, or the scheduling failed
			// and we are asking for a new plan with constraints.
			case NOT_PLANNED: {
				plan(rescueTask,driversFailedToSchedule,materialFailedToSchedule);
				break;
			}
			// Waiting for the planning to be done (waiting for asynchronous messages)
			case PLANNING: {
				return;
			}
			// We have a plan and we want to schedule it.  This starts the scheduling
			case NOT_SCHEDULED: {
				acrossPlan = schedulingKnowledge.ixToAcrossPlanParser.ixToAcrossPlan(ixPlan);
				schedulePlan(acrossPlan);
				break;
			}
			// Waiting for the scheduling to be done (waiting for asynchronous messages)
			case SCHEDULING: {
				// check whether the scheduling is done.
				if(checkSchedulingDone()) {
					// if successfully scheduled then execute
					if(checkSchedulingSuccess()) {
						state = TransactionState.NOT_EXECUTED;
					}
					// if not successfully scheduled then free the resources and replan with constraints.
					else {
						flushScheduled();
						state = TransactionState.NOT_PLANNED;
					}
				}
				break;
			}
			// We have a schedule, now we want the schedule to be executed.
			case NOT_EXECUTED: {
				prepareExecution();
				break;
			}
			// Executing the schedule - always tries to execute the tasks that
			// are ready. (waiting for asynchronous messages)
			case EXECUTING: {
				executeReady();
				// if all tasks done - the transaction is over.
				if(checkExecutingDone()) {
					// release resources 
					flushScheduled();	
					// set state to done
					state = TransactionState.DONE;
				}
				break;
			}
			// Transaction is over - it has been a success.
			case DONE: {
				break;
			}
			// Transaction is over - it failed.
			case FAILED: {
				break;
			}
			default: {
				owner.logSevere("Unknown state of transaction: "+state);
			}
		}
	}
	
	/**
	 * This method says whether this transaction has or has not been finished yet.
	 * @return
	 */
	public boolean finished() {
		return state==TransactionState.DONE || state==TransactionState.FAILED;
	}
	
	/**
	 * Says whether the operation failed or not.  This method is only reasonable to
	 * call once the transaction is finished.
	 * @return
	 */
	public boolean failed() {
		return state==TransactionState.FAILED;
	}
	
	/**
	 * Restarts the transaction.
	 *
	 */
	public void restart() {
		// set state to not planned
		state = TransactionState.NOT_PLANNED;
		// release resources
		flushScheduled();
		// clear the constraint lists
		materialFailedToSchedule.clear();
		driversFailedToSchedule.clear();
	}
	
	/* ------------------- PLANNING ------------------ */ 
	/**
	 * Ask the planning agent for a plan - there are two components of the task specification:
	 * <li> The task definition. </li>
	 * <li> The constraints definition - the constraints are the activities that couldn't be
	 * scheduled. </li>
	 * @param rt - the task definition
	 * @param driverNAConstraints - unavailiability constraints on drivers
	 * @param materialNAConstraints - unavailiability constraints on material
	 */
	// TODO - implement the communication with the constraints.
	private void plan(RescueTask rt,List<String> driverNAConstraints, List<String> materialNAConstraints) {
		// TODO - cancels the handler after unsuccessfull scheduling. REMOVE!!!
		rescueTask.getUnavailableResources().addAll(driverNAConstraints);
		state = TransactionState.PLANNING;
		new RequestInitiatorTask(owner,schedulingKnowledge.plannerAddress,rt) {

			/** Not used. */
			@Override
			protected void informDone() {
				
			}

			/** 
			 * Plan has been received. If there is no plan the transaction fails. 
			 */
			@Override
			protected void informResult(final Object result) {
				EventHandlingTransaction.this.owner.addEvent(new Runnable() {
					public void run() {
						// if no plan received, fail the transaction
						if(result==null) {
							state = TransactionState.FAILED;
						} 
						// take the plan and move on to scheduling 
						else {
							ixPlan = (RescuePlan) result;
							state = TransactionState.NOT_SCHEDULED;
						}	
					}
				});
			}
			
		};
	}
	
	/* ------------------- SCHEDULING ------------------ */
	/**
	 * This method tries to schedule the plan.  It spawns a CNP for every necessary
	 * driver and material in the plan.
	 */
	private void schedulePlan(AcrossPlan ap) {
		state = TransactionState.SCHEDULING;
		// set all the drivers and material unscheduled and the failedToSchedule lists empty.
		for (String driverId : ap.drivers.keySet()) {
			scheduledDrivers.put(driverId, null);
		}
		driversFailedToSchedule.clear();
		for (String materialId : ap.material.keySet()) {
			scheduledMaterial.put(materialId, null);
		}
		materialFailedToSchedule.clear();
		// call CNP for drivers
		cnpForDrivers(ap);
		// call CNP for material
		cnpForMaterial(ap);
	}

	// TODO - not used yet!
	/**
	 * This method starts the CNP for materials needed in this transaction.
	 * @param ap
	 */
	private void cnpForMaterial(AcrossPlan ap) {
		for (final String materialId : ap.material.keySet()) {
			// TODO - fill in request list
			new CNPInitiatorTask(owner,schedulingKnowledge.storageAddresses,new RequestList(),1000,AcrossMessageConstants.SCHEDULE_EMERGENCY_MATERIAL) {
				
				/** Not used. */
				@Override
				protected boolean evaluateAcceptReply(final Message m) {
					return false;
				}

				/** Not implemented */
				@Override
				protected void evaluateAcceptTimeout() {}

				/**
				 * Picks one location with the material and sends it the ok reply.
				 * @return
				 */
				@Override
				protected List<Address> evaluateReplies() {
					List<Address> oneMaterial = new LinkedList<Address>();
					String location = null;
					if(receivedOffers.size()>0) {
						Random r = new Random();
						int ind = r.nextInt(receivedOffers.size());
						List<Address> adrList = new ArrayList<Address>(receivedOffers.keySet());
						Address address = adrList.get(ind);
						Message mes = receivedOffers.get(address);
						oneMaterial.add(address);
						location = (String) mes.getContent();
					}
					if(oneMaterial.size()==0) {
						materialFailedToSchedule.add(materialId);
					} else {
						scheduledMaterial.put(materialId, oneMaterial.get(0));
						scheduledMaterialLocation.put(materialId, location);
					}
					return oneMaterial;
				}
				
			};
		}
	}

	/**
	 * This method starts the CNP for drivers needed in this transaction.
	 * @param ap
	 */
	private void cnpForDrivers(AcrossPlan ap) {
		// TODO - take into consideration the types of the agents.
		// For each necessary driver spawn a CNP.
		for (final String driverId : ap.drivers.keySet()) {
			//TODO - debug
			List<Address> resAddr = new ArrayList<Address>();
			for (Address address : schedulingKnowledge.driversAddresses) {
				if(address.getContainerName().indexOf(driverId)!=-1) {
					resAddr.add(address);
				}
			}
			if(resAddr.size()==0) {
				driversFailedToSchedule.add(driverId);
				continue;
			}
			new CNPInitiatorTask(owner,resAddr,rescueTask.getLocation(),10000,AcrossMessageConstants.SCHEDULE_EMERGENCY_DRIVER) {

				/**
				 * Picks the ready driver that is closest to the accident location.  This method is invoked after all the 
				 * proposals from the drivers have been collected.  
				 */
				@Override
				protected List<Address> evaluateReplies() {
					// return value
					List<Address> driverList = new LinkedList<Address>();
					// the most suitable driver
					Address oneDriver = null;
					// This picks the driver that is closest to the location.
					if(receivedOffers.size()>0) {
						double lowestTime = Double.MAX_VALUE;
						for (Address address : receivedOffers.keySet()) {
							Message mes = receivedOffers.get(address);
							Double timeToPlace = (Double) mes.getContent();
							// driver is on mission
							if(timeToPlace==null) {
								continue;
							} else if (timeToPlace<lowestTime) {
								lowestTime = timeToPlace;
								oneDriver = mes.getSender();
							}
						}
						// if there is no positive answer mark this driver failed to schedule
						if(oneDriver==null) {
							driversFailedToSchedule.add(driverId);
						} 
						// else add the driver to the list of accepted drivers 
						else {
							driverList.add(oneDriver);
							// from now on we can send the orders to the driver.
							scheduledDrivers.put(driverId, oneDriver);
						}
					}
					return driverList;
				}
				
				/**
				 * Add the driver to the scheduled drivers and cancel the task.
				 */				
				@Override
				protected boolean evaluateAcceptReply(final Message m) {
					// cancel task.
					return false;
				}

				/** Not implemented. */
				@Override
				protected void evaluateAcceptTimeout() {}
				
			};
		}
	}

	/**
	 * Call this method when you need to release all your resources.  This can
	 * happen when the scheduling fails or when the transaction is over.
	 *
	 */
	// TODO - unclean implementation for materials - there can be more 
	// than one material scheduled per location - this would not
	// say which one to cancel.
	private void flushScheduled() {
		// flush the possibly previously scheduled drivers
		if(scheduledDrivers.size()>0) {
			for (Address address : scheduledDrivers.values()) {
				if(address!=null) {
					// this request task releases the driver.
					new RequestInitiatorTask(owner,address,AcrossMessageConstants.CANCEL_DRIVER_SCHEDULE) {
						
						/** Not used. */
						@Override
						protected void informDone() {}
	
						/** Not used. */
						@Override
						protected void informResult(Object result) {}
						
					};
				}
			}
		}
		scheduledDrivers.clear();
		// flush the possibly previously scheduled material		
		scheduledMaterial.clear();
	}
	
	/**
	 * This method checks whether the scheduling is over meaning that
	 * the resources are either scheduled or marked unavailiable.
	 * @return
	 */
	private boolean checkSchedulingDone() {
		for (String driverId : scheduledDrivers.keySet()) {
			// if the driver is neither scheduled nor marked unavailiable the scheduling is not done yet.
			if(scheduledDrivers.get(driverId)==null && !driversFailedToSchedule.contains(driverId)) {
				return false;
			}
		}
		for (String materialId : scheduledMaterial.keySet()) {
			// if the material is neither scheduled nor marked unavailiable the scheduling is not done yet.			
			if(scheduledMaterial.get(materialId)==null && !materialFailedToSchedule.contains(materialId)) {
				return false;
			}
		}
		return true;
	}
	
	/**
	 * Tells whether the scheduling was a success or not.  This method returns true value only
	 * if the scheduling is already done!!!
	 * @return
	 */
	private boolean checkSchedulingSuccess() {
		return driversFailedToSchedule.size()==0 && materialFailedToSchedule.size()==0;
	}

	/* ---------------------- EXECUTION --------------------- */
	
	/**
	 * Starts executing the schedule.  It marks all the activities in the plan
	 * waiting for execution.
	 */
	// TODO - do all the preliminary actions here
	private void prepareExecution() {
		for (AcrossActivity aa : acrossPlan.activitiesHandlers.keySet()) {
			activityStates.put(aa, ActivityState.WAITING);
			// replace the wildcards
			for (String storage : scheduledMaterial.keySet()) {
				aa.replaceStringInParams(storage, scheduledMaterial.get(storage).getContainerName());
				aa.replaceStringInParams(storage+"LOCATION", scheduledMaterialLocation.get(storage));
			}
		}
		state = TransactionState.EXECUTING;
	}
	
	// TODO - this checks only for successfull plan execution.  Check for failure as well.
	/**
	 * This method executes (orders the actors to execute) all the tasks that are ready and
	 * waiting.  This method also checks whether all the activities have been done and if 
	 * true it releases the resources.
	 */
	private void executeReady() {
		// for all waiting activities check precedences and if they are ok
		// send the request to the driver to perform the activity.
		for (final AcrossActivity act : activityStates.keySet()) {
			// only the waiting activities are checked
			if(activityStates.get(act)==ActivityState.WAITING) {
				//check precedences
				boolean precOk = true;
				List<AcrossActivity> precActList = acrossPlan.precedences.get(act);
				if(!(precActList==null || precActList.size()==0)) {
					for (AcrossActivity precAct : precActList) {
						if(activityStates.get(precAct)!=ActivityState.DONE) {
							precOk = false;
							break;
						}
					}
				}
				// if precedences ok run the task
				if(precOk) {
					// mark the activity that is being executed
					activityStates.put(act,	ActivityState.EXECUTING);
					new RequestInitiatorTask(owner,scheduledDrivers.get(acrossPlan.activitiesHandlers.get(act)),act) {
						
						/** Not implemented. */
						@Override
						protected void informDone() {}

						// XXX - specify more informative result.
						/**
						 * Result is either true or false. True = done successfully, false = failed.
						 * @param result
						 */
						@Override
						protected void informResult(Object result) {
							final Boolean done = (Boolean) result;
							if(done) {
								System.out.println("Acitivity done: "+act.activityType+" by "+scheduledDrivers.get(acrossPlan.activitiesHandlers.get(act)).getName());
								activityStates.put(act, ActivityState.DONE);
							} else {
								activityStates.put(act, ActivityState.FAILED);
							}
						}
						
					};
				}
			}
		}
	}
	
	/**
	 * This method checks whether all the acvitities have been carried out.
	 * @return
	 */
	protected boolean checkExecutingDone() {
		boolean done = true;
		for (ActivityState act : activityStates.values()) {
			// if any activity is not done or failed the execution is not over yet.
			if(act!=ActivityState.DONE && act!=ActivityState.FAILED) {
				done = false;
				break;
			}
		}
		return done;
	}
	
	/**
	 * These are the states of the transaction.  
	 * @author Eduard Semsch
	 *
	 */
	protected static enum TransactionState {
		/** 
		 * Transaction has not been planned yet. Or it has not been scheduled fine and
		 * it needs to be replanned with constraints.
		 */
		NOT_PLANNED,
		/** 
		 * The problem was sent to the planner to produce a plan. Now the transaction
		 * waits for the planner to send the plan (waits for asynchronous message).
		 */
		PLANNING,
		/** 
		 * The problem has not been scheduled yet. The plan has been obtained and now
		 * it needs to be scheduled.
		 */
		NOT_SCHEDULED,
		/** 
		 * We are scheduling the resources - this means to wait for the asynchrounous
		 * messages from the actors and material providers.
		 */
		SCHEDULING,
		/** The plan has been scheduled, now start executing the schedule. */
		NOT_EXECUTED,
		/** 
		 * The execution phase - we are sending orders to the actors and collecting
		 * their reports (asynchrounous messages).
		 */
		EXECUTING,
		/** The transaction has been done succesfully. */
		DONE,
		/** The transaction failed. */
		FAILED;
	}
}
