package across.agents.driver;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import across.data.AcrossMapArcs;
import across.data.AcrossMapNodes;
import across.data.Arc;
import across.data.Node;
import across.simulation.constants.AcrossControlConstants;
import across.simulation.constants.AcrossDataConstants;
import aglobe.visio3D.PlanUpdate;
import aglobex.simulation.entitysimulator.ControlLink;
import aglobex.simulation.entitysimulator.EntityBehaviour;
import aglobex.simulation.entitysimulator.EntityState;
import aglobex.simulation.global.EntityRecord;
import aglobex.simulation.ontology.WorldDimension;
import aglobex.vecmath.Point3d;

/**
 * Behaviour of the ACROSS driver.  This driver drives along the network of nodes and arcs. Whenever
 * it reaches new node it notifies its controller agent.
 * @author Eduard Semsch
 *
 */
public class DriverBehaviour implements EntityBehaviour {

	/** Lengths of the arcs - if no arc length = -1. */
	private double[][] arcLengths;
	
	/** Node this is the mapping of node names to their respective numbers. */
	private HashMap<String,Integer> nodeNamesToNumbers;
	
	/** Nodes of the map. */
	private List<Node> mapNodes;
	
	/** Either the node where this driver is waiting (if moving = false) of the starting
	 * node of the movement (if moving = true) */
	private String startNode;
	
	/**  If moving = true this node is the node this driver is heading towards. */
	private String endNode;
	
	/** Tells whether this driver is moving. */
	private boolean moving = false;
	
	/** This behaviour's controller (the DriverAgent). */
	private ControlLink myControl;
	
	/** My entity's EntityRecord. */
	private EntityRecord entityRec;
	
	/** Dimension of the world. */
	private WorldDimension worldDim;
	
	/** Vector of current movement. */
	private double[] movementVector;
	
	/** Current position on the map. */
	private double[] position;
	
	/** Velocity of the vehicle. */
	private double velocity;
	
	/** How far is the end node from our current position.  */
	private double lengthToGo = 0;
	
	/** This entity's state. */
	private EntityState es = new EntityState();
	
	/** Is the behaviour active. */
	private boolean active = true;
	
	// TODO - conf
	/** Length of simulation cycle */
	private double simulationCyclePeriod = 100;
	
	/**
	 * This is the controlling function used to steer this behaviour by its controller. The content
	 * has to be a String - representing a node on the map.  If everything is OK there
	 * is no control response, otherwise it sends response according to across.simulation.AcrossControlConstants.
	 * 
	 */
	public void control(Object content, String reason) {
		if(reason==null) {
			myControl.getClass();
		}
		if(content instanceof String && reason.equalsIgnoreCase(AcrossControlConstants.MOVE_TO)) {
			endNode = (String) content;
			if(!moving) {
				double [] endNodePosition = getPositionFromNode(endNode);
				movementVector = new double[] {endNodePosition[0]-position[0],endNodePosition[1]-position[1]};
				double length = Math.sqrt(Math.pow(movementVector[0], 2)+Math.pow(movementVector[1], 2));
				if(length>0 || arcLengths[nodeNamesToNumbers.get(startNode)][nodeNamesToNumbers.get(endNode)]==-1) {
					moving = true;
					lengthToGo = length;
				} else {
					myControl.sendControlResponse(null, AcrossControlConstants.RESPONSE_BAD_NODE);
				}
			} 
		} else if(reason.equalsIgnoreCase(AcrossControlConstants.STOP)) {
			moving = false;
		} else {
			myControl.sendControlResponse(null, AcrossControlConstants.RESPONSE_FAILURE);
		} 
	}
	
	/**
	 * Gets node's coordinates.
	 * @param node
	 * @return
	 */
	private double[] getPositionFromNode(String node) {
		int ind = nodeNamesToNumbers.get(node);
		Node n = mapNodes.get(ind);
		return new double[] {n.getX(),n.getY()};
	}
	
	public void dispose() {
		active = false;
	}

	public void entityCrashed(EntityBehaviour otherObject, long fromTime) {
		myControl.destroyEntity();
	}

	public PlanUpdate getPlanUpdate(boolean removeOld) {
		// TODO Auto-generated method stub
		return null;
	}

	/**
	 * Initializes the entity behaviour in these steps:
	 * <li> Load the map configurations from the file. </li>
	 * <li> Transform the configurations into usable format. </li>
	 * <li> Initialize other inner variables. </li>
	 * <li> Find out vehicle speed and startNode. </li>
	 * <li> Calculate position. </li>
	 */
	public void init(ControlLink controlLink, EntityRecord entityRecord, WorldDimension worldDimension) throws Exception {
		AcrossMapNodes amn = (AcrossMapNodes) entityRecord.entityDescriptor.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_NODES);
		AcrossMapArcs ama = (AcrossMapArcs) entityRecord.entityDescriptor.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_ARCS);
		mapNodes = amn.getNode();
		nodeNamesToNumbers = getNodeNamesToNumbersMap(mapNodes);
		arcLengths = getConnectedNodes(ama.getArc());
		myControl = controlLink;
		entityRec = entityRecord;
		worldDim = worldDimension;
		velocity = Double.parseDouble(entityRecord.entityDescriptor.confParamsString.get(AcrossDataConstants.VEHICLE_VELOCITY));
		startNode = entityRecord.entityDescriptor.confParamsString.get(AcrossDataConstants.ENTITY_START_NODE);
		position = getPositionFromNode(startNode);
		es.position = new Point3d(position[0],position[1],0);
	}
	
	/**
	 * This function returns the mapping from node names to their numbers.
	 * @param amn
	 * @return
	 */
	private HashMap<String,Integer> getNodeNamesToNumbersMap(List<Node> nodes) {
		HashMap<String,Integer> namesToNums = new HashMap<String, Integer>();
		int counter = 0;
		for (Node node : nodes) {
			String name = node.getName();
			if(!namesToNums.containsKey(name)) {
				namesToNums.put(name,counter);
				counter++;
			}
		}
		return namesToNums;
	}
	
	/**
	 * Returns a square array of boolean saying which node is interconnected with which other node.
	 * This function assumes that we are having a unoriented graph.
	 * @param arcs
	 * @return
	 */
	private double[][] getConnectedNodes(List<Arc> arcs) {
		int size = nodeNamesToNumbers.size();
		double [][] retVal = new double[size][size];
		for (double[] bs : retVal) {
			Arrays.fill(bs, -1);
		}
		for (Arc arc : arcs) {
			int x = nodeNamesToNumbers.get(arc.getFrom());
			int y = nodeNamesToNumbers.get(arc.getTo());
			double length = arc.getLength();
			retVal[x][y] = length;
			retVal[y][x] = length;
		}
		return retVal;
	}

	/**
	 * Updates this entity coordinates depending on the time.
	 */
	public EntityState updateEntityState(long timeStep, long currentSimulationTimestamp) {
		if(moving && active) {
			double diff = timeStep;
			double diffLength = diff*velocity/simulationCyclePeriod;
			if(diffLength >= lengthToGo) {
				moving = false;
				startNode = endNode;
				myControl.sendControlResponse(endNode, AcrossControlConstants.RESPONSE_STOPPED);
				diffLength = lengthToGo;
				double[] endNodePos = getPositionFromNode(endNode);
				es.position = new Point3d(endNodePos[0],endNodePos[1],0);
			} else {
				lengthToGo -= diffLength;
				double angle = Math.atan2(movementVector[1], movementVector[0]);
				position[0] = position[0]+Math.cos(angle)*diffLength;
				position[1] = position[1]+Math.sin(angle)*diffLength;
				es.position = new Point3d(position[0],position[1],0);
				if(entityRec.entityDescriptor.typeDescriptor.typeName.equals("TERRORIST_DRIVER_ENTITY_TYPE")) {
					myControl.sendControlResponse(position, AcrossControlConstants.RESPONSE_STOPPED);
				}
			}
		}
		return es;
	}

}
