package across.agents.driver;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import across.agents.driver.data.RoadInfo;
import across.agents.driver.data.RouteInfo;
import across.agents.driver.gui.DriverAgentGUI;
import across.agents.driver.gui.DriverGUIInterface;
import across.agents.driver.util.UpdateTransporter;
import across.agents.emergency.centre.EmergencyAgent;
import across.data.Batch;
import across.data.Page;
import across.data.Param;
import across.data.Proposal;
import across.data.PublicParams;
import across.data.RequestList;
import across.data.TransportOrder;
import across.data.Waypoint;
import across.data.simulation.SimulationObjectDescription;
import across.data.simulation.SimulationObjectLocation;
import across.simulation.constants.AcrossDataConstants;
import across.visio.oldvisio.VisioConnectionAgent;
import aglobe.container.agent.Agent;
import aglobe.container.sysservice.directory.DirectoryException;
import aglobe.container.sysservice.directory.DirectoryListener;
import aglobe.container.sysservice.directory.DirectoryRecord;
import aglobe.container.sysservice.directory.DirectoryService;
import aglobe.container.transport.Address;
import aglobe.container.transport.InvisibleContainerException;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobe.service.gis.client.GISClientService;
import aglobe.service.gis.client.GISTopicListener;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.entity.EntityDescriptor;
import aglobex.simulation.sensor.UniversalSensorAgent;

/**
 * Agent representing a common transporter driver carrying things like sheep and bread.
 * @author Eduard Semsch
 *
 */
public class DriverAgent extends Agent implements GISTopicListener, DirectoryListener, DriverGUIInterface, ClientServerTopicConstants {
	
  /** For visio purposes - identifies the owner agent */
  public static String MY_TRANSPORTER = "MY-TRANSPORTER";

  /** Agent type. */
  public static final String TYPE = "Driver";

  /** This class represents internal agent's data */
  protected DriverAgentSimulationData mySimData = new DriverAgentSimulationData();
  
  /** This class takes care of movement of the driver */
  protected DriverMovement myMovement = new DriverMovement(this);
  
  /** This class takes care of sensoric data. */
  protected DriverSensors mySensors = new DriverSensors(this);
  
  /** GIS shell. */
  protected GISClientService.Shell gisShell;
  
  /** DirectoryService shell. */
  private DirectoryService.Shell dsShell;
  
  /** Driver's gui. */
  public DriverAgentGUI gui;
 
  /** Address of my transporter agent. */
  protected Address transporter = null;
  
  /** Address of the emergency centre */
  protected List<Address> emergencyCentres = new ArrayList<Address>();
  
  /** The configuration of this entity.  */
  private EntityDescriptor entityDescriptor;
  
  /** Subscribe message sent by transporter */
  protected Message subscribe = null;
  
  /** List of undelivered messages - Messages */
  protected LinkedList<Message> undeliveredMessages = new LinkedList<Message>();

  /**  Map of messages to send the done reply (when cargo is unloaded) */
  protected Map<String,Message> doneMessageMap = new HashMap<String,Message>();
  
  /** Info displayed on the 3D visio. */
  protected String visioInfo = "";
  
  /** This says whether the driver has shown its info in visio. */
  protected boolean visioInfoShown = false;

  /** This agent's page (public information). */
  private Page page;
  
  /** Time when the last task will be finished. */
  protected double totalTaskTime = 0;

    /**
     * Method invoked on agent's startup.  The initialization runs in these steps:
     * <li> Call superclass initializer. </li>
     * <li> Initialize GUI. </li>
     * <li> Tries to locate its transporter agent </li>
     * <li> Subscribes topics (bandit, simulation) </li>
     * <li> Builds its page and publishes it. </li>
     * <li> Tries to locate the show_info parameter and behave according to its value. </li>
     * <li> Register with DirectoryService. </li>
     * 
     * @param ai AgentInfo
     * @param initState int
     */
    public void init(AgentInfo ai, int initState) {
        super.init(ai,initState);
        
        gui = new DriverAgentGUI(this);

        UpdateTransporter utrans = new UpdateTransporter(this, transporter);
        // Subscribe all topics at server
        gisShell = (GISClientService.Shell)getContainer().getServiceManager().getService(this,GISClientService.SERVICENAME);
        if (gisShell != null) {
            gisShell.subscribeTopic(TOPIC_ENTITY_CONNECTION, this);
            gisShell.subscribeTopic(TOPIC_ENTITY_CONTROL, this);
            gisShell.subscribeTopic(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND, this);
            gisShell.subscribeTopic(UniversalSensorAgent.TOPIC_SENSORY_DATA, this);
        } else {
            logger.warning("Driver Agent error: GISClientService isn't running on this container");
        }
        dsShell = (DirectoryService.Shell) getContainer().getServiceManager().getService(this,DirectoryService.SERVICENAME);
        if(null != dsShell) {
            Collection<String> driv = new LinkedList<String>();
            driv.add(DriverAgent.TYPE);
            try {
                dsShell.register(this, driv);
                dsShell.subscribe(utrans,across.agents.transporter.TransporterAgent.TYPE);
                dsShell.subscribe(this, EmergencyAgent.TYPE);
            } catch (DirectoryException e) {
                logger.warning("Driver Agent error: Directory service exception: " + e);
            }
        } else {
            logger.warning("Driver Agent error: DirectoryService isn't running on this container");
        }
    }
    
    /**
     * This method processes the incoming configuration file and parses out the map and the driver configurations
     * e.g. acutalNode, capacity...
     * @param ed
     */
    private void handleConfiguration(EntityDescriptor ed) {
  	    this.entityDescriptor = ed;
  	    mySimData.capacity = Double.parseDouble(ed.confParamsString.get(AcrossDataConstants.VEHICLE_CAPACITY));
  	    mySimData.consumption = Double.parseDouble(ed.confParamsString.get(AcrossDataConstants.VEHICLE_CONSUMPTION));
  	    mySimData.velocity = Double.parseDouble(ed.confParamsString.get(AcrossDataConstants.VEHICLE_VELOCITY));

  	    myMovement.handleConfiguration(ed, mySimData);
  	    mySensors.handleConfiguration(ed);
  	  
  	    updatePage();
  	    showConfInGui();
      }


    /**
     * Updates info about this driver in visio.
     * @param ai
     */
    private void updateInfoInVisio() {
       // submit the page to be taken into account by visio for actions
       updatePage();
       gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_PAGE_UPDATE, page);
    }
    
    /**
     * Updates this agent Page - the public info about this agent.
     *
     */
    private void updatePage() {
    	 page = new Page();
         page.setAddress(this.getAddress());
         page.setName(this.getName());
         page.setPublicParams(new PublicParams());
         page.getPublicParams().setType(TYPE);
    }
    
  /**
   * This method performs a clean up
   */
  protected void finish() {
    gui.dispose();

    super.finish();
  }

  /**
   * Handles these messages:
   * <li> SUBSCRIBE - from transporter (content is the home node).  Driver sets its 
   * transporter and shows things in visio. It also updates its page. </li>
   * <li> QUERY_REF - from transporter (content is the transport order - the set of waypoints
   * that the driver should go through and deliver things). The driver replies with the
   * estimated time that will take it to drive through the waypoints (in form or Proposal).
   * The driver DOES NOT carry out the journey. </li>
   * <li> REQUEST - from transporter (content TransportOrder).  This is the command to
   * go through the waypoints. </li>
   * @param m Message
   */
  protected void handleIncomingMessage(Message m) {
    if (m.getPerformative().equals(MessageConstants.SUBSCRIBE)) {
      if (transporter == null) {
        transporter = m.getSender();
        subscribe = m;
        
        myMovement.setHomeNode((String) m.getContent());
        
        // TODO - visio
        int i = transporter.getName().indexOf("Transporter");
        visioInfo = "<b>" + transporter.getName().substring(0, i);
        visioInfo += " ";
        visioInfo += this.getName();
        String info = visioInfo + " (" + (long)mySimData.loadedCapacity + "/" + (long)mySimData.capacity + ")";
        gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_INFO, info);
        Byte load = new Byte( (byte)(mySimData.loadedCapacity/mySimData.capacity * 255) );
        gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_LOAD_STATUS, load.toString());
        visioInfoShown = true;
        
        // XXX - gui
        gui.jTextFieldTransporter.setText(transporter.getName());

        Message reply = m.getReply();
        reply.setPerformative(MessageConstants.AGREE);
        send(reply);

        // JT 04_10_06 Added  - info about owner transporter of this driver
        Param param = new Param();
        param.setName(MY_TRANSPORTER);
        param.setValue(transporter.getName());
        page.getPublicParams().getParam().add(param);
        
        // FIXME - this has to be fixed so that there is some notification once there is a change in public info
        gisShell.submitTopic(across.visio.oldvisio.VisioConnectionAgent.TOPIC_PAGE_UPDATE, page);
      } else {  // Transporter is already assigned
        Message reply = m.getReply();
        reply.setPerformative(MessageConstants.REFUSE);
        send(reply);
      }
    }
    
    if (m.getPerformative().equals(MessageConstants.QUERY_REF)){
      TransportOrder to = (TransportOrder) m.getContent();

      // try to plan the route - by price first
      RouteInfo ri = myMovement.getRouteInformation(to, true);
      
      if (ri == null) {
    	  ri = myMovement.getRouteInformation(to, false);  // by time
      }
      if (ri == null) {
        // unable to plan the route
        /** @todo send failure/refuse message */
        return;
      }
      
      Message proposal = m.getReply();
      proposal.setPerformative(MessageConstants.INFORM_RESULT);
      Proposal p = new Proposal();
      p.setProposalTime((long)ri.routeTime);
      p.setRequestid(""+hashCode()*System.currentTimeMillis());
      p.setTotalPrice((long)ri.routePrice);
      proposal.setContent(p);
      send(proposal);
    }
    
    if (m.getPerformative().equals(MessageConstants.REQUEST)){
      TransportOrder to = (TransportOrder) m.getContent();

      // TODO -  /** @todo ojebavka (a navic prasarna), vim, ze vykladam ve druhem waypointu prvni request list, potom poslu done na prislusnou message */
      if (to.getWaypoint().size() == 2) {
        String requestid = ( (RequestList) ( (Waypoint) to.getWaypoint().get(1)).getRequestList().get(0)).getRequestid();
        doneMessageMap.put(requestid, (Message) m.clone());
      }
      
      // try to plan the route - by price first
      RouteInfo ri = myMovement.carryOutRoute(to, true);
      if (ri == null) {
    	  ri = myMovement.carryOutRoute(to, false);  // by time
      }
      if (ri == null) {
        // unable to plan the route
        /** @todo send failure/refuse message */
        return;
      }
      
      totalTaskTime += ri.routeTime;
      
      // XXX - gui
      gui.jTextFieldTaskTime.setText(Double.toString((double)((int)(100*totalTaskTime))/100));
      gui.addToTextArea("Current plan: " + myMovement.waypoints);
      gui.routeTabModel.fireTableDataChanged();
    }
  }

  
  /**
   * Processes these topics:
   * <li> TOPIC_ENTITY_CONNECTION - incoming configuration </li>
   * <li> TOPIC_SIMULATION_TIME_UPDATE_1_SECOND - simulation tick - updates info in visio </li>
   * <li> TOPIC_ENTITY_CONTROL - communication with the DriverBehaviour - the class that moves the driver around the map </li>
   * <li> TOPIC_SENSORY_DATA - data from the drivers sensor </li>
   *
   * @param topic String
   * @param content Object
   * @param reason String
   */
  public void handleTopic(String topic, Object content, String reason) {
	  if(topic.equals(TOPIC_ENTITY_CONNECTION)) {
		  if(content instanceof EntityDescriptor) {
			  handleConfiguration((EntityDescriptor) content);
		  }
	  } else if (topic.equalsIgnoreCase(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND)) {
		  updateInfoInVisio();
		  myMovement.move();
		  // if the driver idles somewhere, send it home.
		  if(!myMovement.isHome() && !myMovement.isOnRoute()) {
			  myMovement.goHome();
		  }
	  } else if(topic.equalsIgnoreCase(TOPIC_ENTITY_CONTROL)) {
		  myMovement.handleMovementTopic(content,reason);
		  if (!visioInfoShown) {
		        String info = visioInfo + " (" + (long)mySimData.loadedCapacity + "/" + (long)mySimData.capacity + ")";
		        gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_VISIO_INFO, info);
		        Byte load = new Byte( (byte)(mySimData.loadedCapacity/mySimData.capacity * 255) );
		        gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_LOAD_STATUS, load.toString());
		        visioInfoShown = true;
		  }
      } else if (topic.equalsIgnoreCase(UniversalSensorAgent.TOPIC_SENSORY_DATA)){
    	  mySensors.processSensorTopic(content);
    	  for (String objId : mySensors.getObjectsInSight()) {
			 SimulationObjectDescription sod = mySensors.getObjectsDescription(objId);
			 if(sod!=null) {
				 SimulationObjectLocation sol = sod.getLocation();
				 if(myMovement.isVehicleGoingThroughRoad(sol.startLocation, sol.endLocation)) {
					 myMovement.obstacleDetected(sod);
					 reportObstacle(sod);
				 }
			 }
    	  }
      } else {
    	  logger.severe("Unexpected topic: " + topic);
      }
  }
  
  /**
   * Reports the obstacle to the emergency centre.
   * @param sod
   */
  protected void reportObstacle(SimulationObjectDescription sod) {
	  
	  // report barred road
	  for (Address adr : emergencyCentres) {
		  Message mes = Message.newInstance(MessageConstants.INFORM);
		  mes.setContent(sod);
		  mes.setReceiver(adr);
		  mes.setSender(getAddress());
		  try {
		 	 sendMessage(mes);
		  } catch (InvisibleContainerException e) {
			  e.printStackTrace();
		  }
	  }
  }
  
  /**
   * Auxilliary GUI function - displays the entity configuration in its visio.
   *
   */
  	private void showConfInGui() {
  		gui.jTextFieldSpeed.setText(""+mySimData.velocity);
        gui.jTextFieldConsumption.setText(Integer.toString((int)mySimData.consumption));
        gui.jProgressBarCapacity.setMaximum((int)mySimData.capacity);
        gui.jProgressBarCapacity.setValue(0);
        gui.jTextFieldPosition.setText(myMovement.actualLoc);
  	}
  	
   
    /**
     * Notifies the transporter that the driver has finished all allocated tasks and is available for new tasks.
     *
     * @return false if transporter is not accessible and the notification was not sent
     */
    protected boolean sendAvailable() {
        if (null != subscribe) {
            Message reply = subscribe.getReply();
            reply.setPerformative(MessageConstants.INFORM_RESULT);
            reply.setContent(myMovement.actualLoc);
            try {
                sendMessage(reply);
                return true;
            } catch (InvisibleContainerException e) {
                return false;
            }
        }
        return false;
    }

  /**
   * Load and unload all batches in actual location
   * @param l List of RequestLists
   */
  protected void loadAndUnload(List<RequestList> l){
    boolean unloading = true;
    for (RequestList rl : l) {
      Address locationAddress = rl.getDeliveryAddress();
      for (Batch batch : rl.getBatch()) {
          // update capacity
    	  long diff = mySimData.loadedCapacity+batch.getCount();
    	  if(diff<0) {
    		  batch.setCount(-mySimData.loadedCapacity);
    	  } else if(diff>mySimData.capacity) {
    		  batch.setCount((long) (mySimData.capacity-mySimData.loadedCapacity));
    	  }
    	  mySimData.loadedCapacity += batch.getCount();  // if count > 0 -> load   count < 0 -> unload
    	  
          Message m = Message.newInstance(MessageConstants.REQUEST, this.getAddress(), locationAddress);
          m.setProtocol(MessageConstants.REQUEST);
          m.setContent(batch);
          send(m);

          // prepare and send visio action
          String content;
          if (batch.getCount() > 0 ) {
        	  // driver sends topical contract id to the BanditAgent
        	  content = Byte.toString(across.visio.oldvisio.VisioConnectionAgent.ACTION_LOAD);
        	  unloading = false;
          } else {
        	  content = Byte.toString(across.visio.oldvisio.VisioConnectionAgent.ACTION_UNLOAD);
          }
        
          gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_ACTION, content ,getAddress().toString());
        
          this.scheduleEvent(new Runnable() {
        	  public void run() {
                  // update visio info
                  String info = visioInfo + " (" + (long)mySimData.loadedCapacity + "/" + (long)mySimData.capacity + ")";
                  gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_INFO, info);
                  Byte load = new Byte( (byte)(mySimData.loadedCapacity/mySimData.capacity * 255) );
                  gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_LOAD_STATUS, load.toString());
                  gui.jProgressBarCapacity.setValue((int)mySimData.loadedCapacity);
        	  }
          }, 200);
      }
      if (unloading) {
    	  sendDone(rl.getRequestid());
      }
    }
  }
  
  
  /**
   * Sends a message - this only shields off the necessity to handle the InvisibleContainerException
   * @param m Message
   */
  private void send(Message m){
    try {
      sendMessage(m);
    }
    catch (InvisibleContainerException e) {
      logger.info("Target platform inaccessible:" + m.getReceiver().toString());
    }
  }

  /**
   * 
   * @param requestid
   */
  protected void sendDone(String requestid){
    Message reply = Message.newInstance(MessageConstants.INFORM_DONE,getAddress(),transporter);
    reply.setProtocol(MessageConstants.REQUEST);

    if (doneMessageMap.get(requestid) == null) {
      System.out.println("ERROR - double use of requestid");
      return;
    }
    //!!!! predpoklad, ze v TransportOrder je pouze jedna nakladka a jedna vykladka

    TransportOrder t = (TransportOrder)(((Message)doneMessageMap.get(requestid)).getContent());
    Waypoint w = (Waypoint)t.getWaypoint().get(1); // unload waypoint from TransportOrder
    RequestList r = (RequestList)(w.getRequestList().get(0)); // unload waypoint has just one RequestList

    reply.setContent(r); // set RequestList as content

    doneMessageMap.remove(requestid);

    try {
      sendMessage(reply);
    }
    catch (InvisibleContainerException e) {
      undeliveredMessages.add(reply);
    }
  }

 
  /**
   * Resends all undelivered messages to the transporter.
   */
  public void resendMessages() {
    if (undeliveredMessages.size() > 0) {
      System.out.println(this.getName() + " - resending " + undeliveredMessages.size() + " message(s) to transporter");
      LinkedList<Message> aux = new LinkedList<Message>();
      for (Message m : undeliveredMessages) {
        try {
          sendMessage(m);
        }
        catch (InvisibleContainerException e) {
          aux.add(m);
        }
      }
      undeliveredMessages = aux;
    }
  }

  /** GUI function - don't use. */
  public LinkedList getWaypoints() {
    return myMovement.waypoints;
  }

  /** GUI function - don't use. */
  public String getActualNode() {
    return myMovement.actualLoc;
  }

  /** GUI function - don't use. */
  public LinkedList getNodeNames() {
    return null;
  }

  /** GUI function - don't use. */
  public RoadInfo[][] getArcs() {
    return null;
  }

  public void sendRequest(String request) {
	// TODO Auto-generated method stub
  }

	public void handleDeregister(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}
	
	public void handleInvisible(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}
	
	public void handleNewRegister(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}
	
	public void handleVisible(String containerName, DirectoryRecord[] records, String matchingFilter) {
		if(matchingFilter.equals(EmergencyAgent.TYPE)) {
			for (int i = 0; i < records.length; i++) {
				DirectoryRecord record = records[i];
				emergencyCentres.add(record.address);
			}
		}
	} 

}
