/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar  1 12:46:34 2007 by Jeff Dalton
 * Copyright: (c) 2006, 2007, AIAI, University of Edinburgh
 */

package ix.iglobe.test;

import iglobe.plan.RescuePlan;
import iglobe.plan.RescueTask;
import iglobe.util.ReportRequestParticipantTask;
import ix.iglobe.test.shared.IGlobeCallback;
import ix.iglobe.test.shared.IGlobeIXAgent;
import ix.util.ColorGenerator;
import ix.util.Debug;
import ix.util.Strings;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;

import aglobe.container.agent.CMAgent;
import aglobe.container.sysservice.directory.DirectoryException;
import aglobe.container.sysservice.directory.DirectoryService;
import aglobe.container.task.ConversationUnit;
import aglobe.container.task.Task;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;

/**
 * An interactive I-Globe rescue planner.
 *
 * <p>An instance of this class is an I-X agent wrapped in an AGlobe
 * agent, sometimes called the "inner" and "outer" agents respectively.
 * The I-X agent is an I-X Process Pannel (I-P2): see {@link IGlobeIp2}.
 * It is created in its own class-loader so that more than one can
 * be made in the same JVM.  To allow information to get in and out
 * across the class-loader "barrier", two shared interface classes
 * are implemented. The inner agent implements {@link IGlobeIXAgent}.
 * The outer agent creates an instance of {@link IGlobeCallback}
 * and gives it to the inner agent by calling its
 * {@link IGlobeIXAgent#setIGlobeCallback(IGlobeCallback) setIGlobeCallback}
 * method.</p>
 *
 * <p>The {@link IXAgentClassLoader class-loader} creates its own
 * copy of classes whose full name begins with "<tt>ix.</tt>" unless
 * the name contains "<tt>.shared.</tt>".  For all other names,
 * it delegates to its parent class-loader, which is the one that
 * loaded the outer agent. Those classes, including the "<tt>.shared.</tt>"
 * ones, are therefore the same in the inner and outer agents.</p>
 *
 * <p>It is possible to use some I-X classes in the outer agent,
 * but this must be done with care.  Consequently, any functions
 * that require extensive use of I-X classes should be performed
 * in the inner agent.  An example in the current version is
 * that when the outer agent receives a rescue plan, it asks
 * the inner agent to display it in an I-X XML "tree editor".</p>
 *
 * <p>The outer agent contains a rescue-task generator that
 * creates a {@link RescueTask} for a randomly chosen location
 * and number of injured.</p>
 *
 * <p>To use a RescuePlanner:
 * <ol>
 * <li>Run an AGlobe platform that has this class in its classpath.
 * <li>Use "Load Agent" in the platform's "File" menu to creat
 *     an instance. (If you've already done this in an earlier
 *     run and didn't ask the RescuePlanner to exit, it should
 *     happen automatically.)  A window should appear containing
 *     the usual I-P2 GUI. One difference is that, to distinguish
 *     different instances when you create more than one, there
 *     is a coloured border.
 * <li>Select "Generate rescue task" from the I-P2 "Test" menu.
 *     That asks the outer agent to generate a task and send it
 *     to the inner agent, where it is transformed into an activity.
 *     You will be asked to enter the names of any resources that
 *     are not available, separated by commas.  Examples of
 *     relevant resources are doctor and ambulance.  Note that
 *     if you make too many resources unavailable, it will not
 *     be possible to construct any plan at all for the task.
 * <li>Do some planning.
 * <ul>
 * <li>In the I-P2, fully expand that activity using the available
 *     refinements.  You can do this using the automatic planner
 *     (select "I-Plan" from the "Tools" menu) or manually
 *     using the menus in the "Action" column.
 * <li>Once you are happy with the expansion, right-click
 *     on the top-level activity and select "Send to HUMRED"
 *     from the menu. (That menu entry will not be present
 *     until the activity is at least partly expanded.)
 *     This will send the plan to the outer agent which
 *     will ask for it to be displayed.
 * <li>If no plan was found, or you want to pretend that none was,
 *     select "Tell HUMRED no plan was found" from the menu.
 * <li>If you would like the rescue planner to work automatically instead,
 *     there is an entry in the "Test" menu (in the GUI's menu bar)
 *     that can be used to switch to automatic operation
 *     or back to manual.
 * </ul>
 * <li>If you want to get rid of the rescue planner, you can
 *     select "Exit" from the I-P2 "File" menu. This will not
 *     cause the whole platform to exit.  <i>(At present,
 *     only the agent's main window is automatically closed
 *     when you ask it to exit in this way.)</i>
 * </ol>
 * </p>
 */
public class RescuePlanner extends CMAgent implements MessageConstants {

	public static final String TYPE = "IX-rescue planner";
	
    static ColorGenerator colorGen = new ColorGenerator();

    protected IGlobeIXAgent ixAgent = null;
    
    protected DirectoryService.Shell dirShell;
    
    protected Map<String,ReportRequestParticipantTask> plansCallback = new HashMap<String, ReportRequestParticipantTask>();

    @Override
    public void init(AgentInfo ai, int initState) {
	super.init(ai, initState);

	logger.fine("Test Rescue Planner started");

	URL domain = getClass().getClassLoader()
	                 .getResource("domain-library/phase1.lsp");

	int rgb;
	synchronized(colorGen) {
	    rgb = colorGen.nextRGB();
	}

	String[] argv = new String[] {
	    "-domain=" + domain,
	    "-step-limit=500",
	    "-background-color=0x" + Integer.toHexString(rgb)
	};
	dirShell = (DirectoryService.Shell) this.getContainer().getServiceManager().getService(this, DirectoryService.SERVICENAME);
	try {
		dirShell.register(this, new LinkedList<String>(Arrays.asList(new String[]{TYPE})));
	} catch (DirectoryException e) {
		e.printStackTrace();
	}
	setIdleTask(new RescuePlannerIdleTask(this));
	try {
	    ixAgent = startSeparateAgent("ix.iglobe.test.IGlobeIp2", argv);
	}
	catch (Exception e) {
	    throw new RuntimeException(e);
	}
    }

    protected class RescuePlannerIdleTask extends Task implements MessageConstants {

		public RescuePlannerIdleTask(ConversationUnit cu) {
			super(cu);
		}

		@Override
		protected void handleIncomingMessage(Message m) {
			if(m.getPerformative().equals(REQUEST)) {
				ReportRequestParticipantTask rrpt = new ReportRequestParticipantTask(this, m) {

					@Override
					protected void processRequest(Message requestMessage) {
						RescueTask rt = (RescueTask) requestMessage.getContent();
						plansCallback.put(rt.getId(), this);
						ixAgent.newTask(rt);
					}
					
				};
			}
		}
    	
    }
    
    public class AcrossCallback implements IGlobeCallback {

		public void exit() {
			// TODO Auto-generated method stub
			
		}

		public void generateTask() {
			// TODO Auto-generated method stub
			
		}

		public void generateTask(List<String> unavailableResources) {
			// TODO Auto-generated method stub
			
		}

		public void noPlan(RescueTask task) {
			ReportRequestParticipantTask rrpt = plansCallback.get(task.getId());
			rrpt.sendInformResult(null);
		}

		public void takePlan(RescuePlan plan, RescueTask task) {
			ReportRequestParticipantTask rrpt = plansCallback.get(task.getId());
			rrpt.sendInformResult(plan);
		}
    	
    }

    IGlobeIXAgent startSeparateAgent(String className, String[] argv)
           throws Exception {

	// Get URLs for the jar files we need.
	URL ixJar = jarURL("ix.jar");
	URL iglobeJar = jarURL("iglobe.jar");

	// Create a new class-loader.
	URL[] urls = new URL[]{iglobeJar, ixJar};
	ClassLoader parent = getClass().getClassLoader();
	ClassLoader loader = new IXAgentClassLoader(urls, parent);

	// Get the agent class from the class-loader.
	Class c = loader.loadClass(className);

	// Create an instance of the agent class.
	Constructor cons = c.getConstructor(new Class[]{});
	IGlobeIXAgent agent = (IGlobeIXAgent)cons.newInstance(new Object[]{});

	// Call the agent's mainStartup(String[] argv) method.
	Class[] sig = new Class[] { argv.getClass() };
	Method m = c.getMethod("mainStartup", sig);
	m.invoke(agent, new Object[] { argv });

	// Tell the agent how to ask us to do things.
	agent.setIGlobeCallback(new AcrossCallback());

	return agent;

    }

    /* This doesn't seem to work well enough.
    URL jarURL(String jarName) {
	URL url = getClass().getClassLoader().getResource(jarName);
	if (url == null)
	    throw new IllegalStateException("Can't find " + jarName);
	return url;
    }
    */

    URL jarURL(String jarName) throws MalformedURLException {
	String classPath = System.getProperty("java.class.path");
	String pathSeparator = System.getProperty("path.separator");
	String fileSeparator = System.getProperty("file.separator");
	for (String p :
	       (List<String>)
		 Strings.breakAt(pathSeparator, classPath)) {
	    String nameOnly = Strings.afterLast(fileSeparator, p);
	    if (nameOnly.equals(jarName)) {
		Debug.noteln("Found jar", p);
		return new File(p).toURL();
	    }
	}
	throw new IllegalStateException("Can't find " + jarName);
    }

    /**
     * A class-loader that creates its own copies of I-X classes.
     * It loads classes whose full name begins with "<tt>ix.</tt>",
     * unless the name contains "<tt>.shared.</tt>, and delegates
     * other cases to its parent.
     */
    public static class IXAgentClassLoader extends URLClassLoader {

	public IXAgentClassLoader(URL[] urls, ClassLoader parent) {
	    super(urls, parent);
	}

	@Override
	public Class loadClass(String name) throws ClassNotFoundException {
	    return loadClass(name, false);
	}

	@Override
	protected Class loadClass(String name, boolean resolve)
	          throws ClassNotFoundException {
	    // System.out.println("Trying class " + name);
	    if (isSharedClass(name))
		return super.loadClass(name, resolve);
	    // System.out.println("Loading class " + name);
	    Class c = findLoadedClass(name);
	    if (c == null)
		c = findClass(name);
	    if (resolve)
		resolveClass(c);
	    return c;
	}

	boolean isSharedClass(String name) {
	    return !name.startsWith("ix.")
		|| name.indexOf(".shared.") > 0;
	}

    }

    class IGLobeServices implements IGlobeCallback {

	TaskGenerator taskGenerator = new TaskGenerator();

	IGLobeServices() {
	}

	public void generateTask() {
	    ixAgent.newTask(taskGenerator.nextTask());
	}

	public void generateTask(List<String> unavailableResources) {
	    RescueTask task = taskGenerator.nextTask();
	    task.setUnavailableResources(unavailableResources);
	    ixAgent.newTask(task);
	}

	public void takePlan(RescuePlan plan, RescueTask task) {
	    Debug.noteln("Received a plan for", task);
	    ixAgent.viewPlan(plan);
	}

	public void noPlan(RescueTask task) {
	    Debug.noteln("No plan for", task);
	}

	public void exit() {
	    RescuePlanner.this.kill();
	}

    }

    static class TaskGenerator {

	Random random = new Random();

	String[] locations = {"London", "Edinburgh", "Prague"};

	int minInjured = 1;
	int maxInjured = 3;

	TaskGenerator() {
	}

	RescueTask nextTask() {
	    RescueTask task = new RescueTask();
	    task.setLocation(randomth(locations));
	    task.setNumberInjured(between(minInjured, maxInjured));
	    return task;
	}

	private <T> T randomth(T... values) {
	    return values[random.nextInt(values.length)];
	}

	private int between(int low, int high) { // inclusive
	    int delta = high - low;
	    return low + random.nextInt(delta + 1);
	}

    }

}
