/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Nov  7 16:34:18 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package iglobe.plan;

import java.io.Serializable;

import java.util.*;

/**
 * A rescue plan.  The plan consists of activities and constraints.
 *
 * <p>Some of the activities may have been "expanded" into
 * subactivities that are also in the plan.  Since the plan
 * also contains ordering constraints, it is in effect a
 * hierarchical task network such as might be produced by
 * an HTN planner.</p>
 *
 * <p>In I-N-C-A terms, the activities are "Nodes" and the
 * consrtaints are, well, "Constraints".  The other parts of
 * the I-N-C-A ontology, Issues and Annotations, are not
 * currently represented.</p>
 *
 * <p>The following order constraints are implicit and are not
 * directly represented in the plan:
 * <ul>
 * <li>The beginning of each activity is before the end of that activity.
 * <li>A child activity must begin after its parent begins and end
 *     before its parent ends.
 * </ul></p>
 *
 * <p>Note that the {@link RescueActivity} class provides utility
 * methods that can be used to extract some information that is not
 * directly represented in the plan.</p>
 *
 * @see ix.iglobe.test.RescuePlanMaker
 */
public class RescuePlan implements Serializable {

    protected List<RescueActivity> activities;
    protected List<PlanConstraint> constraints;

    protected transient Map<String,RescueActivity> activityMap =
	new HashMap<String,RescueActivity>();

    /**
     * Creates an empty plan.
     */
    public RescuePlan() {
    }

    /**
     * Returns this plan's list of activities.
     */
    public List<RescueActivity> getActivities() {
	return activities;
    }

    /**
     * Sets this plan's list of activities.  It also creates an
     * internal map from ids to activities.
     */
    public void setActivities(List<RescueActivity> activities) {
	this.activities = activities;
	activityMap.clear();
	for (RescueActivity act : activities) {
	    activityMap.put(act.getId(), act);
	}
    }

    /**
     * Returns the activity in this plan that has the specified id.
     *
     * @return The activity if it exists, otherwise null.
     */
    public RescueActivity getActivity(String id) {
	return activityMap.get(id);
    }

    /**
     * Returns this plan's list of constraints.
     */
    public List<PlanConstraint> getConstraints() {
	return constraints;
    }

    /**
     * Sets this plan's list of constraints.
     */
    public void setConstraints(List<PlanConstraint> constraints) {
	this.constraints = constraints;
    }

}
