/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Nov  7 16:17:43 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package iglobe.plan;

import java.io.Serializable;

import java.util.*;

/**
 * The root class for all activities that can appear in a {@link RescuePlan}.
 *
 * <p>Each activity can appear in only one plan object and has
 * an id that is unique withing the plan that contains it.
 * The ids have the form "<tt>activity-</tt>" followed by a
 * "<tt>-</tt>"-separated sequence of numbers. The numbers
 * correspond to the activity's position in a tree in which
 * activities are children of the parent activity they "expand".
 * If a child's id is <tt>activity-</tt>...<tt>-<i>n</i>-<i>m</i></tt>,
 * its parent's is <tt>activity-</tt>...<tt>-<i>n</i></tt>.
 * Thus, even though there is currently no "parent" link,
 * the parent can be determined if necessary, and the
 * {@link #getParent(RescuePlan)} method takes advantage
 * of that.</p>
 *
 * <p>Each activity also has a description of the action that
 * is performed when the activity is executed.  The description
 * takes the form a sentence that begins with a verb and
 * contains the values of all of the relevant fields of
 * the activity.  For example, "transport 1 person Glasow".
 * It needn't be readable as natural language, but it should be
 * usable, suitably parsed, as the "pattern" in an I-X activity.</p>
 */
public class RescueActivity implements Serializable {

    protected String id;
    protected String description;

    /**
     * Creates an empty activity.
     */
    public RescueActivity() {
    }

    /**
     * Return this activity's id.
     */
    public String getId()  {
	return this.id;
    }

    /**
     * Set this activity's id.
     */
    public void setId(String newId) {
	this.id = newId;
    }

    /**
     * Return this activity's description.
     */
    public String getDescription()  {
	return this.description;
    }

    /**
     * Set this activity's description.
     */
    public void setDescription(String newDescription) {
	this.description = newDescription;
    }

    /**
     * Returns this activity's parent activity in the specified plan.
     *
     * @return the parent or null if this is a top-level activity.
     *
     * @throws IllegalArgumentException if a parent ought to exist but doesn't.
     */
    public RescueActivity getParent(RescuePlan plan) {
	String parentId = beforeLast("-", getId());
	if (parentId.indexOf("-") < 0)
	    return null;	// top-level activities have null parents
	else {
	    RescueActivity parent = plan.getActivity(parentId);
	    if (parent != null)
		return parent;
	    else
		throw new IllegalArgumentException
		    ("Activity " + getId() + " has no parent.");
	}
    }

    private static String beforeLast(String separator, String s) {
	int i = s.lastIndexOf(separator);
	return (i == -1) ? s : s.substring(0, i);
    }

    /**
     * Returns the activities that are the immediate predecessors
     * of this activity in the specified plan.  The plan's
     * {@link OrderConstraint}s are used.  <i>Note that some
     * orderings are implicit and will not be considered.
     * Only the ones that appear explicitly as constraints
     * will be used.</i>
     */
    public Set<RescueActivity> getPredecessors(RescuePlan plan) {
	Set<RescueActivity> result = new LinkedHashSet<RescueActivity>();
	for (PlanConstraint c : plan.getConstraints()) {
	    if (c instanceof OrderConstraint) {
		OrderConstraint ord = (OrderConstraint)c;
		if (ord.getToId().equals(id))
		    result.add(plan.getActivity(ord.getFromId()));
	    }
	}
	return result;
    }

}
