/**
 * Title:        Visualizer for CPlanT
 * Description:
 * Copyright:    Copyright (c) 2001
 * Company:      Gerstner Laboratory, FEL-�VUT, Prague
 * @author Michal Dob�ek
 * @version 1.0
 */

package across.visio.visual;

// JAVA imports
import java.awt.Color;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import across.visio.VisualAgent;

/**
 * <p>
 * <code>ColorFactory</code> manages the agent colors for different coloring
 * schemes. Each scheme is identified by instance of its <code>PropertyGetter</code>.
 * Getter's method <code>getProperty</code> has agent as parameter and it returns
 * its property used ifor coloring in particular schema (<code>roleGetter</code>,
 * <code>typeGetter</code>, <code>cityGetter</code>, <code>countryGetter</code>).
 * The used getters are defined here as static constants. <code>ColorFactory</code>
 * manages color assignment to each schema-property_valie pair and returns
 * always the same color.</p>
 * <p>The colors for <b>Role</b> schema are predefined. The colors for other
 * schemas are dynamically assigned using the palette defined in config
 * <code>.xml</code> file. </p>
 * <p>The coloring schema is set by the <code>Visualizer</code>. Each agent then
 * asks for its color (and forces its phantom agents to ask). The Color Key
 * (<code>LegentWindow</code>) is notified whenever the coloring schema changes
 * or new color is assigned so that it can show the changes.</p>
 *
 * @see Visualizer
 * @see VisAgent
 * @see LegendWindow
 */

public class ColorFactory
{
  public static final String LEADER = "Coalition Leader";
  public static final String MEMBER = "Coalition Member";
  public static final String OTHER  = "Other";

  private static final Color LEADERc = new Color(255,192,192);
  private static final Color MEMBERc = new Color(192,192,255);
  private static final Color OTHERc  = new Color(192,192,128);

  private static final List PALETTE= VisualAgent.getConfig().getColors();



  private static ColorSchema defaultschema= new ColorSchema();

  static {
    defaultschema.putColor(LEADER, LEADERc);
    defaultschema.putColor(MEMBER, MEMBERc);
    defaultschema.putColor(OTHER, OTHERc);
  }

  public abstract static class PropertyGetter  {
    public PropertyGetter() {}
    public abstract Object getProperty(VisAgentInterface a);
  }

  public static final PropertyGetter roleGetter= new PropertyGetter()
                                                 {
                                                   public Object getProperty(VisAgentInterface a)
                                                   { return a.getRole(); }
                                                  public String toString()
                                                   { return "Role"; }
                                                 };

  public static final PropertyGetter typeGetter= new PropertyGetter()
                                                 {
                                                   public Object getProperty(VisAgentInterface a)
                                                   { return a.getType(); }
                                                  public String toString()
                                                   { return "Type"; }
                                                 };

  public static final PropertyGetter countryGetter= new PropertyGetter()
                                                 {
                                                   public Object getProperty(VisAgentInterface a)
                                                   { return a.getCountry(); }
                                                  public String toString()
                                                   { return "Country"; }
                                                 };

  public static final PropertyGetter cityGetter= new PropertyGetter()
                                                 {
                                                   public Object getProperty(VisAgentInterface a)
                                                   { return a.getCity(); }
                                                  public String toString()
                                                   { return "City"; }
                                                 };

  private static ColorFactory defaultFactory = null;

  public static ColorFactory getDefaultFactory() {
    if (defaultFactory == null) {
      defaultFactory = new ColorFactory();
    }
    return defaultFactory;
  }

  public static interface ColorFactoryListener
  {
    public void updateKeys(Set keys);
  }

  private HashMap _schemas= new HashMap();
  private ColorSchema    _currentschema;
  private PropertyGetter _currentgetter;
  private List           _listeners;

  private ColorFactory() {
    _schemas.put(roleGetter, defaultschema);
    _currentschema= new ColorSchema();
    _currentgetter= countryGetter;
    _listeners= new LinkedList();
  }

  public void setColorSchema(PropertyGetter g)  {
    _currentschema= (ColorSchema)_schemas.get(g);
    _currentgetter= g;
    if(_currentschema==null)
    {
      _currentschema= new ColorSchema();
      _schemas.put(g, _currentschema);
    }
    updateKeys();
  }

  protected void updateKeys()
  {
    Set keys= _currentschema.getKeys();
    Iterator i= _listeners.iterator();
    while(i.hasNext())
      ((ColorFactoryListener)i.next()).updateKeys(keys);
  }

  public Color getColor(VisAgentInterface a)
  {
    return _currentschema.getColorFor(_currentgetter.getProperty(a));
  }

  public Color getKeyColor(Object key)
  {
    return _currentschema.getKeyColor(key);
  }

  public void registerColorFactoryListener(ColorFactoryListener l)
  {
    _listeners.add(l);
    l.updateKeys(_currentschema.getKeys());
  }

  public void deregisterColorFactoryListener(ColorFactoryListener l)
  {
    _listeners.remove(l);
  }

  private static class ColorSchema
  {
    int _count= 0;
    HashMap _colors= new HashMap();

    public ColorSchema() {}

    public Color getColorFor(Object o)
    {
      Color c= (Color)_colors.get(o);
      if(c==null)
      { if (_count >= PALETTE.size() ) {_count = 0;} // petr
        c= (Color)PALETTE.get(_count++);
        _colors.put(o, c);
        defaultFactory.updateKeys();
      }
      return c;
    }

    public void putColor(Object key, Color value)
    {
      _colors.put(key, value);
    }

    public Color getKeyColor(Object key)
    {
      return (Color)_colors.get(key);
    }

    public Set getKeys()
    {
      return _colors.keySet();
    }
  }
}
