package across.visio.oldvisio;

import java.util.LinkedList;
import java.util.List;

import across.data.Batch;
import across.data.Page;
import aglobe.container.agent.Agent;
import aglobe.container.transport.Address;
import aglobe.container.transport.MessageTransport;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.CommunicationInfo;
import aglobe.ontology.Message;
import aglobe.service.gis.server.GISServerService;
import aglobe.service.gis.server.GISTopicServerListener;
import aglobe.visio3D.ontology.Chart;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.global.EntitiesPositionInfo;
import aglobex.simulation.global.EntityRecord;
import aglobex.simulation.global.ServerServerTopicConstants;
import aglobex.simulation.ontology.Configuration;



/**
 * <p>Title: Across</p>
 * <p>Description: Simulates node pod for the all connected containers. Moves with entities on the roads.
 * Commands come from the driver agents.</p>
 */

public class VisioConnectionAgent extends Agent
    implements GISTopicServerListener,ClientServerTopicConstants,ServerServerTopicConstants {

  /**
   * Topic used for updating visio info text near the world entity. In the content is String object.
   */
  public final static String TOPIC_VISIO_INFO = "VISIO_INFO";

  /**
   * Topic used for updating public params. In the content is page object.
   */
  public final static String TOPIC_PAGE_UPDATE = "PAGE_UPDATE";

  /**
   * Topic used for removing public params (remove agent). In the content there is agent address in string.
   */
  public final static String TOPIC_PAGE_REMOVED = "PAGE_REMOVED";

  /**
   * Topic used for sending load status to the visio. Content is number 0 - 255 where 255 is full load. Number
   * is encoded in string with Byte.toString(value)
   */
  public final static String TOPIC_LOAD_STATUS = "LOAD_STATUS";

  /**
   * Topic used for creating/updating chart in the visio. Content contains chart definition as aglobe.visio3D.ontology.Chart,
   * reason contains chart name. Each container/agent could have more charts with differnt chart name.
   */
  public final static String TOPIC_SET_CHART = "SET_CHART";

  /**
   * Topic used for removing chart previously defined by the TOPIC_SET_CHART topic.
   * Name of the chart to remove is specified in the reason. Content is null.
   */
  public final static String TOPIC_REMOVE_CHART = "REMOVE_CHART";

  /**
   * Topic used for sending information about code transfer. Content contains Receiver address in string format.
   */
  public final static String TOPIC_CODE_TRANSFER = "CODE_TRANSFER";

  /**
   * Topic used for sending action to the visio. Content is one of the ACTION_xx constant encoded in String and in the Reason
   * there is address of the sender encoded with .toString()
   */
  public final static String TOPIC_VISIO_ACTION = "VISIO_ACTION";

  public final static String TOPIC_VISIO_REL_INFO = "VISIO_REL_INFO";

  
  public static final String TOPIC_NEW_CONTRACT = "NEW_CONTRACT";
  public static final String TOPIC_HUMAGENT_CONTAINER = "HUMAGENT_CONTAINER";
  public static final String TOPIC_BATCH_LOADED_ON_LOCATION = "BATCH_LOADED_ON_LOCATION";
  public static final String TOPIC_BATCH_UNLOADED_FROM_LOCATION = "BATCH_UNLOADED_FROM_LOCATION";
  public static final String TOPIC_BATCH_LOADED_ON_DRIVER = "BATCH_LOADED_ON_DRIVER";
  public static final String TOPIC_BATCH_UNLOADED_FROM_DRIVER = "BATCH_LOADED_FROM_DRIVER";
  public static final String TOPIC_BATCHES_LEFT_IN_SPACE = "BATCH_LEFT_IN_SPACE";
  
  
  /**
   * Action contants
   */
  public final static byte ACTION_LOAD = 1;
  public final static byte ACTION_UNLOAD = 2;
  public final static byte ACTION_CFP_GOODS = 11;
  public final static byte ACTION_CFP_TRANSPORT = 12;
  public final static byte ACTION_TRANSPORT_START = 21;
  public final static byte ACTION_GOODS_STOLEN = 41;
  public final static byte ACTION_VEHICLE_DESTROYED = 42;
  public final static byte ACTION_GOODS_FRAUDED = 45;
  public final static byte ACTION_STANDIN_CREATED = 51;
  public final static byte ACTION_STANDIN_KILLED = 52;
  public final static byte ACTION_STANDIN_MIG_OUT = 53;
  public final static byte ACTION_STANDIN_MIG_IN = 54;
  public final static byte ACTION_STANDIN_CLONNED = 55;

  public final static byte ACTION_POLL_START = 71;
  public final static byte ACTION_POLL_YES = 72;
  public final static byte ACTION_POLL_NO = 73;
  public final static byte ACTION_POLL_EXPELLED = 74;
  
  public final static byte ACTION_VEHICLE_STOPPED = 81;
  public final static byte ACTION_CONVOY_STARTED = 82;

  /**
   * reference to the gis server service shell
   */
  GISServerService.Shell gisShell;

  /**
   * Object handling all the visualization.
   */
  private Visualizer visualizer;

  /**
   * configuration of the world.
   */
  private Configuration conf;
    
    /**
     * Initialization of NodePodAgent
     * @param ai Agent should get its name from here
     * @param initState int
     */
    public void init(AgentInfo ai, int initState) {
      super.init(ai,initState);

      gisShell = (GISServerService.Shell)getContainer().getServiceManager().getService(this,GISServerService.SERVICENAME);

      // visio topics
      if (gisShell != null) {
        gisShell.subscribeTopic(TOPIC_PAGE_UPDATE,this);
        gisShell.subscribeTopic(TOPIC_PAGE_REMOVED, this);
        gisShell.subscribeTopic(MessageTransport.TOPIC_COMMUNICATION_INFO, this);
        gisShell.subscribeTopic(TOPIC_VISIO_INFO, this);
        gisShell.subscribeTopic(TOPIC_LOAD_STATUS, this);
        gisShell.subscribeTopic(TOPIC_VISIO_ACTION, this);
        gisShell.subscribeTopic(TOPIC_SET_CHART, this);
        gisShell.subscribeTopic(TOPIC_REMOVE_CHART, this);
        gisShell.subscribeTopic(TOPIC_CODE_TRANSFER, this);
        gisShell.subscribeTopic(TOPIC_HUMAGENT_CONTAINER, this);
        gisShell.subscribeTopic(TOPIC_NEW_CONTRACT, this);
        gisShell.subscribeTopic(TOPIC_BATCH_LOADED_ON_LOCATION, this);
        gisShell.subscribeTopic(TOPIC_BATCH_UNLOADED_FROM_LOCATION, this);
        gisShell.subscribeTopic(TOPIC_BATCH_LOADED_ON_DRIVER, this);
        gisShell.subscribeTopic(TOPIC_BATCH_UNLOADED_FROM_DRIVER, this);
        gisShell.subscribeTopic(TOPIC_BATCHES_LEFT_IN_SPACE, this);
      } else {
        logger.severe("VisioConnectionAgent must be started on the master container !!!");
      }
      
      // Server topics
      gisShell.subscribeTopic(TOPIC_CONFIGURATION, this);
      gisShell.subscribeTopic(TOPIC_ENTITY_LOGGED, this);
      gisShell.subscribeTopic(TOPIC_ENTITY_LOGOUT, this);
      gisShell.subscribeTopic(TOPIC_POSITION_UPDATE,this);
      gisShell.subscribeTopic(TOPIC_ENTITY_CRASH, this);
      
      // request configuration
      gisShell.sendTopicToLocal(TOPIC_CONFIGURATION_REQUEST, null, getName());
    }
    
  /**
   * Handles incoming message
   * @param m Message
   */
  protected void handleIncomingMessage(Message m) {
    logger.severe("Unexpected incoming message: "+m);
  }

  /**
   * Called by GISServer service to infrom that some client service subscribe for
   * specific topic info.
   *
   * @param topic String - topic name
   * @param remoteContainerName String - remote container name, if null it is
   *   local subscriber
   * @param remoteContainerAddress Address - remote container address, if null
   *   it is local subscriber
   */
  public void handleLoginTopic(String topic, String remoteContainerName,
                               Address remoteContainerAddress) {
   // nothing
  }

  /**
   * Called by GISServer service to infrom that some client service unsubscribe
   * for specific topic info.
   *
   * @param topic String - topic name
   * @param remoteContainerName String - remote container name, if null it is
   *   local subscriber
   */
  public void handleLogoutTopic(String topic, String remoteContainerName) {
   // nothing
  }

  /**
   * Handles incomming configuration.
   *
   */
  private void handleConfiguration() {
	  if(conf==null) {
		  logSevere("Null configuration!");
		  throw new RuntimeException("Null configuration!");
	  } else {
		 if(visualizer==null) {
			  visualizer = Visualizer.newInstance(this,conf);
			  gisShell.sendTopicToLocal(TOPIC_CURRENT_ENTITIES_REQUEST, null, getName());
		  }
	  }
  }
  
  /**
   * Handles these topics:
   * <li> TOPIC_CONFIGURATION </li>
   * <li> TOPIC_POSITION_UPDATE </li>
   * <li> TOPIC_ENTITY_LOGGED </li>
   * <li> TOPIC_ENTITY_LOGOUT </li>
   * <li> TOPIC_CURRENT_ENTITIES </li>
   * <br><br>
   * Handles these visio topics:
   * <table border="1">
   * <tr> <td> Topic </td> <td> Explanation </td> </tr>
   * <tr> <td> TOPIC_COMMUNICATION_INFO </td> <td> Display communication between two entities (content aglobe.ontology.CommunicationInfo) </td> </tr>
   * <tr> <td> TOPIC_PAGE_UPDATE </td> <td> Display information about an agent in visio (content across.data.Page) </td> </tr>
   * <tr> <td> TOPIC_PAGE_REMOVED </td> <td> Stop displaying agent page in visio (content Address in String) </td> </tr>
   * <tr> <td> TOPIC_STANDIN_REMOVED </td> <td> Stop displaying agent page in visio (content Address in String) </td> </tr>
   * <tr> <td> TOPIC_VISIO_INFO </td> <td> Updates info boxes for all entities </td> </tr>
   * <tr> <td> TOPIC_LOAD_STATUS </td> <td> Updates driver info - (content Byte representing load) </td> </tr>
   * <tr> <td> TOPIC_VISIO_ACTION </td> <td> Carries out a visio action specified in the content (Byte) </td> </tr>
   * <tr> <td> TOPIC_SET_CHART </td> <td> </td> </tr>
   * <tr> <td> TOPIC_REMOVE_CHART </td> <td> </td> </tr>
   * <tr> <td> TOPIC_CODE_TRANSFER </td> <td> </td> </tr>
   * <tr> <td> TOPIC_NEW_CONTRACT </td> <td> </td> </tr>
   * <tr> <td> TOPIC_BATCH_LOADED_ON_DRIVER </td> <td> </td> </tr>
   * <tr> <td> TOPIC_BATCH_UNLOADED_FROM_DRIVER </td> <td> </td> </tr>
   * <tr> <td> TOPIC_BATCH_LOADED_ON_LOCATION </td> <td> </td> </tr>
   * <tr> <td> TOPIC_BATCH_UNLOADED_FROM_LOCATION </td> <td> </td> </tr>
   * <tr> <td> TOPIC_HUMAGENT_CONTAINER </td> <td> </td> </tr>
   * <tr> <td> TOPIC_BATCHES_LEFT_IN_SPACE </td> <td> </td> </tr>
   * <tr> <td> TOPIC_VISIO_REL_INFO </td> <td> </td> </tr>
   * </table>
 
   */
  public void handleTopic(String topic, Object content, String reason,
                          String remoteContainerName,
                          Address remoteClientAddress) {
	if(topic.equalsIgnoreCase(TOPIC_CONFIGURATION)) {
		conf = (Configuration) content;
		handleConfiguration();
    } else if(topic.equalsIgnoreCase(TOPIC_POSITION_UPDATE)) {
   		visualizer.positionUpdate((EntitiesPositionInfo) content);
    } else if(topic.equalsIgnoreCase(TOPIC_ENTITY_CRASH)) {
    } else if (topic.equalsIgnoreCase(TOPIC_ENTITY_LOGGED)){
   		visualizer.entityLogged((EntityRecord) content);
    } else if (topic.equalsIgnoreCase(TOPIC_ENTITY_LOGOUT)) {
   		visualizer.entityLogout((EntityRecord) content);
    } else if (topic.equalsIgnoreCase(TOPIC_CURRENT_ENTITIES)) {
    	visualizer.entityLogged((EntityRecord) content);
    } else if (topic.equalsIgnoreCase(MessageTransport.TOPIC_COMMUNICATION_INFO)) {
      if (remoteContainerName != null) {
    	  if(content instanceof CommunicationInfo) {
	        CommunicationInfo gi = (CommunicationInfo) content;
	        visualizer.incomingCommunication(gi.getFrom(), gi.getTo());
    	  }
      }
    } else if (topic.equalsIgnoreCase(VisioConnectionAgent.TOPIC_PAGE_UPDATE)) {
    	if (content instanceof Page) {
            visualizer.incomingAgentInfo( (Page) content);
        } else {
            logger.warning("Unknown type of agentInfo: " + content.getClass().getName());
        }
    } else if (topic.equalsIgnoreCase(TOPIC_VISIO_INFO) && ((remoteContainerName != null) || (reason!=null))) {
    	if(remoteContainerName!=null) {
    		visualizer.updateVisioInfo(remoteContainerName,(String)content);
    	} else {
    		visualizer.updateVisioInfo(reason,(String)content);
    	}
    } else if (topic.equalsIgnoreCase(TOPIC_LOAD_STATUS)) {
	      byte load = 0;
	      try {
	        load = Byte.parseByte( (String) content);
	      }
	      catch (NumberFormatException ex2) {
	      }
	      visualizer.updateVisioInfo(remoteContainerName,load);
    } else if ((topic.equalsIgnoreCase(TOPIC_VISIO_ACTION)) && (remoteContainerName != null)) {
	      byte action = 0;
	      try {
	        action = Byte.parseByte( (String) content);
	      }
	      catch (NumberFormatException ex3) {
	        logger.warning("Bad action id: "+(String)content);
	        return;
	      }
	      Address agentAddress = Address.getAddress(reason);
	      visualizer.showVisioAction(remoteContainerName,agentAddress,action);
    } else if ((topic.equalsIgnoreCase(TOPIC_SET_CHART)) && (remoteContainerName != null)) {
      visualizer.visioSetChart(remoteContainerName, reason, (Chart)content);
    } else if ((topic.equalsIgnoreCase(TOPIC_SET_CHART)) && (remoteContainerName == null)) {
       visualizer.visioSetChart(reason, reason, (Chart) content);
    } else if ((topic.equalsIgnoreCase(TOPIC_REMOVE_CHART)) && (remoteContainerName != null)) {
      visualizer.visioRemoveChart(remoteContainerName,reason);
    } else if ((topic.equalsIgnoreCase(TOPIC_CODE_TRANSFER)) && (remoteContainerName != null)) {
      visualizer.sendCodeTransferToVisio(remoteContainerName, Address.getAddress((String)content));
    } else if ((topic.equalsIgnoreCase(TOPIC_NEW_CONTRACT)) && (remoteContainerName != null)) {
        visualizer.addNewContract((List)content, reason);
    } else if ((topic.equalsIgnoreCase(TOPIC_BATCH_LOADED_ON_DRIVER)) && (remoteContainerName != null)) {
        Batch b = (Batch)content;
        visualizer.batchLoadedOnContainer(b, remoteContainerName);
    } else if ((topic.equalsIgnoreCase(TOPIC_BATCH_UNLOADED_FROM_DRIVER)) && (remoteContainerName != null)) {
        Batch b = (Batch)content;
        visualizer.batchUnloadedFromContainer(b, remoteContainerName);
    } else if ((topic.equalsIgnoreCase(TOPIC_BATCH_LOADED_ON_LOCATION)) && (remoteContainerName != null)) {
        Batch b = (Batch)content;
        visualizer.batchLoadedOnContainer(b, remoteContainerName);
    } else if ((topic.equalsIgnoreCase(TOPIC_BATCH_UNLOADED_FROM_LOCATION)) && (remoteContainerName != null)) {
        Batch b = (Batch)content;
        visualizer.batchUnloadedFromContainer(b, remoteContainerName);
    } else if ((topic.equalsIgnoreCase(TOPIC_HUMAGENT_CONTAINER)) && (remoteContainerName != null)) {
        visualizer.setHumanitarianContainer((Address)content, reason);
    } else if (topic.equalsIgnoreCase(TOPIC_BATCHES_LEFT_IN_SPACE) && (remoteContainerName != null)) {
        visualizer.batchesLeft((LinkedList)content, remoteContainerName);
    } else if (topic.equalsIgnoreCase(TOPIC_VISIO_REL_INFO)) {
    	if(visualizer!=null) {
    		visualizer.handleTopic(topic, content, reason, remoteContainerName, remoteClientAddress);
    	}
    } else {
    	logger.warning("Unexpected incoming topic: "+topic+" ("+remoteContainerName+")");
    }
  }
}
