package across.util.skn.util.membership;

import across.data.trust.AgentTrustfulness;
import across.util.skn.util.FuzzySet;

/**
 * Fuzzy membership deduced from the sequence. We derive all the fuzzy number propoerties and values from sequence.
 */
public abstract class IterativeFuzzyMembership<MemberObject> extends Membership<MemberObject>
{
    /** Number of samples in the sequence. */
    protected long n = 0;
    /** Average of the sequence */
    protected double avg = 0.5; // initial value has no sense, as n is zero at the beginning
    /** Average of the x*x sequence, usefull for variance estimate */
    protected double avg2 = 0.25; // initial value has no sense, as n is zero at the beginning
    /** Minimum value in the sequence */
    protected double min = 1;
    /** Maximum value in the sequence */
    protected double max = 0;

    /** This function is called when new sample is observed.
     * Updates the properties of the sequence.
     * @param value observed value of the sample*/
    public void newSample(double value)
    {
        // calculate the properties of the sequence
        //System.out.println("Old Membership: " + avg + " avg2 " + avg2 + " n: " + n + " min: " + min + " max: " + max);
        avg = (avg*n + value)/(n+1);
        avg2 = (avg2*n + value*value)/(n+1);
        n++;
        min = Math.min(min, value);
        max = Math.max(max, value);
//        System.out.println("New Membership: " + avg + " avg2 " + avg2 + " n: " + n + " min: " + min + " max: " + max);
        // and determine the characteristics, depending on the representation we chose
        applyChangesToShape();
    }
    /**
     * Replaces the old sequence values with the new ones. Used for Reputatio and other like...
     * @param agt new sequence values to be used
     */
    public void setData(AgentTrustfulness agt)
    {
        //System.out.println("Old Membership: " + avg + " avg2 " + avg2 + " n: " + n + " min: " + min + " max: " + max);
//    	 reset the properties of the sequence
    	avg = agt.getAvg();
        avg2 = agt.getAvg2();
        n= agt.getNumOfObservations();
        min = agt.getMin();
        max = agt.getMax();
//        System.out.println("New Membership: " + avg + " avg2 " + avg2 + " n: " + n + " min: " + min + " max: " + max);
        // and determine the characteristics, depending on the representation we chose
        applyChangesToShape();
    }
    
    /**
     * Replaces the old sequence values with the new ones. Used for Reputatio and other like...
     * @param agt new sequence values to be used
     */
    public void setData(IterativeFuzzyMembership original) {
		n = original.n;
		avg = original.avg;
        avg2 = original.avg2;
        min = original.min;
        max = original.max;
        applyChangesToShape();
    }

    /**
     * This method reads the sequence properties and modifies the points array of the FuzzyNumberMembership class.
     */
    protected abstract void applyChangesToShape();

    /**
     * Returbs an ESTIMATE of variance, defined as (Ex)^2 - E(x^2)
     * @return estimate of variance, reasonably exact for big number of members in the sequence
     */
    public double getVariance()
    {
        return Math.abs(avg2 - avg*avg);
    }

    /**
     * Returns true if the value is based on sufficient ammount of data, False otherwise.
     * @return true if the ammount of data on which the trust is based is sufficient
     */
    @Override public boolean hasEnoughData() {
        if (n >= 5 ) return true;
        return false;
    }

    /**
     * Compares this value with a number whose X values were shifted by multiplication with coeff with the other one.
     * Note that by multiplication with coeff smaller than 1 we shrink the uncertainity and vice versa.
     * @param coeff coefficient used to multiply the x values of the current number
     * @param other membership function to compare with
     * @return 1 if bigger than other, -1 if smaller than other, 0 if incomparable or equivalent
     */
    public abstract int compareWithCoefficient(double coeff, IterativeFuzzyMembership other);


    public IterativeFuzzyMembership(MemberObject memberObject, FuzzySet set) {
        super(memberObject, set);
    }

    public IterativeFuzzyMembership(MemberObject o, FuzzySet set, IterativeFuzzyMembership<MemberObject> original) {
		super(o,set);
		setData(original);
	}
    
	/**
     * Returns a string representation of the object. In general, the
     */
    public String toString()
    {
        StringBuilder sb = new StringBuilder("avg: ");
        sb.append(avg);
        sb.append(" sigma :" );
        sb.append(Math.sqrt(getVariance()));
        sb.append(" min :" );
        sb.append(min);
        sb.append(" max :" );
        sb.append(max);
        return sb.toString();
    }
    
    public AgentTrustfulness getAgentTrustfulness(String name, long currentTime) {
		AgentTrustfulness res = new AgentTrustfulness();
		res.setAgentName(name);
		res.setCurrentTime(currentTime);
		res.setNumOfObservations(n);
		res.setAvg(avg);
		res.setAvg2(avg2);
        res.setMin(min);
        res.setMax(max);
		return res;
	}
}
