package across.util.skn.plan.across;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;

import across.data.TransportBatch;
import across.data.TransportCfp;
import across.util.ListBatchTools;
import across.util.skn.plan.Action;

/**
 * This class represents a plan with Actions and corresponding commitments.
 *
 */
public class AcPlan {

	/** Template to which the plan corresponds */
	protected AcPlanTemplate planTemplate;

	/** Original Call for Proposals with client request details */
	protected TransportCfp tcfp;
	/** References the plan base with list of resources available for task allocation */
	protected PlanBase planBase;

	/** Contains tasks organized by action descriptor */
	protected LinkedHashMap<String,AcTaskTransport> tasks = new LinkedHashMap<String, AcTaskTransport>();

	public double objectiveValueLp = Double.NEGATIVE_INFINITY;
	
	/**
	 * Creates the plan object relative to the plan template and client request.
	 * @param template Template of the plan - Actions and Objectives, no allocations or partners
	 * @param tcfp Original client request - id and list of batches to carry
	 */
	public AcPlan(AcPlanTemplate template, TransportCfp tcfp, PlanBase pb) {
		super();
		planBase = pb;
		planTemplate = template;
		this.tcfp = tcfp;
		allocateTasksToActions();
	}

	/** Allocates tasks/batches to branches in the plan.
	 * We assume that all the cargo goes from the first prerequisite to the first target (achieves).*/
	protected void allocateTasksToActions()
	{
		// width-first search - open list
		List<AcTaskTransport> openActions = new LinkedList<AcTaskTransport>();
		//iterate through the plan and cat in halves/thirds in each Objective where more than one way is possible
		// start from the prerequisites - allocate them the original cfp
		expandList(planTemplate.getPrerequisites(), (tcfp.getTransportBatch()), openActions);
			// there is nomrally a single element, this is only to be on the safe side - with multiple starts, we divide the batches between them
			while(! openActions.isEmpty())
			{
				AcTaskTransport task = openActions.remove(0);
				expandTask(task, openActions);
				tasks.put(task.getActionReference().getActionDescriptor(), task);
			}
	}

	private void expandTask(AcTaskTransport task, List<AcTaskTransport> openList)
	{
		expandList(task.actionReference.getAllows(), task.batches, openList);
	}

	private void expandList(Collection<Action> actions, List<TransportBatch> batches, List<AcTaskTransport> openList)
	{
		double ratio  =  1/(double)actions.size();
		// distribute over achieves first
		for (Iterator<? extends Action> itac = actions.iterator(); itac.hasNext();) {
			AcTransport actr = (AcTransport) itac.next();
			AcTaskTransport nt = getTaskForAction(actr);
			//TODO - prepare better strategy and integrate it here
			allocatePartToTask(nt, batches, ratio);
			openList.add(nt);
		}
	}

	protected AcTaskTransport getTaskForAction(AcTransport ac)
	{
		AcTaskTransport res = tasks.get(ac.getActionDescriptor());
		if (null == res)
		{
			res = new AcTaskTransport(ac, this);
			tasks.put(ac.getActionDescriptor(), res);
		}
		return res;
	}

	/**
	 * Allocates the part of batches (determined by ratio) to the task ac
	 * @param ac task to which we allocate the batches
	 * @param batches batches to allocate
	 * @param ratio ratio of batches to allocate
	 */
	private void allocatePartToTask(AcTaskTransport ac, List<TransportBatch> batches, double ratio)
	{
		long totcount = ListBatchTools.getTotalCount(batches);
		long alloc = Math.round(totcount * ratio);
		List<TransportBatch> allocated = new LinkedList<TransportBatch>();
		for (TransportBatch batch : batches) {
			if (alloc <= 0) break;
			long bsize = Math.min(alloc, batch.getCount());
			TransportBatch copied = ListBatchTools.copyTransportBatch(batch, bsize);
			allocated.add(copied);
			alloc -= bsize;
		}
		ac.addBatches(allocated);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder("TaskedPlan: ");
		sb.append(tasks.size());
		sb.append(" from: ");
		sb.append(ListBatchTools.getTransportCfpStart(tcfp));
		sb.append(" to: ");
		sb.append(ListBatchTools.getTransportCfpTarget(tcfp));
		sb.append(" ID: ");
		sb.append(tcfp.getRequestid());
		sb.append(" tasks \n");
		for (AcTaskTransport actt : tasks.values()) {
			sb.append(actt);
		}
		return sb.toString();
	}

	public Collection<AcTaskTransport> getTaskList() {
		return tasks.values();
	}

	public AcTaskTransport getTask(String actionDescriptor) {
		return tasks.get(actionDescriptor);
	}

  public AcPlanTemplate getTemplate() {
    return planTemplate;
  }

public TransportCfp getTcfp() {
	return tcfp;
}



}
