package across.util.skn.plan;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;

import across.util.skn.plan.across.AcTransport;
import across.util.skn.util.membership.SimpleFuzzyMembership;

/**
 * This class represents a specified action - it instantiates ActionType and specifies properties.
 * It also holds the information about its trustfulness. This class is a common base for objectives and plans as well.
 */
public abstract class Action {
	
	/** Objectives that are prerequisits of the action. */ 
	protected Collection<Action> prerequisites = new LinkedList<Action>();
//	/** Objectives that are results of the action. */
//	protected Collection<Action> achieves = new LinkedList<Action>();
	/** Actions/Objectives that are enabled when this objective is achieved. */
	protected Collection<Action> allows = new LinkedList<Action>();
	/** Contains the string identifier of the action type: e.g. "TRANSPORT", "STORAGE", ... */
	private String actionType; 
	
	
	
	/** Represents the trustfulness of this action as observed so far */
	SimpleFuzzyMembership trustfulness;

	/**
	 * @return Returns the prerequisites.
	 */
	public Collection<Action> getPrerequisites() {
		return prerequisites;
	}
	
	public Collection<Action> getAllows() {
		return allows;
	}

	/**
	 * @param prerequisites The prerequisites to set.
	 */
	public void setPrerequisites(Collection<Action> prerequisites) {
		this.prerequisites = prerequisites;
	}
	
	/** Returns true if the requiredAction is a prerequisite for this action start */
	public boolean requires(Action requiredAction)
	{
		for (Action pit : prerequisites) {
			if (pit.equivalentTo(requiredAction))
				return true;
		}
		return false;
	}
	
	/** Returns true if the allowsdAction is enabled by current action completion; i.e. this action is a prerequisite for allowedAction. */
	public boolean allows(Action allowedAction)
	{
		for (Action ait : allows) {
			if (ait.equivalentTo(allowedAction))
				return true;
		}
		return false;
	}
	
	public void addPrerequisite(Action act) {
		prerequisites.add(act);
	}
	
	public void addAllows(Action act) {
		allows.add(act);
	}
	
	private static Collection<Action> gatherActionLists(Collection<Collection<Action>> actionLists)
	{
		LinkedHashMap<String, Action> uniqueActions = new LinkedHashMap<String, Action>();
		for (Collection<across.util.skn.plan.Action> alist : actionLists) {
			for (Action action : alist) {
				String ad = action.getActionDescriptor();
				if ( ! uniqueActions.containsKey(ad))
				{
					// no similar action is in the list yet - add it to the list
					uniqueActions.put(ad,action);
				}
			}
		}
		return new LinkedList<Action>(uniqueActions.values());
	}
	
	/** Called by linking actions so that the object can update its links. Updates called object prerequisites list. */ 
	protected void prerequisiteActionsMerged(Action newAction, Collection<Action> oldActions)
	{
		actionsMerged(prerequisites, newAction, oldActions);
	}
	
	/** Called by linking actions so that the object can update its links. Updates called object allows list. */ 
	protected void allowsActionsMerged(Action newAction, Collection<Action> oldActions)
	{
		actionsMerged(allows, newAction, oldActions);
	}
	
	private void actionsMerged(Collection<Action> listName, Action newAction, Collection<Action> oldActions)
	{
		for (Action oaction : oldActions) {
			if (listName.remove(oaction) && ! listName.contains(newAction))
			{
				listName.add(newAction);
			}
		}
	}
	
	/** Returns true if the action triggers no subsequent actions. */
	public boolean isPlanStart()
	{
		if (prerequisites.isEmpty()) 
			return true;
		return false;
	}
	
	/** Returns true if the action has no prerequisites. */
	public boolean isPlanTarget()
	{
		if (allows.isEmpty()) 
			return true;
		return false;
	}
	
	/** Shall return true if the action is equivalent the one passed as an argument. */
	public boolean equivalentTo(Action otherAction) {
		if (! (otherAction instanceof AcTransport)) return false;
		if (getActionDescriptor().equalsIgnoreCase(otherAction.getActionDescriptor()))
			return true;
		return false;
	}

	/** Returns a string that dentifies action semantics - used to check equivalence */
	public abstract String getActionDescriptor();	
	
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("## Action: ");
		sb.append(getActionDescriptor());
		sb.append("\n");
		sb.append("Requires: ");
		for (Action act : prerequisites) {
			sb.append(act.getActionDescriptor());
			sb.append(" # ");
		}
		sb.append("\nAllows: ");
		for (Action act : allows) {
			sb.append(act.getActionDescriptor());
			sb.append(" # ");
		}
		sb.append("\n");
		return sb.toString();
	}

	/**
	 * @return Returns the string identifier of the action type: e.g. "TRANSPORT", "STORAGE", ...
	 */
	protected String getActionType() {
		return actionType;
	}

	/**
	 * @param type string identifier of the action type: e.g. "TRANSPORT", "STORAGE"
	 */
	public Action(String type) {
		super();
		actionType = type;
	}
	
	/**
	 * Creates a new Action that is a merge of actions passed in as a parameter.
	 * All old prerequisites, allows and achieves are merged into the new one, with no duplicities (verified by content, equivalentTo and actionDescriptor).
	 * @param actions Actions to merge into a single new Action
	 */
	public Action(List<Action> actions) {
		super();
		// set the type
		actionType = actions.get(0).getActionType();
		// merge the lists
		Collection<Collection<Action>> prerLists = new LinkedList<Collection<Action>>();
		Collection<Collection<Action>> allowsLists = new LinkedList<Collection<Action>>();
		for (Action sac : actions) {
			prerLists.add(sac.prerequisites);
			allowsLists.add(sac.allows);
		}
		prerequisites = gatherActionLists(prerLists);
		allows = gatherActionLists(allowsLists);
		// notify the connected actions about the merge
		// note that our prerequisites ALLOW us, etc...
		for (Action prer : prerequisites) {
			prer.allowsActionsMerged(this, actions);
		}
		for (Action all : allows) {
			all.prerequisiteActionsMerged(this, actions);
		}
	}
	
	/** Returns true if it doesn't depend on any other action outcome */
	public boolean hasPrerequisites() {
		if (prerequisites.isEmpty()) return false;		
		return true;
	}
	
	/** Returns true if it doesn't trigger any other actions. */
	public boolean hasAllows() {
		if (allows.isEmpty()) return false;		
		return true;
	}
}
