package across.util.skn;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;

import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

import across.agents.transporter.TransporterAgent;
import across.data.PublicParams;
import across.data.Resource;
import across.data.trust.AgentTrustfulness;
import across.data.trust.Trustfulness;
import across.data.trust.TrustworthinessObservation;
import across.data.trust.TrustworthinessObservationList;
import across.util.skn.gui.AlliancesDataModel;
import across.util.skn.gui.ServiceProvidersDataModel;
import across.util.skn.listeners.NewAgentRegisteredListener;
import across.util.skn.listeners.TrustReputationListener;
import across.util.skn.plan.Planner;
import across.util.skn.plan.ResourceKnowledge;
import across.util.skn.util.AgentSet;
import across.util.skn.util.membership.CrispMembership;
import across.util.skn.util.membership.iterative.IterativeTwoCategories;
import across.util.skn.util.membership.iterative.agent.AgentSigmaMinMaxMembership;
import aglobe.container.agent.Agent;
import aglobe.container.sysservice.directory.DirectoryRecord;
import aglobe.container.transport.Address;
import aglobe.container.transport.InvisibleContainerException;
import aglobe.ontology.Message;

/*
 * This class contains the knowledge base of the agent.
 */
public class CommunityKnowledge
{
    private DefaultMutableTreeNode alliRoot = new DefaultMutableTreeNode("Alliances");
    private DefaultTreeModel alliTreeModel = new DefaultTreeModel(alliRoot);

    private DefaultMutableTreeNode myAllianceRoot = new DefaultMutableTreeNode("My Alliance Members");
    private DefaultTreeModel myAllianceTreeModel = new DefaultTreeModel(myAllianceRoot);

    AlliancesDataModel alliDataModel = new AlliancesDataModel(alliTreeModel, alliRoot, new Boolean(false));
    AlliancesDataModel myAllianceDataModel = new AlliancesDataModel(myAllianceTreeModel, myAllianceRoot, new Boolean(true));

    private DefaultMutableTreeNode serviceProvidersRootNode = new DefaultMutableTreeNode("Services");
    private DefaultTreeModel serviceProvidersTreeModel = new DefaultTreeModel(serviceProvidersRootNode);
    ServiceProvidersDataModel serviceProvidersDataModel = new ServiceProvidersDataModel(serviceProvidersTreeModel,
           serviceProvidersRootNode);

    /** Contains the knowledge about known agents, includeing references to their alliances and containers - AgentKnowledge type indexed by name*/
    public Map<String,AgentKnowledge> agents = new HashMap<String,AgentKnowledge>();
    /** Contains the knowledge about known alliances, including references to their agents. - AllianceKnowledge type indexed by AllianceName*/
    public Map<String,AllianceKnowledge> alliances = new HashMap<String,AllianceKnowledge>();
    /** Contains the knowledge about visited containers, including references to known agents hosted by container. - ContainerKnowledge type indexed by containerAddress */
    public Map<Address,ContainerKnowledge> containers = new HashMap<Address,ContainerKnowledge>();
    /** Services map, containing agent references organized by keywords (services agents register) ServiceKnowledge type indexed by string service name*/
    public Map<String,ServiceKnowledge> services = new HashMap<String,ServiceKnowledge>();
    /** Contains all coalitions where the agent is currently involved. Indexed by stringified RequestID.*/
    protected Map<String,AcrossCoalitionKnowledge> coalitions = new HashMap<String,AcrossCoalitionKnowledge>();
    /** Keeps the knowledge about trusted agents. All agents are in this set, with membership equal to the trust in them. */
    protected AgentSet trustedAgents = new AgentSet();
    /** Contains agents with memebrship equal to their aggregate reputation - derived from other's observations received by this agent */
    protected AgentSet aggregateReputationAgents = new AgentSet();
    /** Keeps the reputation information received from other agents, in a form of sets of agents trusted by the others. */
    protected LinkedHashMap<String ,AgentSet> reputationAgents = new LinkedHashMap<String ,AgentSet>();

    /** References the planner with plan base. May be null if not used. */
    protected Planner planner;

    private String myAlliance;

    /** Holds references to listeners for the first appearance of the new agent */
    private List<NewAgentRegisteredListener> newAgentRegisteredListeners = new LinkedList<NewAgentRegisteredListener>();
    /** Holds the reference to listeners that receive trust and reputation values. */
    protected List<TrustReputationListener> trustReputationListeners = new LinkedList<TrustReputationListener>();


    /** Reference to the owner agent, used for proactive message sending */
    private Agent ownerAgent;
    private AgentKnowledge ownerAgentKnowledge;
    /** This structure represents the categories of trust. it is derived from self-trust.*/
    protected IterativeTwoCategories trustCategories = new IterativeTwoCategories();

    public CommunityKnowledge(Agent ownerAgent) {
        this.ownerAgent = ownerAgent;
    }

    /** Sends a message from the default (idle) task of the owner agent */
    protected void sendMessage(Message m) throws InvisibleContainerException
    {
        ownerAgent.sendMessage(m);
    }

    /** Submits a topic to the master conainer. Works only for TransporterAgent owners !*/
    protected void submitTopic(String topic, Object content, String reason)
    {
        if (ownerAgent instanceof TransporterAgent)
        {
            ((TransporterAgent)ownerAgent).gisShell.submitTopic(topic, content, reason);
        }
    }

    /**
     * Handles the reception of updated public page of an agent - either creation, appearance or info update.
     * @param page Agent Public page as recieved from yellow pages.
     */
    public synchronized void agentPageUpdated(Address agentAddress, PublicParams publicParams)
    {
        AgentKnowledge ag = agents.get(agentAddress.getName());
        if (null == ag)
        {
            // first time encounter
            ag = new AgentKnowledge(agentAddress, publicParams, this);
            agents.put(agentAddress.getName(), ag);
        }
        else
        {
            ag.publicInfoUpdated(agentAddress, publicParams, null);
        }
    }

    /**
     * Handles the reception of updated public page of an agent - either creation, appearance or info update.
     * @param page Agent Public page as recieved from yellow pages.
     */
    public synchronized void agentPageUpdated(Address agentAddress, PublicParams publicParams, String contName)
    {
        AgentKnowledge ag = agents.get(agentAddress.getName());
        if (null == ag)
        {
            // first time encounter
            ag = new AgentKnowledge(agentAddress, publicParams, contName, this);
            agents.put(agentAddress.getName(), ag);
        }
        else
        {
            ag.publicInfoUpdated(agentAddress, publicParams, contName);
        }
    }

	public AgentKnowledge agentFound(Address sender) {
		AgentKnowledge ag = agents.get(sender.getName());
		if(null == ag)
		{
			// new agent registered
			ag = new AgentKnowledge(sender, this);
			agents.put(sender.getName(),ag);
		}
		return ag;
	}

    /**
     * Handles the reception of updated agent information - creation of the service or after agent migration, when the agent
     * has registered with new container's directory service.
     * @param dr Agent Information as received from the directory
     */
    public synchronized void agentRegistered(DirectoryRecord dr)
    {
        AgentKnowledge ag = agents.get(dr.address.getName());
        if (null == ag)
        {
            // first time encounter
            ag = new AgentKnowledge(dr, this);
            agents.put(dr.address.getName(), ag);
            // call the registered callback
            for (NewAgentRegisteredListener listener : newAgentRegisteredListeners) {
                listener.handleNewAgentRegistered(ag);
            }
        }
        else
        {
            ag.directoryRegistration(dr);
        }
    }

    /**
     * Handles the reception updated agent info - agent has ceased to provide some service on
     * given container. It is either migrating somewhere else or will not provide this service any further.
     * has registered with new container's directory service.
     * @param dr Agent Information as received from the directory
     */
    public synchronized void agentDeRegistered(DirectoryRecord dr)
    {
        AgentKnowledge ag = agents.get(dr.address.getName());
        if (null != ag)
        {
            ag.directoryDeRegistration(dr);
        }
    }

    /**
     * Finds the container if exists, or creates new ContainerKnowledge record if not.
     * In any case, agent is added to the agents referenced by the container.
     * @param containerName Name of the container where we have found the agent.
     * @param knownAgentOnContainer Address of the agent. If null, no agent is added.
     * @return Reference to the existing or created ContainerKnowledge.
     */
    public synchronized ContainerKnowledge updateOrCreateContainer(Address containerAddress, AgentKnowledge knownAgentOnContainer)
    {
        ContainerKnowledge cont = containers.get(containerAddress);
        if(null == cont)
        {
            // first observation, create new record
            cont = new ContainerKnowledge(containerAddress, true);
            containers.put(containerAddress, cont);
        }
        else
        {
            // to avoid problems with delayed messages
            cont.setAccessible(true);
        }
        cont.addAgent(knownAgentOnContainer);
        return cont;
    }

    /** This function is called to store changes in container visibility. As all agents reference their container, it is
     * propagated to agents also.
     * @param containerRecords List of records of currently visible containers as received from GIS service in the topic
     */
    public synchronized void updateContainerVisibility(List<Address> containerRecords)
    {
        HashSet<Address> vc = new HashSet<Address>(containerRecords);
        for (Iterator<ContainerKnowledge> cit = containers.values().iterator(); cit.hasNext();)
        {
            ContainerKnowledge ckn = cit.next();
            if (vc.contains(ckn.address))
            {
                ckn.setAccessible(true);
            }
            else
            {
                ckn.setAccessible(false);
            }
        }
    };

    /**
     * Finds the service knowledge if exists, or creates new ServiceKnowledge record if not.
     * In any case, service provider reference is added to serviceProviders list of the ServiceKnowledge object.
     * @param serviceName Name of the service.
     * @param serviceProvider Reference to at least one agent providing the service
     * @return Reference to the existing or created SeeviceKnowledge.
     */
    public synchronized ServiceKnowledge updateOrCreateService(String serviceName, AgentKnowledge serviceProvider)
    {
        ServiceKnowledge sk = services.get(serviceName);
        if (null == sk)
        {
            // create the service record
            sk = new ServiceKnowledge(serviceName, serviceProvider);
            services.put(serviceName, sk);
        } else {
          sk.addMember(serviceProvider);
        }
        serviceProvidersDataModel.update(sk);
        return sk;
    }

    /**
     * Finds the alliance if exists, or creates new AllianceKnowledge record if not.
     * In any case, agent is added to the agents referenced by the alliance.
     * @param allianceName Address of the container where we have found the agent.
     * @param knownMember Address of the agent. If null, no agent is added.
     * @return Reference to the existing or created AllianceKnowledge. Null if the allianceName is null.
     */
    public synchronized AllianceKnowledge updateOrCreateAlliance(String allianceName, AgentKnowledge knownMember)
    {
        if (null != allianceName)
        {
            AllianceKnowledge alli = alliances.get(allianceName);
            if(null == alli)
            {
                // first observation, create new record
                alli = new AllianceKnowledge(allianceName);
                alliances.put(allianceName, alli);
            }
            // create new membership - Membership adds itself to the alliance alli
            alli.addMember(new CrispMembership<AgentKnowledge>(knownMember, alli));

            alliDataModel.update(alli);
            myAllianceDataModel.update(alli);

            return alli;
        }
        return null;
    }

    /**
     * Returns list of containers with location agents and no transporters from given alliance.
     * @param alliance alliance to avoid
     * @return List of containers, ContainerKnowledge objects.
     */
    public List<ContainerKnowledge> findUnoccupiedContainers(AllianceKnowledge alliance)
    {
        List<ContainerKnowledge> conts = new LinkedList<ContainerKnowledge>();
        // apss through the containers
        for (Iterator<ContainerKnowledge> it = containers.values().iterator(); it.hasNext();)
        {
            ContainerKnowledge record = it.next();
            // default is unoccupied
            conts.add(record);
            // apss through subject agents on container
            for (Iterator<AgentKnowledge> ag = record.surveyedAgentList.iterator(); ag.hasNext();)
            {
                AgentKnowledge agent = ag.next();
                if ( null != agent.alliance && alliance.equals(alliance))
                {
                    // there is someone on the container
                    conts.remove(record);
                    break;
                }
            }
        }
        return conts;
    }

    /**
     * Remove agents from the same coalition.
     * @param transporters Collection - collection of the agent addresses
     * @return Collection - new collection
     */
    public synchronized Collection<Address> removeAgentsFromSameCoalition(Collection<Address> transporters) {
      LinkedList<Address> retVal = new LinkedList<Address>();
      HashMap<String,List<Address>> coalitionMembers = new LinkedHashMap<String,List<Address>>();
      for (Iterator<Address> iter = transporters.iterator(); iter.hasNext(); ) {
        Address item =  iter.next();
        AgentKnowledge akn = getAgent(item.getName());
        if ((akn != null) && (akn.alliance != null)) {
          if (!coalitionMembers.containsKey(akn.alliance.getName())) {
            coalitionMembers.put(akn.alliance.getName(),new ArrayList<Address>());
          }
          coalitionMembers.get(akn.alliance.getName()).add(item);
        } else {
          retVal.add(item);
        }
      }
      // randomly select one from each alliance
      if (coalitionMembers.size() != 0) {
        Random rnd = new Random(System.currentTimeMillis());
        for (Map.Entry<String, List<Address>> elem : coalitionMembers.entrySet()) {
          retVal.add( elem.getValue().get( rnd.nextInt(elem.getValue().size()) ) );
        }
      }
      return retVal;
    }

    public synchronized Map getAlliances() {
      return alliances;
    }
    public synchronized Map getContainers() {
      return containers;
    }
    public Map getAgents() {
        return agents;
    }
    public Map getServices() {
        return services;
    }

    /** Adds new listrener for the first registration of the agent identified by name.
     *
     * @param list listener to add
     */
    public synchronized void subscribeNewAgentRegisteredListener(NewAgentRegisteredListener list) {
        if (!newAgentRegisteredListeners.contains(list))
        {
            newAgentRegisteredListeners.add(list);
        }
    }

    /**
     * Removes the listener for the firs registration of the agent identified by name.
     * @param list listener to remove
     */
    public synchronized void unsubscribeNewAgentRegisteredListener(NewAgentRegisteredListener list) {
        newAgentRegisteredListeners.remove(list);
    }

    /** Adds new listener for the trust/reputation updates.
    *
    * @param list listener to add
    */
   public synchronized void subscribeTrustReputationListeners(TrustReputationListener list) {
       if (!trustReputationListeners.contains(list))
       {
    	   trustReputationListeners.add(list);
       }
   }

    /**
     * Removed the listener for the trust/reputation updates.
     * @param list listener to remove
     */
    public synchronized void unsubscribeTrustReputationListeners(TrustReputationListener list) {
    	trustReputationListeners.remove(list);
    }

    /** Returns reference to Agent Knowledge, null if not available
     * @param name name of the agent
     * @return AgentKnowledge
     */
    public AgentKnowledge getAgent(String name) {
        return agents.get(name);
    }

    /** Returns reference to Alliance Knowledge, null if not available
     * @param name name of the Alliance
     * @return AllianceKnowledge object reference
     */
    public AllianceKnowledge getAlliance(String name)  {
        return alliances.get(name);
    }

    /** Returns reference to ServiceKnowledge, null if not available
     * @param name name of the Service
     * @return ServiceKnowledge object reference
     */
    public ServiceKnowledge getService(String name)  {
        return services.get(name);
    }

    /** Returns the AgentKnowledge relative to the owner. */
    public AgentKnowledge getOwnerAgentKnowledge() {
        if (null == ownerAgentKnowledge)
        {
            ownerAgentKnowledge = getAgent(ownerAgent.getAddress().getName());
        }
        return ownerAgentKnowledge;
    };

    /** Appends a new coalition to the coalition map.
     *  We check for existence, not to replace leader's leader
     * record by plain member record if it participates as a member also.
     * returns true if coalitionknowledge wasn't there before
     *  */
    public boolean appendCoalition(String coalkey, AcrossCoalitionKnowledge ckn) {
      boolean added = false;
      if (!coalitions.containsKey(coalkey)) {
        coalitions.put(coalkey, ckn);
        //System.out.println(getOwnerAgentKnowledge().getName() + " Coalition Added: " + coalkey + " with: " + ckn );
        if (ownerAgent instanceof across.agents.transporter.TransporterAgent) {
          ( (across.agents.transporter.TransporterAgent) ownerAgent).gui.addColaition(
              coalkey);
          added = true;
        }
      }
      return added;
    };

    /**
     * Removes the coalition from the map, typically after the termination of cooperation.
     *  */
    public void removeCoalition(String coalkey) {
        coalitions.remove(coalkey);
    };

    /** Returns coalition specified by requestid */
    public AcrossCoalitionKnowledge getCoalition(String requestid) {
        return coalitions.get(requestid);
    }

    public Class getTrustMembership() {
        String param = ownerAgent.getContainer().getProperty("membership"); // get class name from parameter
        if (param != null) {
          try {
            Class c = Class.forName(param); // try to convert param to Class
            return c;
          }
          catch (Exception e){};
        }
        return AgentSigmaMinMaxMembership.class; // default value
    }

    public DefaultTreeModel getAlliancesTreeModel() {
      return alliTreeModel;
    }
    public DefaultTreeModel getMyAllianceTreeModel() {
      return myAllianceTreeModel;
    }

    public DefaultTreeModel getServiceProvidersTreeModel() {
      return  serviceProvidersTreeModel;
    }

    public void setAlliance(String _myAlliance) {
      myAlliance = _myAlliance;
      myAllianceDataModel.setAlliance(myAlliance);
      myAllianceDataModel.update((AllianceKnowledge)getAlliances().get(myAlliance));
    }
    /** Use only in across-specific code */
	public Agent getOwnerAgent() {
		return ownerAgent;
	}

	/** Handles the reception of new batch of trustfulness values. */
	public void handleTrustfulnessValues(Address sender, Trustfulness trustfulness) {

		//ownerAgent.getLogger().severe(ownerAgent.getName() + " ###*### : reputation values received: from " + sender.getName()+ "\n"+ trustfulness);
		AgentKnowledge ak = getAgent(sender.getName());
		if (null != ak)
		{
			for (Iterator iter = trustfulness.getAgentTrustfulness().iterator(); iter.hasNext();) {
				AgentTrustfulness agtf = (AgentTrustfulness) iter.next();
				ak.reputationReceivedFromAgent(agtf);
			}
		}else
		{
			ownerAgent.getLogger().severe(ownerAgent.getName() + " : reputation values received from unknown agent: " + sender);
		}
	}

	/** Handles the reception of new batch of trustfulness values.
	 * Submits the new Trustfulness values to subscribed reputation receivers. */
	public void handleTrustworthinessObservation(Address sender, TrustworthinessObservationList twlist) {
		AgentKnowledge agent = getAgent(sender.getName());
		if (null != agent)
		{
//			 fill in this list to submit it later
			Trustfulness tfln = new Trustfulness();
			 List<TrustworthinessObservation> tol = twlist.getTrustworthinessObservation();
			 //getOwnerAgent().getLogger().severe("### Reputation observation received from: " + sender.getName() + " \n" + twlist);
			 for (TrustworthinessObservation obs : tol) {
				AgentKnowledge evaluated = getAgent(obs.getAgentName());
				if (evaluated != null) {
//					update the agent knowledge with observations
//					TODO - now we update only the coalition leader - but we shall probably update the knowledge of all members of the coalition
					agent.reputationObservationReceivedFromAgent(evaluated, obs.getTrust());
//					 update the aggregateReputation of the evaluated agent
					//getOwnerAgent().getLogger().severe("### AggRep for: " + evaluated.getName() + " OLD: " + evaluated.getAggregateReputation());
					evaluated.getAggregateReputation().newSample(obs.getTrust());
					//getOwnerAgent().getLogger().severe("### AggRep for: " + evaluated.getName() + " NEW: " + evaluated.getAggregateReputation());
					// include into the data sent out once we have enough data
					if (evaluated.getAggregateReputation().hasEnoughData())
					{
						tfln.getAgentTrustfulness().add( evaluated.getAggregateReputation().getAgentTrustfulness(evaluated.getName(), 1) );
					}
				}
				else
				{
					getOwnerAgent().getLogger().severe("Reputation observation received for unknown agent: " + obs.getAgentName() +" \n\n"+ sender + " \n" + twlist);
				}
			}
			if (tfln.getAgentTrustfulness().size() > 0) {
				// distribute the updated data to reputation-subscribed agents
				for (TrustReputationListener list : trustReputationListeners) {
					list.handleNewReputationValues(tfln);
				}
			}
		}else
		{
			ownerAgent.getLogger().severe(ownerAgent.getName() + " : reputation observation received from unknown agent: " + sender);
		}
	}

	public Planner getPlanner() {
		return planner;
	}

	public void setPlanner(Planner planner) {
		this.planner = planner;
	}

	/** Used to update the status of particular resource.
	 * Just a helper function if ever planner or plan base are null.
	 *  */
	public ResourceKnowledge updateOrCreateResource(AgentKnowledge ownerAgent, Resource res, boolean isAvailable)
        {
		if (planner != null && planner.getPlanBase() != null)
			return planner.getPlanBase().updateOrCreateResource(ownerAgent, res, isAvailable);
		return null;
	}
}
