package across.util.skn;

import static across.util.skn.CoalitionKnowledge.Strategy.FLAT;
import static across.util.skn.CoalitionKnowledge.Strategy.TRUST_PROPORTIONAL;
import static across.util.skn.CoalitionKnowledge.Strategy.TRUST_UNC_PROPORTIONAL;
import static across.util.skn.CoalitionKnowledge.Strategy.UNC_PROPORTIONAL;
import across.data.trust.Trustfulness;
import across.util.skn.listeners.TrustReputationListener;
import across.util.skn.util.AgentSet;
import across.util.skn.util.membership.Membership;

/**
 * Represents agents' knowledge about a coalition in which it participates.
 * Contains references other members and the infromation about them.
 * This object is defined at the beginning of the coalition cooperation and exists until the goals are achieved or
 * the cooperation is terminated.
 *  */
public abstract class CoalitionKnowledge<CoalitionDefinition> extends AgentSet
{
    /** maximum expected cooperation payoff */
    double maxPayoff;
    /** Minimum expected cooperation payoff or maximum loss. (Loss is negative payoff.)*/
    double minPayoff;
    /** Actual payoff so far - may vary during coalition lifecycle... */
    double actualPayoff;
    /** Holds the reference to the owning structure. Used to resolve agent names */
    protected CommunityKnowledge caller;

    /** Defines various possible trust distribution strategies: FLAT - same value for all agents,
     * Proportional - trust is proportional to apriori trust.
     */
    public enum Strategy{FLAT, TRUST_PROPORTIONAL, UNC_PROPORTIONAL, TRUST_UNC_PROPORTIONAL};

    /** Defines a strategy how to distribute the trustfulness between agents */
    Strategy trustUpdateStrategy = Strategy.FLAT;

    /** Store the cooperation request in this variable. */
    CoalitionDefinition requestObject;

    /**
     * Calculates the coalition trustfulness from the payoff and payoff-trustfulness function
     * @return Coalition trustfulness, bounded to interval [0,1]
     */
    public abstract double getTrustfulness();

    /**
     * This method generates trust update event for all coalition members. It distributes the coalition trustfulness
     * using the strategy specified for this coalition.
     */
    public void updateMemberTrust()
    {
        double trustfulness = getTrustfulness();
        long memCount = getAllMembersCount();
        switch (trustUpdateStrategy)
        {
            case FLAT:
                {
                    for (Membership<AgentKnowledge> membership : members) {
                        membership.getMember().updateTrustWithNewEvent(trustfulness);
                    }
                    break;
                }
            case TRUST_PROPORTIONAL:
                {
                    if ( 0 < memCount)
                    {
                        double avgTrustCenter = 0;
                        for (Membership<AgentKnowledge> membership : members)
                        {
                            avgTrustCenter += membership.getMember().trust.getMembershipFunctionCenter();
                        }
                        avgTrustCenter /= memCount;
                        for (Membership<AgentKnowledge> membership : members) {
                            membership.getMember().updateTrustWithNewEvent(trustfulness*membership.getMember().trust.getMembershipFunctionCenter()/avgTrustCenter);
                        }
                    }
                    break;
                }
            case UNC_PROPORTIONAL:
                {
                    if ( 0 < memCount)
                    {
                        double avgTrustUnc = 0;
                        for (Membership<AgentKnowledge> membership : members)
                        {
                            avgTrustUnc += membership.getMember().trust.getMembershipFunctionUncertainity();
                        }
                        avgTrustUnc /= members.size();
                        for (Membership<AgentKnowledge> membership : members) {
                            membership.getMember().updateTrustWithNewEvent(trustfulness*membership.getMember().trust.getMembershipFunctionUncertainity()/avgTrustUnc);
                        }
                    }
                    break;
                }
            case TRUST_UNC_PROPORTIONAL:
            {
                if ( 0 < memCount)
                {
                    double avgTrustUnc = 0;
                    double avgTrustCenter = 0;
                    for (Membership<AgentKnowledge> membership : members)
                    {
                        avgTrustUnc += membership.getMember().trust.getMembershipFunctionUncertainity();
                        avgTrustCenter += membership.getMember().trust.getMembershipFunctionCenter();
                    }
                    avgTrustUnc /= memCount;
                    avgTrustCenter /= memCount;
                    for (Membership<AgentKnowledge> membership : members) {
                        double tct = trustfulness*membership.getMember().trust.getMembershipFunctionCenter()/avgTrustCenter;
                        double tuc = trustfulness*membership.getMember().trust.getMembershipFunctionUncertainity()/avgTrustUnc;
                        membership.getMember().updateTrustWithNewEvent((tct + tuc)/2);
                    }
                }
                break;
            }
        }
        // submit the new trust values to listeners
        Trustfulness trfln = new Trustfulness();
        for (Membership<AgentKnowledge> membership : members) {
            AgentKnowledge ak = membership.getMember();
            trfln.getAgentTrustfulness().add(ak.getTrust().getAgentTrustfulness(ak.getName(),1));
        }
        for (TrustReputationListener list : caller.trustReputationListeners) {
        	list.handleNewTrustValues(trfln);	
		}
    };

    public CoalitionKnowledge(CoalitionDefinition requestObject, CoalitionKnowledge.Strategy strategy, CommunityKnowledge caller) {
        this.requestObject = requestObject;
        trustUpdateStrategy = strategy;
        this.caller = caller;
    }
}
