package across.util.pathfinding;

import java.util.LinkedList;

import across.agents.driver.data.NodeValue;
import across.agents.driver.data.RoadInfo;
import across.agents.driver.data.Route;
import across.agents.driver.data.RouteInfo;
import across.data.Waypoint;

/**
 * Pathfinding algorithm - using Dijkstra algorithm.
 * @author Eduard Semsch
 *
 */
public class DijkstraPathfinding implements Pathfinding {

	 /** List of map locations - their indexes are the indexes in the roads field */
	 private LinkedList<String> indexedLocations;
	 
	 /** Info for the map roads */
	 private RoadInfo[][] roads;
	 
	 public DijkstraPathfinding(LinkedList<String> indexedLocations, RoadInfo[][] roads) {
		this.indexedLocations = indexedLocations;
		this.roads = roads;
	 }	

	// FIXME - does not look very good!
	  /**
	   * Search the graph of routes using the Dijkstra's algorithm
	   * http://en.wikipedia.org/wiki/Dijkstra's_algorithm
	   * @param from String - starting node
	   * @param to String - target node
	   * @param weightTime double
	   * @param weightPrice double
	   * @return Route Optimal route between the nodes, null if the route doesn't exist
	   */
	  public Route planRoute(String from, String to, double weightTime, double weightPrice){
	    LinkedList<NodeValue> open = new LinkedList<NodeValue>();      // set of all nodes
	    LinkedList<NodeValue> closed = new LinkedList<NodeValue>();    // empty on the begining

	    Route route = new Route();

	    int fromIndex = indexedLocations.indexOf(from);
	    int toIndex = indexedLocations.indexOf(to);

	    // if the nodes don't exist quit
	    if ( (fromIndex == -1) || (toIndex == -1) ) {
	      return null;
	    }

	    // Fill the open list with individual nodes
	    for(int i=0; i<indexedLocations.size(); i++){
	      NodeValue nodeVal = new NodeValue(i);
	      if (i == fromIndex) {
	    	  nodeVal.setValue(0);  // Set the value of the from node to 0
	      }
	      open.add(nodeVal);
	    }

	    while (open.size()>0){  // while the open set is not empty
	    
	      // search for the cheapest node in the open list
	      NodeValue minValNode = (NodeValue) open.getFirst();
	      for (NodeValue node : open) {
		      if ( (minValNode.getValue() == -1) || ( (node.getValue() >= 0) && (node.getValue() < minValNode.getValue()) ) ) {
	        	minValNode = node;
	        }
	      }

	      // verify that the target node is reachable, if unreachable quit      
	      if (minValNode.getValue() == -1) {
	    	  return null; // unreachable
	      }

	      // if we are at the target node - compose the route and return it
	      if (minValNode.getNodeIndex() == toIndex) {
	        // count price and time, compose the route
	    	LinkedList<Waypoint> wplist = new LinkedList<Waypoint>();
	    	wplist.add(new Waypoint(indexedLocations.get(toIndex)));
	    	
	        NodeValue node = minValNode;
	        double time = 0;
	        double price = 0;
	        while(node.previous != -1){
	          RoadInfo arc = roads[node.getNodeIndex()][node.getPrevious()];
	          time+=arc.time;
	          price+=arc.price;
	          // if this is not the first node
	          if (!((String)indexedLocations.get(node.previous)).equalsIgnoreCase(from)) {
	        	  wplist.addFirst(new Waypoint((String)indexedLocations.get(node.previous)));
	          }
	          // look for node whose index is same as node.previous
	          for (NodeValue value : closed) {
	            if (value.getNodeIndex() == node.getPrevious()) node = value;
	          }
	        }
	        route.addWaypoints(wplist);
	        RouteInfo routeInfo = new RouteInfo(time, price);
	        route.setRouteInfo(routeInfo);
	        return route;
	      }

	      // put this node from the open set into the closed set
	      open.remove(minValNode);
	      closed.add(minValNode);
	      // System.out.println("Expanding node: " + nodeNames.get(minValNode.getNodeIndex()));

	      // relax the outgoing arcs
	      for(int i=0; i<indexedLocations.size(); i++) {
	        RoadInfo arc = roads[minValNode.getNodeIndex()][i];
	        if (arc != null && arc.passable){  // arc form the cheapest node, find the appropriate end poind and verify the value
	          // find end point in the open set
	          NodeValue endPoint = null;
	          for (NodeValue node : open) {
	        	 if (node.getNodeIndex() == i) {
	        		 endPoint = node;
	        		 break;
	             }
	          }
	          if (endPoint == null) {
	        	  continue; // end point of this arc wasn't found in the open set - is already in the closed
	          }

	          // count value of this arc
	          double arcValue = weightPrice*arc.price + weightTime*arc.time;

	          // new value is lower than the current or value of the arc wasn't set yet
	          double currentValue = endPoint.getValue();
	          if ( (endPoint.getValue()==-1) || (currentValue > (minValNode.getValue() + arcValue)) ) {
	            endPoint.setValue(minValNode.getValue() + arcValue);  // set new value
	            endPoint.setPrevious(minValNode.getNodeIndex());
	            // logger.info("Setting value to: " + nodeNames.get(endPoint.getNodeIndex()) + " value: " + (minValNode.getValue() + arcValue));
	            continue;
	          }
	        } // end of one arc relaxation
	      } // end of checking the outgoing arcs
	    } // end of while cycle
	    // (this point shouldn't be reached)
	    return null;
	  }
	  
	
}
