package across.simulation.weather;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import across.data.Disaster;
import across.data.DisasterArea;
import across.data.DisasterAreaConfig;
import across.data.DisasterConfig;
import across.data.Effect;
import across.data.Weather;
import across.simulation.constants.AcrossDataConstants;
import across.simulation.constants.AcrossTopicConstants;
import across.simulation.constants.AcrossWeatherConstants;
import across.util.GoodsConstants;
import across.util.XMLtools;
import aglobe.container.agent.Agent;
import aglobe.container.transport.Address;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobe.service.gis.server.GISServerService;
import aglobe.service.gis.server.GISTopicServerListener;
import aglobex.simulation.global.ServerServerTopicConstants;
import aglobex.simulation.ontology.Configuration;
import aglobex.simulation.ontology.Param;

/**
 * This agent simulates the weather in the simulation. It sends topics about the weather.
 * 
 */
public class WeatherAgent extends Agent implements MessageConstants, GISTopicServerListener, ServerServerTopicConstants {

	/** Type of this agent */
	public static final String TYPE = "WeatherAgent";
	
	/** Weather will be sent through updates via gisServices - masterShell will convey the messages to containers.*/
	private GISServerService.Shell gisMaster;

	/** Containers receiving weather information */
	private LinkedHashMap<String,List<String>> subscribedContainers = new LinkedHashMap<String,List<String>>();

	/** Contents config files */
	private DisasterAreaConfig disasterArea;
	
	/** Configuration of the disaster */
	private DisasterConfig disasterConfig;
	
	/** Simulation time */
	private int time = 0;
	
	/** This agent's gui */
	public WeatherAgentGUI gui = new WeatherAgentGUI(this);
	
	/** Scenario configuration */
	private Configuration conf;
	
	/** Last time the weather was updated */
	private long lastTimeWeatherUpdated = -1;
	
	/** Period of weather updates */
	private long updatePeriod = 1000;

	/**
	 * Initialization. Happens in these steps:
	 * <li> Super initializator. </li>
	 * <li> Gis master shell. </li>
	 * <li> Subscribing of topics: TOPIC_SIMULATION_TIME_UPDATE and "weather" </li>
	 * <li> GUI initialization. </li>
	 * <li> Requests configuration </li>
	 */
	public void init(AgentInfo a, int initState){

		super.init(a, initState);
		
		// get the gisMaster Shell -
		gisMaster = (GISServerService.Shell) getContainer().getServiceManager().getService(this,GISServerService.SERVICENAME);
		if (null != gisMaster ) {
			gisMaster.subscribeTopic(TOPIC_SIMULATION_TIME_UPDATE,this);
            gisMaster.subscribeTopic(AcrossTopicConstants.TOPIC_WEATHER_UPDATE_REQUEST,this);
            gisMaster.subscribeTopic(TOPIC_CONFIGURATION, this);
            gisMaster.subscribeTopic(TOPIC_CONF_OBJECT,	this);
    		gisMaster.sendTopicToLocal(TOPIC_CONFIGURATION_REQUEST, null, getName());
		}
		else {
			logger.severe("GISServer service not found!!!");
			this.stop();
			return;
		}

		gui.setTitle(a.getReadableName());
		gui.setBounds(0, 0, 400, 280);
		gui.setVisible(true);
		gui.log("LOGIN:");
		
	}
	
	/**
	 * Login location agents and sort to disaster area according to AREA_FILE
	 * 
	 * @param topic type of the information (Ex. TYPE)
	 * @param remoteContainerName name subscribed remote container
	 * @param remoteContainerAddress address subscribed remote container
	 * */
	public void handleLoginTopic(String topic, String remoteContainerName, Address remoteContainerAddress) {
      
	}

	/**
	 * Logout containers (location agents)
	 * 
	 * @param topic type of the information (Ex. TYPE)
	 * @param remoteContainerName name subscribed remote container
	 * */
	
	public void handleLogoutTopic(String topic, String remoteContainerName) {
		subscribedContainers.remove(remoteContainerName);
	}

	/** 
	 * On a time tick, send weather to location agents to start production and trade. 
	 * 
	 * @param topic type of the information
	 * @param reason String
	 * @param remoteContainerName
	 * @param remoteClientAddress
	 * */
	public void handleTopic(String topic, Object content, String reason, String remoteContainerName, Address remoteClientAddress) {
		if(TOPIC_CONFIGURATION.equalsIgnoreCase(topic)) {
			conf = (Configuration) content;
			for (Param par : (List<Param>) conf.getParam()) {
				if(par.getName().equals(AcrossDataConstants.DISASTER_AREA)) {
					gisMaster.sendTopicToLocal(TOPIC_CONF_OBJECT_REQUEST, new String[] {par.getValue(),"across.data.DisasterAreaConfig"}, getName());
				} else if (par.getName().equals(AcrossDataConstants.DISASTER_CONFIG)) {
					gisMaster.sendTopicToLocal(TOPIC_CONF_OBJECT_REQUEST, new String[] {par.getValue(),"across.data.DisasterConfig"}, getName());
				}
			}
		} else if(TOPIC_SIMULATION_TIME_UPDATE.equalsIgnoreCase(topic)) {
			long time = Long.parseLong((String) content);
			if(lastTimeWeatherUpdated==-1 || ((time-lastTimeWeatherUpdated) >= updatePeriod)) {
				lastTimeWeatherUpdated = time;
				nextTurn();
			}
		} else if(TOPIC_CONF_OBJECT.equalsIgnoreCase(topic)) {
			if(content instanceof DisasterConfig) {
				disasterConfig = (DisasterConfig) content;
			} else if (content instanceof DisasterAreaConfig) {
				disasterArea = (DisasterAreaConfig) content;
			}
		} else if (topic.equals(AcrossTopicConstants.TOPIC_WEATHER_UPDATE_REQUEST)) {
	        gui.log("Container name: " + remoteContainerName);
	        if (!subscribedContainers.containsKey(remoteContainerName)) {
	          List<String> areaName = new LinkedList<String> ();
	          for (Iterator it = disasterArea.getDisasterArea().iterator();
	               it.hasNext(); ) {
	            DisasterArea area = (DisasterArea) it.next();
	            List areaContainers = area.getContainerName();
	            for (Object object : areaContainers) {
	              String contName = (String) object;
	              if (contName.equalsIgnoreCase(remoteContainerName)) {
	                areaName.add(area.getName());
	              }
	            }
	          }
	          subscribedContainers.put(remoteContainerName, areaName);
	        }
	      }
	}

	/**
	 * Setting restriction of commodity production and visibility change according to disaster. 
	 * Submit the weather ontology to client container.
	 */
	private void nextTurn() {
		// if not configured yet do not send weather
		if(disasterConfig==null || disasterArea==null) {
			return;
		}
		//actualization of parameters
		time++;
		gui.log("\n******"+"\nTime: " + time + "\n******");

		for (Map.Entry<String,List<String>> cName : subscribedContainers.entrySet()) {
			List<String> areaName = cName.getValue();
			Weather weather = new Weather();

			// commodity constants - WATER, GRAIN, FISH, SHEEP
			// production 100%
			weather.getParam().add(XMLtools.makeParam(GoodsConstants.FISH, "1"));
			weather.getParam().add(XMLtools.makeParam(GoodsConstants.GRAIN, "1"));
			weather.getParam().add(XMLtools.makeParam(GoodsConstants.SHEEP, "1"));
			weather.getParam().add(XMLtools.makeParam(GoodsConstants.WATER, "1"));

			// reading  the config disaster  file with restrictions for locations
			for (Disaster disaster : disasterConfig.getDisaster()) {
				for (String disAr : disaster.getAreaName()) {
					for (String area : areaName) {
						// the time range of the disaster
						int start = (int) Math.floor(Double.parseDouble(disaster.getStart()));
						int end = (int) Math.floor(Double.parseDouble(disaster.getEnd()));
						if (area.equalsIgnoreCase(disAr)
								&&((time%start)<=end))	{
							gui.log("Disaster "+disaster.getName().toUpperCase()+" for location "+cName.getKey());

							for (Effect paramu : disaster.getEffect()) {
								for (across.data.Param pa : paramu.getParam()) {
									for (across.data.Param paW : weather.getParam()) {
										//the restriction value for the actual commodity is the minimal value 
										if((paW.getName().equalsIgnoreCase(pa.getName()))&&
												(Double.parseDouble(pa.getValue())<Double.parseDouble(paW.getValue()))){
											paW.setValue(pa.getValue());
										}
									}
								}
							}
							// send visibility change to locations
							for (DisasterArea dArea : disasterArea.getDisasterArea()) {
								if (dArea.getName().equals(disAr)) {
									for (String dAr : dArea.getContainerName()) {
										// send disaster information to visibility server
										gisMaster.sendTopicToLocal(AcrossTopicConstants.TOPIC_DISASTER_VISIBILITY, new DisasterVisibilityParams((String)dAr, disaster.getVisibilityRadius(), disaster.getVisibilityFailurePct()));
										// send disaster information to affected container
										if (time == Double.parseDouble(disaster.getStart())) {
											gisMaster.sendTopic((String)dAr, AcrossTopicConstants.TOPIC_DISASTER_VISIBILITY, (String)disaster.getName().toUpperCase(), AcrossWeatherConstants.WEATHER_DISASTER_START);
										} else if (time == Double.parseDouble(disaster.getEnd())) {
											gisMaster.sendTopic((String)dAr, AcrossTopicConstants.TOPIC_DISASTER_VISIBILITY, (String)disaster.getName().toUpperCase(), AcrossWeatherConstants.WEATHER_DISASTER_END);
										} else {
											gisMaster.sendTopic((String)dAr, AcrossTopicConstants.TOPIC_DISASTER_VISIBILITY, (String)disaster.getName().toUpperCase(), AcrossWeatherConstants.WEATHER_DISASTER_IS);
										}
									}
								}
							}
						}
					}
				}
			}

			// submit the weather ontology to client container
			gisMaster.sendTopic(cName.getKey(), AcrossTopicConstants.TOPIC_WEATHER_UPDATE, weather );

			// write up restriction for loacation to GUI
			gui.log("Sent restriction for location " + cName.getKey());
			for (Object wob : weather.getParam()) {
				across.data.Param wp = (across.data.Param) wob;
				gui.log("\t"+wp.getName()+" : "+wp.getValue());
			}
		}
	}

	public void handleIncomingMessage(Message msg) {
		
	}

	/**
	 * Stop this agent.
	 *
	 */
	public void killPressed() {
		stop();
	}

	// TODO - doc
	/**
	 * Parameters of a disaster.
	 * @author Eduard Semsch
	 *
	 */
	public static class DisasterVisibilityParams {
		String AffectedContainer;
		double VisibilityRadius;
		double VisibilityFailurePct;

		public DisasterVisibilityParams(String affectedContainer, double visibilityRadius, double visibilityFailurePct) {
			super();
			AffectedContainer = affectedContainer;
			VisibilityRadius = visibilityRadius;
			VisibilityFailurePct = visibilityFailurePct;
		}

		public String getAffectedContainer() {
			return AffectedContainer;
		}
		public double getVisibilityFailurePct() {
			return VisibilityFailurePct;
		}
		public double getVisibilityRadius() {
			return VisibilityRadius;
		}
	}

}

