package across.agents.terrorists.driver;

import across.agents.emergency.centre.EmergencyAgent;
import across.agents.terrorists.driver.util.ObstacleAdder;
import across.data.simulation.SimulationObject;
import across.data.simulation.SimulationObjectDescription;
import across.data.simulation.SimulationObjectLocation;
import across.simulation.constants.AcrossControlConstants;
import across.simulation.constants.AcrossDataConstants;
import across.simulation.constants.AcrossObjectsTypes;
import across.simulation.constants.AcrossTopicConstants;
import across.visio.oldvisio.VisioConnectionAgent;
import aglobe.container.agent.Agent;
import aglobe.container.sysservice.directory.DirectoryListener;
import aglobe.container.sysservice.directory.DirectoryRecord;
import aglobe.container.sysservice.directory.DirectoryService;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.AglobeParam;
import aglobe.ontology.Message;
import aglobe.service.gis.client.GISClientService;
import aglobe.service.gis.client.GISTopicListener;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.SensorDescription;
import aglobex.simulation.ontology.entity.EntityDescriptor;
import aglobex.simulation.sensor.UniversalSensorAgent;

/**
 * This driver agent roams around the simulation map and creates obstacles on the roads.
 * @author Eduard Semsch
 *
 */
public class TerroristDriverAgent extends Agent implements ClientServerTopicConstants, GISTopicListener, DirectoryListener {

	/** GIS shell */
	protected GISClientService.Shell gisShell;
	
	/** Directory service shell. */
	protected DirectoryService.Shell dirShell;
	
	/** Speed of vehicle in km/h. */
	protected double velocity;         
	  
	/** Consumption in litres/100 km */
	protected double consumption;   
	
	/** Class handling this agent's movement */
	protected TerDriverMovement myMovement = new TerDriverMovement(this);
	
	/** Class for determining the frequency of adding obstacles */ 
	private ObstacleAdder obstacleAdder = new ObstacleAdder(1000,20000,50);
	
	/** Counter of the obstacles added by this agent to the simulation */
	private int obstacleCounter = 0;
	
	@Override
	public void init(AgentInfo ai, int initState) {
		gisShell = (GISClientService.Shell) this.getContainer().getServiceManager().getService(this, GISClientService.SERVICENAME);
		dirShell = (DirectoryService.Shell) this.getContainer().getServiceManager().getService(this, DirectoryService.SERVICENAME);
		if(gisShell!=null) {
			gisShell.subscribeTopic(TOPIC_ENTITY_CONNECTION, this);
			gisShell.subscribeTopic(TOPIC_ENTITY_CONTROL, this);
			gisShell.subscribeTopic(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND, this);
			gisShell.subscribeTopic(UniversalSensorAgent.TOPIC_SENSORY_DATA,this);
		} else {
			logSevere("Unable to locate GIS!");
			this.stop();
		}
		dirShell.subscribe(this, EmergencyAgent.TYPE);
	}

	@Override
	protected void handleIncomingMessage(Message m) {
		logWarning("Unexpected message");
	}

	/**
	 * Handles these simulation topics:
	 * <li> TOPIC_ENTITY_CONNECTION - configuration </li>
	 * <li> TOPIC_ENTITY_CONTROL - movement </li>
	 * <li> TOPIC_SIMULATION_TIME_UPDATE_1_SECOND - simulation time </li> 
	 * @param topic
	 * @param content
	 * @param reason
	 */
	public void handleTopic(String topic, Object content, String reason) {
		if(topic.equalsIgnoreCase(TOPIC_ENTITY_CONNECTION)) {
			if(content instanceof EntityDescriptor) {
				handleConfiguration((EntityDescriptor) content);
			}
		} else if (topic.equalsIgnoreCase(TOPIC_ENTITY_CONTROL)) {
			myMovement.handleMovementTopic(content, reason);
			if(!myMovement.isOnRoute()) {
				myMovement.goToRandomLocation();
			}
		} else if (topic.equalsIgnoreCase(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND)) {
			obstacleAdder.simulationTick();
			if(obstacleAdder.isReadyToAddObstacle()) {
				if(addObstacle()) {
					obstacleAdder.obstacleAdded();
				}
			}
			if(!myMovement.isOnRoute()) {
				myMovement.goToRandomLocation();
			}
		}
	}

	/**
	 * Handles incoming configuration from the server.
	 * @param ed
	 */
	protected void handleConfiguration(EntityDescriptor ed) {
  	    this.velocity = Double.parseDouble(ed.confParamsString.get(AcrossDataConstants.VEHICLE_VELOCITY));
  	    this.consumption = Double.parseDouble(ed.confParamsString.get(AcrossDataConstants.VEHICLE_CONSUMPTION));

  	    myMovement.handleConfiguration(ed, velocity, consumption);
  	  
  	    // sensor registration
        SensorDescription sd = new SensorDescription();
        double range = ed.typeDescriptor.userParamsDouble.get(AcrossDataConstants.DRIVER_SENSOR_RANGE);
        AglobeParam rangeParam = new AglobeParam();
        rangeParam.setValue(Double.toString(range));
        sd.setSensorType("spherical");
        sd.getAglobeParam().add(rangeParam);
        gisShell.submitTopicToServer(UniversalSensorAgent.TOPIC_REGISTER_SENSOR, sd);
        
        myMovement.goToRandomLocation();
        
        String visioInfo = "<b>" + this.getName();
        gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_INFO, visioInfo);
	}
	
	/**
	 * Adds obstacle to the simulation.
	 *
	 */
	protected boolean addObstacle() {
		SimulationObject so = new SimulationObject();
		double[] position = myMovement.getPosition();
		if(position==null) {
			return false;
		}
		so.setSimulationClass("across.simulation.objects.ObjectBehaviour");
		
		SimulationObjectDescription sod = new SimulationObjectDescription();
		
		sod.setX(position[0]);
		sod.setY(position[1]);
		sod.setZ(new Double(0));
		
		sod.setName("BusCrash"+getName().substring("Terrorist".length())+obstacleCounter);
		obstacleCounter++;
		
		sod.setObjectType(AcrossObjectsTypes.ROAD_OBSTACLE);
		sod.setExtentOfFire(0);
		sod.setInsurgencyLevel(50);
		sod.setNumberOfWounded(10);
		sod.setConstructionDamage(0);
		so.setSimulationObjectDescription(sod);
		
		SimulationObjectLocation sol = new SimulationObjectLocation();
		sol.setStartLocation(myMovement.actualLoc);
		if(myMovement.headingTo==null) {
			sol.setEndLocation(myMovement.actualLoc);	
		} else {
			sol.setEndLocation(myMovement.headingTo);
		}
		
		sod.setLocation(sol);
		
		gisShell.submitTopicToServer(AcrossTopicConstants.TOPIC_SIMULATION_OBJECT, so, AcrossControlConstants.CREATE_SIMULATION_OBJECT);
		gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_VISIO_ACTION, ""+VisioConnectionAgent.ACTION_STANDIN_CREATED, this.getAddress().toString());
		
		return true;
	}

	/* -------------------- DIRECTORY LISTENER ---------------- */
	public void handleDeregister(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}

	public void handleInvisible(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}

	public void handleNewRegister(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}

	public void handleVisible(String containerName, DirectoryRecord[] records, String matchingFilter) {
	}
	
}
