package across.agents.location.task;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import across.agents.location.LocationAgent;
import across.data.Batch;
import across.data.ItemCoverage;
import across.data.Proposal;
import across.data.RequestList;
import across.visio.VisualAgent;
import aglobe.container.transport.Address;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobe.service.gis.client.GISClientService;
import aglobe.util.Logger;
import aglobex.simulation.protocol.cnp.CNPInitiatorTask;

/**
 *This task is used to acquire goods from other Location agents using CNP protocol.
 */
public class LocAcquireGoodsTask extends CNPInitiatorTask {
	
	/** What is to be aquired */
    RequestList cfpContent;
    
    /** Map of proposals selected in evaluateReplies() */
    HashMap<Address,Proposal> selectedProposalMap = new HashMap<Address,Proposal>(); 
    
    protected GISClientService.Shell gisShell;

    // TODO - conf
    public static final int TIMEOUT = 20000;

    public LocAcquireGoodsTask(LocationAgent cu, Collection<Address> participants, RequestList cfpContent, int timeout) {
        super(cu, participants, cfpContent, timeout, null, false);
        this.cfpContent = cfpContent;

         // send a copy to topic  content and participants
         gisShell = cu.gisShell;
         List data = new LinkedList();
         data.add(cfpContent);
         data.add(participants);
         data.add(getAddress().getName());
         gisShell.submitTopic(VisualAgent.TOPIC_LOCATION_CFP_AND_PARTICIPANTS, data);
         
         start();
    }

    public LocAcquireGoodsTask(LocationAgent cu, Collection<Address> participants, RequestList cfpContent, int timeout, String reason) {
        super(cu, participants, cfpContent, timeout, reason, false);
        this.cfpContent = cfpContent;

        // send a copy to topic  content and participants
         gisShell = cu.gisShell;
         List data = new LinkedList();
         data.add(cfpContent);
         data.add(participants);
         data.add(getAddress().getName());
         gisShell.submitTopic(VisualAgent.TOPIC_LOCATION_CFP_AND_PARTICIPANTS, data);

         start();
    }

    /**
     * Evaluates the replies and selects the best proposition(s). String content only.
     * @return list of best propositions.
     */
    @Override
    protected List<Address> evaluateReplies() {
      // Proposals are sorted by their utility function and stored into the SortedSet - stored object is ProposalWithUtil
      // (implements Comparable interface) representing the utility and the appropriate Proposal message
      SortedSet<ProposalWithUtil> sortedProposals = new TreeSet<ProposalWithUtil>();

      for (Message m : receivedOffers.values()) {
        if (m.getPerformative().equalsIgnoreCase(MessageConstants.PROPOSAL)) {
          try {
            Proposal prop = (Proposal) m.getContent();

            // send a copy to topic - received offer from agent...
            List data = new LinkedList();
            data.add(prop);
            data.add(m.getSender().getName());
            gisShell.submitTopic(VisualAgent.TOPIC_LOCATION_RECEIVED_OFFER, data);

            // count the utility of this proposal
            double utility = 0;
            // iterate through all batches in my cfp
            for (Batch tb : cfpContent.getBatch()) {
              // find corresponding coverage in proposal- iterate through the list
              for (ItemCoverage ic : prop.getItemCoverage()) {
                if (ic.getItemid() == tb.getBatchid()) {
                  // we use max with 1 to avoid division by 0
                  long available = Math.max(((LocationAgent)towner).books.getPrivateAndAvailableStock(tb.getComodityName()),1);
                  // changed by david: use max with 1 to avoid cancellation of manual requests when utility is 0
                  long demand = Math.max(((LocationAgent)towner).books.getDemand(tb.getComodityName()),1);
                  // Increase the overall utility with utility of this batch
                  // Utility = (available ammount / daily demand) * batch coverage
                  utility += ((double)demand / available) * ic.getCoverage() * ic.getCoverage();
                  break;
                }
              }
            }
            if (utility == 0) { // This proposal is useless for me, don't add it into the map
            	continue; 
            }
            // This variable represents how much I will pay for one utility unit
            // We have to select the proposals with the lowest price for 1 utility unit
            double utilityPrice = (double)1000 / utility;// make it price independent...
            // Add the proposal with its utility into the SortedSet
            sortedProposals.add(new ProposalWithUtil(m, utilityPrice));
          }  catch (ClassCastException e) {
            Logger.logSevere("CFP answer Proposal recieved with wrong content - Proposal expected. From: " + m.getSender());
            sendNotUnderstood(m, "Wrong Content.");
          }
        }
      }

      // Now we have the set of proposals sorted by their utility - choose some of them using the greedy algorithm
      // Selected offer(s) will be stored in the following list (Addresses of senders)
      List<Address> selected = new LinkedList<Address>();

      // In this array is count of commodities booked so far
      long bookedCount[] = new long[cfpContent.getBatch().size()];
      // fill the array with 0
      Arrays.fill(bookedCount, 0);

      // Iterate through all proposals in TreeSet - set is sorted in ascending order
      for (ProposalWithUtil pwu : sortedProposals) {
        Message m = pwu.propMsg;
        Proposal prop = (Proposal) m.getContent();

        // It is possible to select this proposal
        boolean isPossible = true;

        // Look on all batches in proposal, whether (booked[i] + batch.count * coverage) < (batch.count + 20% overlap)
        // None of the batches in the proposal can get over +20% of the requested commodity to accept the proposal
        int i = 0;
        for (Batch tb : cfpContent.getBatch()) {
          // find corresponding coverage in proposal - iterate through the list
          for (ItemCoverage ic : prop.getItemCoverage()) {
            if (ic.getItemid() == tb.getBatchid()) {
              if ( (bookedCount[i] + tb.getCount() * ic.getCoverage()) > (tb.getCount() * 1.2) ) {
                isPossible = false;
              }
              break;
            }
          }
          if (!isPossible) { // Jump on the next proposal
        	  break; 
          }
          i++;
        }

        if (isPossible) {
          // Add counts of all batches to the bookedCount array values
          int i2 = 0;
          for (Batch tb : cfpContent.getBatch()) {
            // find corresponding coverage in proposal - iterate through the list
            for (ItemCoverage ic : prop.getItemCoverage()) {
              if (ic.getItemid() == tb.getBatchid()) {
                bookedCount[i2] += tb.getCount() * ic.getCoverage();
                break;
              }
            }
            i2++;
          }
          // Add address of the selected proposal sender to the list
          selected.add(m.getSender());
          // send a winner announcement to topic
          gisShell.submitTopic(VisualAgent.TOPIC_LOCATION_WINNER, m);
          // Add proposal in the map of selected proposals - used in evaluateAcceptReply method
          selectedProposalMap.put(m.getSender(), prop);
        }
      }

      // return selected proposal(s) or empty list if there is no selectable proposition...

      return selected;
    }

    /**
     *
     * @param m Message
     * @return boolean
     */
    @Override
    protected boolean evaluateAcceptReply(Message m) {
      if (m.getPerformative().equalsIgnoreCase(MessageConstants.FAILURE) ) {
        // failed to respect the contract...do nothing, no book entry so far...
      } else if(m.getPerformative().equalsIgnoreCase(MessageConstants.INFORM_DONE) ) {
        // retrieve the proposal - check if we have chosen this one
        Proposal originalProposal = (Proposal)selectedProposalMap.get(m.getSender());
        if (participants.contains(m.getSender())) {
          if (null != m.getContent()) {
            // we have received done with a real coverage - update the books to expect the right ammount.
            Proposal loadedGoods = (Proposal) m.getContent();
            // iterate through the reserved goods and update the expected and demand with real values
            ((LocationAgent)towner).books.goodsAcquired(cfpContent, originalProposal, loadedGoods);
          } else {
            //local partner, no transport, thus no transport coverage
            ((LocationAgent)towner).books.goodsAcquired(cfpContent, originalProposal);
          }
        } else {
          Logger.logSevere("Done recieved from refused CFP participant: " + m + "\nParticipants: " + participants);
        }
      }

      // determine if we finish - we close the task if all replies were recieved
      // this is an overkill for only one participant, but will help in the future
      synchronized (participants) {
        participants.remove(m.getSender());
        if ( 0 == participants.size()) return false;
        return true;
      }
    }
    
    @Override
    protected void evaluateAcceptTimeout() {
    }

    /**
     * Inner class used to store the proposal with its utility in the sorted map
     */
    private class ProposalWithUtil implements Comparable {
      Message propMsg;
      double utilityPrice;

      public ProposalWithUtil(Message propMsg, double utilityPrice){
        this.propMsg = propMsg;
        this.utilityPrice = utilityPrice;
      }

      public int compareTo(Object o) {
        double uDif = utilityPrice - ((ProposalWithUtil)o).utilityPrice;
        if (uDif == 0) return propMsg.getSender().toString().compareTo(((ProposalWithUtil)o).propMsg.getSender().toString());
        return (int)(uDif*10000);
      }

      public boolean equals(Object o) {
        double uDif = utilityPrice - ((ProposalWithUtil)o).utilityPrice;
        if (uDif == 0) return propMsg.getSender().toString().equals(((ProposalWithUtil)o).propMsg.getSender().toString());
        return false;
      }
    }

}
