package across.agents.location;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import across.agents.location.accounts.StockOfferDemand;
import across.agents.location.generator.Generator;
import across.agents.location.generator.PopulationGenerator;
import across.agents.location.gui.LocationAgentGUI;
import across.agents.location.task.LocAcquireGoodsTask;
import across.agents.location.task.LocIdleTask;
import across.agents.location.util.LocationUtils;
import across.agents.transporter.TransporterAgent;
import across.data.Batch;
import across.data.Page;
import across.data.Param;
import across.data.PublicParams;
import across.data.RequestList;
import across.data.Weather;
import across.data.simulation.SimulationObjectDescription;
import across.simulation.constants.AcrossDataConstants;
import across.simulation.constants.AcrossMessageConstants;
import across.simulation.constants.AcrossSimulationConstants;
import across.simulation.constants.AcrossTopicConstants;
import across.util.skn.AgentKnowledge;
import across.util.skn.CommunityKnowledge;
import across.util.skn.ServiceKnowledge;
import across.util.skn.listeners.NewAgentRegisteredListener;
import across.util.skn.update.UpdateListener;
import across.visio.oldvisio.VisioConnectionAgent;
import aglobe.container.agent.CMAgent;
import aglobe.container.sysservice.directory.DirectoryException;
import aglobe.container.sysservice.directory.DirectoryListener;
import aglobe.container.sysservice.directory.DirectoryRecord;
import aglobe.container.sysservice.directory.DirectoryService;
import aglobe.container.transport.Address;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.AglobeParam;
import aglobe.service.gis.client.GISClientService;
import aglobe.service.gis.client.GISTopicListener;
import aglobe.util.AglobeXMLtools;
import aglobe.visio3D.VisioConstants;
import aglobe.visio3D.ontology.Chart;
import aglobe.visio3D.ontology.SimpleBar;
import aglobex.protocol.subscribe.SubscribeInitiatorTask;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.SensorDescription;
import aglobex.simulation.ontology.SensoryData;
import aglobex.simulation.ontology.entity.EntityDescriptor;
import aglobex.simulation.protocol.cnp.CNPTaskOwner;
import aglobex.simulation.sensor.UniversalSensorAgent;

/**
 * Created by IntelliJ IDEA.
 * User: rehakm1
 * Date: 14.5.2004
 * Time: 15:36:08
 * To change this template use Options | File Templates.
 */

/**
 * This agent represents one or more entities demanding and/or supplying goods on the market.
 * These agents use CNP to trade goods between them. They also use CNP to ontract transporters.
 */
public class LocationAgent extends CMAgent implements CNPTaskOwner, GISTopicListener, NewAgentRegisteredListener, DirectoryListener, ClientServerTopicConstants {
	/** This agent's type */
	public static final String TYPE = "Location";
	
	/** main bookkeeping structure for the entity - contains suply store and demand...*/
	public StockOfferDemand books = new StockOfferDemand();
	
	/** List of the generators inside this Location, that generate supply and demand of goods from weather and other info */
	protected List<Generator> generators = new LinkedList<Generator>();
	
	/** Simulation data of the location agent */
	protected LocationAgentSimulationData mySimData = new LocationAgentSimulationData();
	
	/** Effects by the simulation objets on the location */
	protected SimulationObjectsEffects objEffects = new SimulationObjectsEffects(this,mySimData);
	
	/** GIS client shell to subscribe the weather */
	public GISClientService.Shell gisShell;
	
	/** Directory client shell. */
	protected DirectoryService.Shell dsShell;
	
	/** Receiver recieving weather info from Weather agent via gisShell */
	protected GISTopicListener weatherListener;

	/** Minimum size of the lot to acquire. */
	public int minLotSize;
	
	/** Page to store the info that will be submitted to visio agents */
	public Page page;
	
	/** Contains the information about all agents, containers and alliances known to agent.*/
	public CommunityKnowledge community = new CommunityKnowledge(this);
	
	/** gui of the agent */
	public LocationAgentGUI gui;
	
	/** Contains the information about commodity production of the location agent (use at updateBarsInVisio). */
	public List<Param> commodityParam;
	
	/** Idle task of the LocationAgent. */
	public LocIdleTask idleTask = null;
	
	/** Entity descriptor of this LocationAgent - it holds this agent's configuration. */
	protected EntityDescriptor entityDescriptor;
	
	/**
	 * Initializes the agent - run by the A-globe platform.  Initialization in these steps:
	 * <li> super initializer </li>
	 * <li> getting parameters from the configuration - LOCATION_AUTO_REQUESTS and TEST_STANDINS </li>
	 * <li> setting up gui </li>
	 * <li> setting idleTask </li>
	 */
	public void init(AgentInfo ai, int initState) {
		super.init(ai, initState);
		gisShell = (GISClientService.Shell) getContainer().getServiceManager().getService(this,GISClientService.SERVICENAME);
		
        gisShell.subscribeTopic(TOPIC_ENTITY_CONNECTION, this);
        gisShell.subscribeTopic(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND, this);
        gisShell.subscribeTopic(UniversalSensorAgent.TOPIC_SENSORY_DATA, this);
        gisShell.subscribeTopic(AcrossTopicConstants.TOPIC_SIMULATION_OBJECT,this);
		
		gui = new LocationAgentGUI(this);
		idleTask = new LocIdleTask(this);
		setIdleTask(idleTask);
		
		// XXX - gui
		gui.setTitle(ai.getReadableName());
		gui.setBounds(0, 0, 500, 142);
		gui.setAll(books);
		
	}
	
	/**
	 * This function handles the incoming configuration for this LocationAgent.
	 * @param ed
	 */
	private void handleConfiguration(EntityDescriptor ed) {
		minLotSize = ed.typeDescriptor.userParamsInteger.get(AcrossDataConstants.LOCATION_MIN_LOT_SIZE);
		mySimData.secondsPerDay = ed.typeDescriptor.userParamsInteger.get(AcrossDataConstants.LOCATION_SECONDS_PER_DAY);
		mySimData.locVicinity = ed.typeDescriptor.userParamsDouble.get(AcrossDataConstants.LOCATION_VICINITY_RANGE);
		
		// Register itself with yellow pages and subscribe the info about other Loca and Transporters 
		directoryRegisterAndSubscribe(ed);
		
		// Find the weather agent and subscribe for weather updates. 
		weatherInformationSubscribe();

        // configures the StockOfferDemand class 
        books.setMinLotSize(minLotSize);
        
		// init the generators using the params
        commodityParam = LocationUtils.getGeneratorParams(ed);
		Generator.initGenerators(commodityParam,generators);
		
		// register sensor
		SensorDescription sd = new SensorDescription();
        double range = ed.typeDescriptor.userParamsDouble.get(AcrossDataConstants.LOCATION_SENSOR_RANGE);
        AglobeParam rangeParam = new AglobeParam();
        rangeParam.setValue(Double.toString(range));
        sd.setSensorType("spherical");
        sd.getAglobeParam().add(rangeParam);
        gisShell.submitTopicToServer(UniversalSensorAgent.TOPIC_REGISTER_SENSOR, sd);
	}
	
	/**
	 * Registers the agent with directory and subscribes to receive the
	 * information about other Locations and Transports.
	 *
	 * @param ai AgentInfo
	 */
	protected void directoryRegisterAndSubscribe(EntityDescriptor ed)	{
		//prepare the page - it will be submitted to server
		page = new Page();
		page.setAddress(this.getAddress());
		page.setName(this.getName());
		page.setPublicParams(new PublicParams());
		// include all params, featuring tribe, region and generators...
		for (String key : ed.confParamsString.keySet()) {
			Param par = new Param();
			par.setName(key);
			String value = ed.confParamsString.get(key);
			par.setValue(value);
			page.getPublicParams().getParam().add(par);
			if("region".equalsIgnoreCase(par.getName())) {
				mySimData.region = value;
			} else if ("tribe".equalsIgnoreCase(par.getName())) {
				mySimData.tribe = value;
			}
		}
		
		page.getPublicParams().setType(TYPE);
		
		community.subscribeNewAgentRegisteredListener(this);
		
		// register with the new directory service
		dsShell = (DirectoryService.Shell) getContainer().getServiceManager().getService(this,DirectoryService.SERVICENAME);
		if (null != dsShell) {
			Collection<String> locations = new ArrayList<String>();
			locations.add(TYPE);
			try	{
				UpdateListener ulist = new UpdateListener(this.community, this);
				dsShell.register(this, locations);
				// subscribe the updates from the directory service
				dsShell.subscribe(ulist,TYPE);
				dsShell.subscribe(ulist,TransporterAgent.TYPE);
			} catch (DirectoryException e) {
				logger.severe("Directory Service exception during registration.: " + e + e.getMessage()+ e.getStackTrace());
			}
		} else {
			logger.severe("Directory Service not found on container: " + getContainer().getContainerName());
		}
	}
	
	/**
	 * Registers with weather agent to receieve weather information. This info is transmitted to generators. see handleTopic below.
	 */
	private void weatherInformationSubscribe() { 
		if ( gisShell!= null) {
			gisShell.subscribeTopic(AcrossTopicConstants.TOPIC_WEATHER_UPDATE, this);
			gisShell.subscribeTopic(AcrossTopicConstants.TOPIC_DISASTER_VISIBILITY, this);
			gisShell.submitTopicToServer(AcrossTopicConstants.TOPIC_WEATHER_UPDATE_REQUEST, null);
			gisShell.submitTopic(VisioConnectionAgent.TOPIC_VISIO_INFO, "<c00FF00>" + getContainer().getContainerName(),null);
			gisShell.submitTopic(VisioConnectionAgent.TOPIC_PAGE_UPDATE, page, null);
		} else {
			logSevere("GISClient not found.");
		}
	}
	
	
    /**
     * Handles these topics:
     * <li> TOPIC_ENTITY_CONNECTION - configuration topic</li>
     * <li> TOPIC_SIMULATION_TIME_UPDATE_1_SECOND - simulation time topic </li>
     * <li> TOPIC_DISASTER_VISIBILITY - is here a disaster? </li>
     * <li> TOPIC_WEATHER_UPDATE - new weather </li>
     * <li> TOPIC_SENSORY_DATA - sensory data </li>
     */
	public void handleTopic(String topic, Object content, String reason) {
    	if(topic.equalsIgnoreCase(ClientServerTopicConstants.TOPIC_ENTITY_CONNECTION)) {
    		if(content instanceof EntityDescriptor) {
    			handleConfiguration((EntityDescriptor) content);
    		}
    	} else if (topic.equalsIgnoreCase(ClientServerTopicConstants.TOPIC_SIMULATION_TIME_UPDATE_1_SECOND)) {
        	//call timeouts                
        	idleTask.handleTimeout();
        	// increase the simulation time
        	mySimData.simTime++;
        	
			// iterate over all the generators and let them update their supply/demand
        	if(mySimData.simTime%mySimData.secondsPerDay==0) {
				long totPopSize = 0;
				long totMood = 0;
				for (Generator gen : (List<Generator>) generators) {
					gen.generationStep(mySimData.weather, books, objEffects);
					if (gen instanceof PopulationGenerator)	{
						PopulationGenerator pg = (PopulationGenerator) gen;
						totPopSize += pg.getPopSize();
						totMood = pg.getCurrentMood();
					}
				}
				objEffects.startAggregation(Long.parseLong((String) content));
				// update the values shown in the visio
				gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_VISIO_INFO, "<c00FF00>" + getName() + "<cFF0000> " + totPopSize + "<cFF00FF> " + totMood,null);
				gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_PAGE_UPDATE, page);
				gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_VISIO_ACTION, ""+VisioConnectionAgent.ACTION_GOODS_STOLEN, getAddress().toString());
				gui.setAll(books);
	        	updateBarsInVisio(mySimData.weather);
        	}
        	if(isDemand()) {
	        	startCNP();
        	}
        }
    	// This city is in the disaster area.
    	else if (topic.equalsIgnoreCase(AcrossTopicConstants.TOPIC_DISASTER_VISIBILITY)) {
            // not implemented
        } else if (topic.equalsIgnoreCase(AcrossTopicConstants.TOPIC_WEATHER_UPDATE)) {
        	mySimData.weather = (Weather) content;
        } else if (topic.equalsIgnoreCase(UniversalSensorAgent.TOPIC_SENSORY_DATA)){
        	objEffects.parseSensoryData((SensoryData) content);
        } else if (topic.equalsIgnoreCase(AcrossTopicConstants.TOPIC_SIMULATION_OBJECT) && reason.equalsIgnoreCase(AcrossSimulationConstants.OBJECT_DESCRIPTION)) {
        	objEffects.objectDescription((SimulationObjectDescription) content);
        }
        else {
        	logSevere("Unexpected topic!"+topic);
        }
	}
	
	
	// TODO - heuristic
	/**
	 * This method should determine whether or not to start a CNP for goods.
	 */
	protected boolean isDemand() {
		Random rand = new Random();
		if(mySimData.simTime%(mySimData.secondsPerDay+rand.nextInt(5))==0) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Starts the CNP for goods - it tries to get the goods from the other locations.  The demand
	 * is partially random and partially calculated based upon the city storage.
	 * @param skipTimeChecking
	 */
	protected void startCNP() {
		// decide what to do - one auction per commodity, cumulated auction or nothing...
		// pass through the demand book and find candidates to buy
		List<RequestList> requestLists;
		requestLists = books.prepareSingleListGoodsDemand(this,true);
		
		// sends the messages (CFP) to the providers that have the commodity needed.
		for (RequestList rl : requestLists) {
			if (0 < rl.getBatch().size()) {
				
				Collection<Address> accessibleProviders = null;
				
				accessibleProviders = ( (ServiceKnowledge) community.
						getServices().get(TYPE)).
						getAccessibleProviders(getAddress());
				
				if (accessibleProviders == null) {
					accessibleProviders = new LinkedList<Address>();
				}
				
				StringBuffer aps = new StringBuffer();
				for (Address address : accessibleProviders) {
					aps.append(address.getName());
					aps.append(";");
				}
				
				if (accessibleProviders.size()>0)	{
						// This class handles the LocationAgent CNP
						new LocAcquireGoodsTask(this, accessibleProviders, rl,
								LocAcquireGoodsTask.TIMEOUT,
								aps.toString());
					gisShell.submitTopic(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_ACTION, ""+across.visio.oldvisio.VisioConnectionAgent.ACTION_CFP_GOODS ,getAddress().toString());
				}
			}
		}
	}
	
	/**
	 * starts a single CNP with a new request list got from GUI
	 * @param requestList request list constructed in a New Request Dialog
	 */
	public void startManualRequest(RequestList requestList) {
		for (Batch batch : requestList.getBatch()) {
			books.command(batch.getComodityName(),batch.getCount());
		}
		startCNP();
		gui.setAll(books);
		for (Batch batch : requestList.getBatch()) {
			books.cancelDemand(batch.getComodityName(),batch.getCount());
		}
	}
	
	/**
	 * Updates social knowledge after a new agent registered in the system.
	 */
	public void handleNewAgentRegistered(AgentKnowledge agent) {
		for (Iterator sit = agent.servicesProvided.iterator(); sit.hasNext();) {
			ServiceKnowledge serviceKnowledge = (ServiceKnowledge) sit.next();
			if (TransporterAgent.TYPE.equalsIgnoreCase(serviceKnowledge.serviceName) && !getAddress().equals(agent.address)) {
				SubscribeInitiatorTask task = new SubscribeInitiatorTask(this, agent.address, AcrossMessageConstants.PUBLIC_INFO_REQUEST, false) {
					/**
					 * Gets the agent's Page object 
					 */
					protected void subscribeInformResult(Object object) {
						Page page2 = (Page)object;
						community.agentPageUpdated(page2.getAddress() , page2.getPublicParams());
					}
					
					protected void subscribeAgreed() {
					}
					
					protected void subscribeRefused() {
						logger.severe(getName()+": Transporter " + subscribeContent + " has refused my subscribe.");
					}
				};
				task.start();
			}
		}
	}
	
	/**
	 * Handles the event when an agent becomes visible.
	 */
	public final void handleVisible(String containerName,
			DirectoryRecord[] records,
			String matchingFilter) {
		//not used
	}
	
	/**
	 * Handles that some agents/services becomes invisible to me.
	 * 
	 * @param containerName String - container name where agents/services become
	 *   invisible
	 * @param records DirectoryRecord[] - array of the agents/services which
	 *   become invisible. Only address and containerName fields are valid;
	 *   services field couldn't have valid data. An agent/service can handle
	 *   its own invisibility too.
	 * @param matchingFilter String - matching filter to which is this callback.
	 *   Can be used if an someone has more subscriptions to the same listener
	 *   with cross filter.
	 */
	public final void handleInvisible(String containerName, DirectoryRecord[] records,
			String matchingFilter) {
		// not used
	}
	
	/**
	 * Unused - necessary to implement
	 */
	public final void handleDeregister(String containerName, DirectoryRecord[] records,
			String matchingFilter) {
		//not used
		
	}
	
	/**
	 * Unused - necessary to implement
	 */
	public final void handleNewRegister(String containerName, DirectoryRecord[] records,
			String matchingFilter) {
		//not used
	}
	
	// XXX - rewrite
	/** 
	 * Display commodity production of the agent in visio (use chart and simple bar).
	 * 
	 * @param content restriction of the commodity production by disasters (Ex. water 0.5,
	 *        grain 0.9, fish 1, sheep 0.3)
	 * @todo Do some more sophisticated hooking with the visio!!! 
	 */
	public void updateBarsInVisio(Weather wea) {
		
		// weather information unavailiable yet
		if(wea==null) {
			return;
		}
		
		Chart chart = new Chart();
		
		for (Param param : commodityParam) {
			// default simple bar
			SimpleBar bar = AglobeXMLtools.makeSimpleBar("null",VisioConstants.COMMUNICATION_COLOR_RED,(byte)0);
			
			for (Object wob : wea.getParam()) {
				Param wp = (Param) wob;			
				//testing an existing commodity production by the agent
				if ((param.getName().startsWith(wp.getName(),4))) {
					byte value;
					value = Byte.parseByte(String.valueOf(Math.round(Double.parseDouble(wp.getValue())*Byte.MAX_VALUE)));
					//distribution value of commodity according to the colors
					if (Double.parseDouble(wp.getValue())<0.33){
						bar = AglobeXMLtools.makeSimpleBar(wp.getName(),VisioConstants.COMMUNICATION_COLOR_RED,(byte)(value*2));
					} else if(Double.parseDouble(wp.getValue())<0.66){
						bar = AglobeXMLtools.makeSimpleBar(wp.getName(),VisioConstants.COMMUNICATION_COLOR_YELLOW,(byte)(value*2));
					} else{
						bar = AglobeXMLtools.makeSimpleBar(wp.getName(),VisioConstants.COMMUNICATION_COLOR_GREEN,(byte)(value*2));
					}
				}				
			}
			// XXX - rewrite
			if (bar.getName().compareToIgnoreCase("null")!=0){
				chart.getContent().add(bar);
			}
		}		
		//the chart update in the visio
		if (chart.getContent().size()!=0) {
			gisShell.submitTopic(VisioConnectionAgent.TOPIC_SET_CHART, chart, getContainer().getContainerName());
		}	
	}
	
	/**
	 * Returns current simulation time, as recieved from weather agent.
	 * @return simulation time
	 */
	public long getCurrentTime() {
		return mySimData.simTime;
	}

}
