package across.agents.emergency.centre;

import iglobe.plan.RescueTask;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Map.Entry;

import across.agents.emergency.centre.EventHandlingTransaction.TransactionState;
import across.agents.emergency.repair.RepairVehicleAgent;
import across.agents.emergency.storage.EmergencyStorageAgent;
import across.data.simulation.SimulationObjectDescription;
import across.simulation.constants.AcrossDataConstants;
import across.simulation.constants.AcrossMessageConstants;
import across.util.ixplanparser.IxPlanParser;
import across.util.ixplanparser.ParserKnowledge;
import across.visio.oldvisio.VisioConnectionAgent;
import aglobe.container.agent.CMAgent;
import aglobe.container.sysservice.directory.DirectoryException;
import aglobe.container.sysservice.directory.DirectoryListener;
import aglobe.container.sysservice.directory.DirectoryRecord;
import aglobe.container.sysservice.directory.DirectoryService;
import aglobe.container.task.ConversationUnit;
import aglobe.container.task.Task;
import aglobe.container.transport.Address;
import aglobe.ontology.AgentInfo;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobe.service.gis.client.GISClientService;
import aglobe.service.gis.client.GISTopicListener;
import aglobe.util.AglobeXMLtools;
import aglobe.visio3D.VisioConstants;
import aglobe.visio3D.ontology.Chart;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.entity.EntityDescriptor;

/**
 * This agent is in charge of handling the incoming accidents and running the rescue actions.
 * @author Eduard Semsch
 *
 */
public class EmergencyAgent extends CMAgent implements ClientServerTopicConstants, GISTopicListener, DirectoryListener {

	/** Type of this agent */
	public static final String TYPE = "EmergencyCentre";
	
	/** GIS shell */
	protected GISClientService.Shell gisShell;
	
	/** DirectoryService Shell */
	protected DirectoryService.Shell dirShell;
	
	/** Scheduling knowledge of the agent. */
	protected SchedulingKnowledge schedulingKnowledge = new SchedulingKnowledge();
	
	/** Rescue actions currently running.  Map ObjectId (disaster event) <-> handling transaction. */ 
	protected Map<String,EventHandlingTransaction> rescueActions = new HashMap<String, EventHandlingTransaction>(); 
	
	/** Accidents that were registered and are subject for resolving. */
	protected List<String> registeredAccidents = new LinkedList<String>();
	
	/** Name of the planner this emergency centre uses. */
	protected String plannerName;
	
	/** Other emergency centres. */
	protected List<Address> otherEmergencyAgents = new ArrayList<Address>();
	
	/** Resolves who handles which accident between the emergency centres. */
	protected AccidentsHandlingResolver handlingResolver;
	
	/**
	 * Initializes the agent in these steps:
	 * <li> Directory Service - subscription of repair vehicles and planner and own advertising. </li>
	 * <li> GIS - subscription of simulation topics. </li>
	 * <li> Set the idle task. </li>
	 */
	@Override
	public void init(AgentInfo ai, int initState) {
		gisShell = (GISClientService.Shell) this.getContainer().getServiceManager().getService(this, GISClientService.SERVICENAME);
		dirShell = (DirectoryService.Shell) this.getContainer().getServiceManager().getService(this, DirectoryService.SERVICENAME);
		if(gisShell!=null && dirShell!=null) {
			gisShell.subscribeTopic(TOPIC_ENTITY_CONNECTION, this);
			gisShell.subscribeTopic(TOPIC_ENTITY_CONTROL, this);
			gisShell.subscribeTopic(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND, this);
		} else {
			logSevere("Unable to locate GIS!");
			this.stop();
		}
		try {
			dirShell.register(this, new ArrayList<String>(Arrays.asList(new String[] {TYPE})));
		} catch (DirectoryException e) {
			e.printStackTrace();
		}
		dirShell.subscribe(this, RepairVehicleAgent.TYPE);
		dirShell.subscribe(this, EmergencyStorageAgent.TYPE);
		dirShell.subscribe(this, EmergencyAgent.TYPE);
		this.setIdleTask(new EmergencyAgentIdleTask(this));	
		handlingResolver = new AccidentsHandlingResolver(this) {

			@Override
			public int getPriority(SimulationObjectDescription sod) {
				if(rescueActions.get(sod.getName())!=null) {
					return Integer.MAX_VALUE;
				} else {
					return this.hashCode()%10000-10000*rescueActions.size();
				}
			}

			@Override
			public void readyForHandling(SimulationObjectDescription sod) {
				handleAccident(sod);
			}
			
		};
	}

	/**
	 * Agent's idle task - handles newly coming messages.  
	 * @author Eduard Semsch
	 *
	 */
	public class EmergencyAgentIdleTask extends Task implements MessageConstants {

		public EmergencyAgentIdleTask(ConversationUnit cu) {
			super(cu);
		}

		/**
		 * Handles these messages:
		 * <li> performative = INFORM, content = SimulationObjectDescription - this collects the
		 * information about current accidents. </li>
		 */
		@Override
		protected void handleIncomingMessage(Message m) {
			if(m.getProtocol()!=null && m.getProtocol().equals(AcrossMessageConstants.ACCIDENT_HANDLING_RESOLUTION_PROTOCOL)) {
				handlingResolver.incomingMessage(m);
			}
			// report accident
			else if(m.getPerformative().equals(INFORM)) {
				SimulationObjectDescription sod = (SimulationObjectDescription) m.getContent();
				if(!registeredAccidents.contains(sod.getName())) {
					registeredAccidents.add(sod.getName());
					handlingResolver.startResolution(sod, otherEmergencyAgents);
				}
			}
		}
		
	}
	
	/**
	 * Starts the accident handling.  The accident specified by its location is later on handled
	 * by the repair vehicles of this emergency centre.
	 * @param objLoc
	 */
	public void handleAccident(SimulationObjectDescription sod) {
		// if the event is not already handled, handle it
		if(!rescueActions.keySet().contains(sod.getName())) {
			RescueTask rt = new RescueTask(sod.getName()+"@"+sod.getLocation().startLocation+"<->"+sod.getLocation().endLocation,sod.getNumberOfWounded());
			rt.setId(sod.getName());
			rt.setUnavailableResources(new LinkedList<String>());
			EventHandlingTransaction trans = new EventHandlingTransaction(rt,EmergencyAgent.this,schedulingKnowledge);
			rescueActions.put(sod.getName(),trans);
		}
	}
	
	/**
	 * Handles these topics:
	 * <li> TOPIC_ENTITY_CONNECTION - entity configuration. </li>
	 * <li> TOPIC_SIMULATION_TIME_UPDATE_1_SECOND - on this topic's comming all the
	 * pending transactions are woken and checked for completion.  If the transactions are 
	 * completed successfully they are removed from the set of pending transactions.  If they
	 * are failed, they are restarted. </li>
	 */
	public void handleTopic(String topic, Object content, String reason) {
		if(topic.equalsIgnoreCase(TOPIC_ENTITY_CONNECTION)) {
			if(content instanceof EntityDescriptor) {
				EntityDescriptor ed = (EntityDescriptor) content;
				handleConfiguration(ed);
			}
		}
		else if(topic.equalsIgnoreCase(TOPIC_SIMULATION_TIME_UPDATE_1_SECOND)) {
			displayEventsChart();
			// run the transactions
			for (Iterator<Entry<String,EventHandlingTransaction>> iter = rescueActions.entrySet().iterator(); iter.hasNext();) {
				Entry<String,EventHandlingTransaction> eventEntry = iter.next();
				EventHandlingTransaction eht = eventEntry.getValue();
				if(eht.finished()) {
					if(eht.failed()) {
						eht.restart();
					} else {
						registeredAccidents.remove(eventEntry.getKey());
						iter.remove();
					}
				} else {
					eht.run();
				}
			}
		}
		
	}

	/**
	 * Handles the incoming configuration.
	 * @param ed
	 */
	protected void handleConfiguration(EntityDescriptor ed) {
		// display visio information
        String visioInfo = "<b>" + getName();
        gisShell.submitTopicToServer(across.visio.oldvisio.VisioConnectionAgent.TOPIC_VISIO_INFO, visioInfo);
        
        // locate planner
        plannerName = ed.confParamsString.get(AcrossDataConstants.EMERGENCY_CENTRE_PLANNER_NAME);
		dirShell.subscribe(this, plannerName);
		
		// TODO - do this smarter
		// create parser knowledge for planner
		ParserKnowledge pk = new ParserKnowledge();
		pk.getVehicles().addAll(new ArrayList<String>(Arrays.asList(new String[]{"ambulance","truck","police-team"})));
		List<String> docPickupMaterials = new ArrayList<String>(Arrays.asList(new String[]{"doctor","paramedic"}));
		List<String> docDropMaterials = new ArrayList<String>(Arrays.asList(new String[]{"injured-person"}));
		pk.getPickupMaterials().addAll(docPickupMaterials);
		pk.getDropMaterials().addAll(docDropMaterials);
		List<String> docTransportMaterials = new ArrayList<String>();
		docTransportMaterials.addAll(docPickupMaterials);
		docTransportMaterials.addAll(docDropMaterials);
		pk.getWhichVehicleTransportWhichMaterial().put("ambulance", docTransportMaterials);
		pk.getWhichVehicleTransportWhichMaterial().put("truck", docTransportMaterials);
		pk.getWhichVehicleTransportWhichMaterial().put("police-team", null);
		pk.getTemplatePlaces().put("hospital", "Storage");
		schedulingKnowledge.ixToAcrossPlanParser = new IxPlanParser(pk);
	}
	
	/* ----------------------------- VISIO ---------------------------*/
	
	/** This displays the handled events */
	protected void displayEventsChart() {
		if(rescueActions.size()==0) {
			return;
		}
		Chart chart = new Chart();
		for (EventHandlingTransaction eht : rescueActions.values()) {
			String objId = eht.rescueTask.getLocation();
			objId = objId.substring(0,objId.indexOf("@"));
			chart.getContent().add(AglobeXMLtools.makeSimpleBar(objId, getColorForState(eht.state), (byte) 255));			
		}
		gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_SET_CHART, chart, getContainer().getContainerName());
	}
	
	/**
	 * Gets color for transaction state.
	 * @param number
	 * @return
	 */
	private byte getColorForState(TransactionState state) {
		if(state==TransactionState.NOT_PLANNED || state==TransactionState.PLANNING) {
			return VisioConstants.COMMUNICATION_COLOR_BLUE;
		} else if (state==TransactionState.NOT_SCHEDULED || state==TransactionState.SCHEDULING) {
			return VisioConstants.COMMUNICATION_COLOR_YELLOW;
		} else if (state==TransactionState.NOT_EXECUTED || state==TransactionState.EXECUTING) {
			return VisioConstants.COMMUNICATION_COLOR_ORANGE;
		} else if (state==TransactionState.DONE) {
			return VisioConstants.COMMUNICATION_COLOR_GREEN;
		} else {
			return VisioConstants.COMMUNICATION_COLOR_RED;
		}
	}
	
	/** Displays an action in visio. */
	protected void displayVisioAction(byte action) {
		gisShell.submitTopicToServer(VisioConnectionAgent.TOPIC_VISIO_ACTION, ""+action, getAddress().toString());
	}
	
	/* ------------------------------ DIRECTORY LISTENER ------------------*/
	public void handleDeregister(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}

	public void handleInvisible(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}

	public void handleNewRegister(String containerName, DirectoryRecord[] records, String matchingFilter) {
		// TODO Auto-generated method stub
		
	}

	/**
	 * Collects addresses from drivers, hospitals and from the planner agent.
	 */
	public void handleVisible(String containerName, DirectoryRecord[] records, String matchingFilter) {
		if(matchingFilter.equalsIgnoreCase(RepairVehicleAgent.TYPE)) {
			for (int i = 0; i < records.length; i++) {
				schedulingKnowledge.driversAddresses.add(records[i].address);
			}
		} else if (matchingFilter.equalsIgnoreCase(EmergencyStorageAgent.TYPE)) {
			for (int i = 0; i < records.length; i++) {
				schedulingKnowledge.storageAddresses.add(records[i].address);
			}
		}
		else if (matchingFilter.equalsIgnoreCase(plannerName)) {
			for (int i = 0; i < records.length; i++) {
				if(records[i].containerName.equals(getContainer().getContainerName())) {
					schedulingKnowledge.plannerAddress = records[i].address;
				}
			}
		} else if (matchingFilter.equalsIgnoreCase(EmergencyAgent.TYPE)) {
			for (int i = 0; i < records.length; i++) {
				if(!records[i].address.equals(getAddress())) {
					otherEmergencyAgents.add(records[i].address);
				}
			}
		}
	}

}
