package across.agents.emergency.centre;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import across.data.simulation.SimulationObjectDescription;
import across.simulation.constants.AcrossMessageConstants;
import aglobe.container.task.ConversationUnit;
import aglobe.container.task.Task;
import aglobe.container.transport.Address;
import aglobe.container.transport.InvisibleContainerException;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;

public abstract class AccidentsHandlingResolver {

	protected ConversationUnit cu;
	
	protected Map<String,Integer> priorityMap = new HashMap<String, Integer>();
	
	public AccidentsHandlingResolver(ConversationUnit cu) {
		this.cu = cu;
	}
	
	/** Starts the accident handling resolution protocol */
	public void startResolution(SimulationObjectDescription sod,List<Address> adrList) {
		new AccidentHandlingResolutionProtocolInitiator(cu,sod,adrList);
	}
	
	/** Incoming message with the AcrossMessageConstants.ACCIDENT_HANDLING_RESOLUTION_PROTOCOL*/
	public void incomingMessage(Message mes) {
		new AccidentHandlingResolutionProtocolParticipant(cu,mes);
	}
	
	
	protected class AccidentHandlingResolutionProtocolInitiator extends Task implements MessageConstants {

		private SimulationObjectDescription sod;
		
		private List<Address> addList;

		private Map<Message,AccidentsHandlingProtocolData> replies = new HashMap<Message,AccidentsHandlingProtocolData>();
		
		public AccidentHandlingResolutionProtocolInitiator(ConversationUnit cu, SimulationObjectDescription sod, List<Address> otherCentres) {
			super(cu);
			this.sod = sod;
			addList = otherCentres;
			priorityMap.put(sod.getName(), getPriority(sod));
			for (Address address : otherCentres) {
				Message mes = Message.newInstance(INFORM);
				mes.setProtocol(AcrossMessageConstants.ACCIDENT_HANDLING_RESOLUTION_PROTOCOL);
				mes.setReceiver(address);
				mes.setSender(this.getAddress());
				AccidentsHandlingProtocolData content = new AccidentsHandlingProtocolData();
				content.priority = priorityMap.get(sod.getName());
				content.sod = sod;
				mes.setContent(content);
				try {
					sendMessage(mes);
				} catch (InvisibleContainerException e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		protected void handleIncomingMessage(Message m) {
			replies.put(m,(AccidentsHandlingProtocolData) m.getContent());
			if(replies.size()==addList.size()) {
				boolean handle = true;
				int highest = priorityMap.get(sod.getName());
				Address highestAddr = null;
				// find the highest priority and check if it is mine
				for (Message mes: replies.keySet()) {
					AccidentsHandlingProtocolData pd = replies.get(mes);
					if(pd.priority >= highest) {
						handle = false;
						highest = pd.priority;
						highestAddr = mes.getSender();
					}  
				}
				if(!handle && highest==priorityMap.get(sod.getName())) {
					if(highestAddr.hashCode()<getAddress().hashCode()) {
						highest += getAddress().hashCode();
						priorityMap.put(sod.getName(), highest);
						handle = true;
					}
				}
				AccidentsHandlingProtocolData pd = new AccidentsHandlingProtocolData();
				pd.sod = sod;
				pd.priority = highest;
				// tell all the highest priority
				for (Message mes: replies.keySet()) {
					Message rep = mes.getReply();
					rep.setContent(pd);
					try {
						sendMessage(rep);
					} catch (InvisibleContainerException e) {
						e.printStackTrace();
					}
				}
				if(handle) {
					priorityMap.remove(sod);
					readyForHandling(sod);
				}
			}
		}

	}
	
	protected class AccidentHandlingResolutionProtocolParticipant extends Task implements MessageConstants {

		public AccidentHandlingResolutionProtocolParticipant(ConversationUnit cu, Message mes) {
			super(cu);
			Message rep = mes.getReply();
			AccidentsHandlingProtocolData pd = (AccidentsHandlingProtocolData) mes.getContent();
			priorityMap.put(pd.sod.getName(), getPriority(pd.sod));
			AccidentsHandlingProtocolData myPd = new AccidentsHandlingProtocolData();
			myPd.sod = pd.sod;
			myPd.priority = priorityMap.get(pd.sod.getName());
			rep.setContent(myPd);
			try {
				sendMessage(rep);
			} catch (InvisibleContainerException e) {
				e.printStackTrace();
			}
		}

		@Override
		protected void handleIncomingMessage(Message m) {
			AccidentsHandlingProtocolData pd = (AccidentsHandlingProtocolData) m.getContent();
			// if not already handled, handle it
			if(priorityMap.get(pd.sod)!=null && pd.priority==priorityMap.get(pd.sod)) {
				priorityMap.remove(pd.sod);				
				readyForHandling(pd.sod);
			}
		}
		
	}
	
	/** This function is called once the accident handling has been resolved. */
	public abstract void readyForHandling(SimulationObjectDescription sod);
	
	/** This function should return the priority for this agent to handle the accident. */
	public abstract int getPriority(SimulationObjectDescription sod);
}
