package across.agents.driver;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import across.data.simulation.SimulationObjectDescription;
import across.simulation.constants.AcrossDataConstants;
import across.simulation.constants.AcrossEntityConstants;
import across.simulation.constants.AcrossSimulationConstants;
import across.simulation.constants.AcrossTopicConstants;
import aglobe.ontology.AglobeParam;
import aglobe.service.gis.client.GISTopicListener;
import aglobex.simulation.ontology.Sensor;
import aglobex.simulation.ontology.SensorDescription;
import aglobex.simulation.ontology.SensoryData;
import aglobex.simulation.ontology.SensoryDataRecord;
import aglobex.simulation.ontology.entity.EntityDescriptor;
import aglobex.simulation.sensor.UniversalSensorAgent;

/**
 * This class is responsible for processing information from the vehicle's 
 * sensors.
 * @author Eduard Semsch
 *
 */
public class DriverSensors implements GISTopicListener {

	/** The owner agent (RepairVehicleAgent). */
	protected DriverAgent owner;
	
	/** Last received sensory data. */
	protected SensoryData currentSensoryData;
	
	/** All entities in sight - their container names + the sensory information about them. */
	protected Map<String,SensoryDataRecord> entitiesInSight = new HashMap<String,SensoryDataRecord>();
	
	/** Entities in sight sorted by type. */
	protected Map<String,Set<String>> entitiesInSightByType = new HashMap<String, Set<String>>();
	
	/** 
	 * Objects nearby this vehicle. The map holds all the objects (simulation objects) in the sensor range.
	 * If the value of in the map is null it means that the object is in sight, but its description is not
	 * known yet.  The objects out of sight are periodically removed.  
	 */
	protected Map<String,SimulationObjectDescription> objectsInSight = new HashMap<String,SimulationObjectDescription>();
	
	public DriverSensors(DriverAgent owner) {
		this.owner = owner;
	}

	/* ================================ INTERFACE ============================== */
	
	/** Call this method from the agent in order to process sensory information. */
	public void processSensorTopic(final Object content) {
		currentSensoryData = (SensoryData) content;
		parseSensoryData(currentSensoryData);
	}

	/** Method for handling configuration of the agent. */
	public void handleConfiguration(EntityDescriptor ed) {
		
		// sensor registration
        SensorDescription sd = new SensorDescription();
        double range = ed.typeDescriptor.userParamsDouble.get(AcrossDataConstants.DRIVER_SENSOR_RANGE);
        AglobeParam rangeParam = new AglobeParam();
        rangeParam.setValue(Double.toString(range));
        sd.setSensorType("spherical");
        sd.getAglobeParam().add(rangeParam);
        owner.gisShell.submitTopicToServer(UniversalSensorAgent.TOPIC_REGISTER_SENSOR, sd);
		
		owner.gisShell.subscribeTopic(AcrossTopicConstants.TOPIC_SIMULATION_OBJECT, this);
	}
	
	/** Tells whether an entity is in sight. */
	public boolean isEntityInSight(String entId) {
		return entitiesInSight.containsKey(entId);
	}
	
	/** Tells whether a simulation object is in sight. */
	public boolean isObjectInSight(String objId) {
		return objectsInSight.containsKey(objId);
	}
	
	/** Is any entity of specified type in sight? */
	public boolean isEntityTypeInSight(String entType) {
		return entitiesInSightByType.get(entType)!=null && entitiesInSightByType.get(entType).size()!=0;
	}

	/** Returns all the simulation objects (accidents) in sight. */
	public List<String> getObjectsInSight() {
		return new ArrayList<String>(objectsInSight.keySet());
	}
	
	/** Returns all entities of one type in sight. */
	public Set<String> getEntitiesOfTypeInSight(String entType) {
		Set<String> entInSigh = entitiesInSightByType.get(entType);
		if(entInSigh==null) {
			return new HashSet<String>();
		} else {
			return entInSigh;
		}
	}
	
	/** Returns the sensory record of specified entity. */
	public SensoryDataRecord getSDROfEntity(String entId) {
		if(!isEntityInSight(entId)) {
			return null;
		} else {
			return entitiesInSight.get(entId);
		}
	}
	
	/** Returns the description of a simulation object. */
	public SimulationObjectDescription getObjectsDescription(String objId) {
		if(!isObjectInSight(objId)) {
			return null;
		} else {
			return objectsInSight.get(objId);
		}
	}
	
	/* ================================ PROCESSING ============================== */
	
	/**
	 * Handles these topics:
	 * <li> AcrossTopicConstants.TOPIC_SIMULATION_OBJECT - incoming objects description. </li>
	 */
	public void handleTopic(String topic, Object content, String reason) {
		if(topic.equalsIgnoreCase(AcrossTopicConstants.TOPIC_SIMULATION_OBJECT)) {
			SimulationObjectDescription sod = (SimulationObjectDescription) content;
			objectsInSight.put(sod.getName(), sod);
		}
	}

	/** Synchronization method. */
	public void addEvent(Runnable e) {
		owner.addEvent(e);		
	}
	
	/**
	 * Actualizes the entities in sight of this entity.  Also asks for description of the
	 * simulation objects in sight.
	 * @param sd
	 */
	protected void parseSensoryData(SensoryData sd) {
		// in this list we keep the entities that are in sight right now.
		List<String> currentEntities = new LinkedList<String>();
		for (Sensor sen : sd.sensor) {
			if(sen.sensoryDataRecord.size()>0) {
				for (SensoryDataRecord sdr : sen.sensoryDataRecord) {
					currentEntities.add(sdr.targetContainerName);
					
					// put the newly coming data to the entities map
					entitiesInSight.put(sdr.targetContainerName, sdr);
					
					// put the newly coming data to the entities by type map
					Set<String> entitiesOfOneType = entitiesInSightByType.get(sdr.targetEntityType);
					if(entitiesOfOneType==null) {
						entitiesOfOneType = new HashSet<String>();
						entitiesInSightByType.put(sdr.targetEntityType, entitiesOfOneType);
					}
					entitiesOfOneType.add(sdr.targetContainerName);
					
					// put the newly coming data to the objects map if the enitity is an object
					if(sdr.targetEntityType.equalsIgnoreCase(AcrossEntityConstants.SIMULATION_OBJECT_TYPE)) {
						String objId = sdr.targetContainerName;
						if(!objectsInSight.containsKey(objId)) {
						   objectsInSight.put(objId, null);
						}
						askForObjectDescription(objId);
					}					
	    		}
	    	}
		}
		removeEntitiesOutOfSight(currentEntities);
	}

	/** Removes the entities out of sight from the lists. */
	protected void removeEntitiesOutOfSight(List<String> currentEntities) {
		// iterate over the map of entities in sight and remove the entities that are not in sight any more.
		for (Iterator<Entry<String,SensoryDataRecord>> iter = entitiesInSight.entrySet().iterator(); iter.hasNext();) {
			
			Entry<String,SensoryDataRecord> element =  iter.next();
			String entId = element.getKey();
			SensoryDataRecord sdr = element.getValue();
			
			if(!currentEntities.contains(entId)) {
				// remove from list of entities by type
				entitiesInSightByType.get(sdr.targetEntityType).remove(entId);
				// remove from objects in sight
				objectsInSight.remove(entId);
				// remove from list of entities
				iter.remove();
			}
		}
	}
	
	/** This method asks the simulation server for the simulation object description. */
	protected void askForObjectDescription(String objId) {
		owner.gisShell.submitTopicToServer(AcrossTopicConstants.TOPIC_SIMULATION_OBJECT, objId,AcrossSimulationConstants.OBJECT_DESCRIPTION);
	}
	
}
