package across.agents.driver;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import across.agents.driver.data.Route;
import across.agents.driver.data.RouteInfo;
import across.agents.driver.data.TransportRoute;
import across.data.AcrossMapArcs;
import across.data.AcrossMapDistances;
import across.data.AcrossMapNodes;
import across.data.Arc;
import across.data.Batch;
import across.data.Node;
import across.data.RequestList;
import across.data.TransportOrder;
import across.data.Waypoint;
import across.data.simulation.SimulationObjectDescription;
import across.simulation.constants.AcrossControlConstants;
import across.simulation.constants.AcrossDataConstants;
import across.util.pathfinding.AStarModPathfinding;
import across.util.pathfinding.Pathfinding;
import across.util.pathfinding.RoadEvaluator;
import aglobe.container.transport.InvisibleContainerException;
import aglobe.ontology.Message;
import aglobe.ontology.MessageConstants;
import aglobex.simulation.global.ClientServerTopicConstants;
import aglobex.simulation.ontology.entity.EntityDescriptor;

public class DriverMovement {

	/** List of map locations - their indexes are the indexes in the roads field */
	protected List<Node> mapNodes;
	 
	/** Info for the map roads */
	protected List<Arc> mapArcs;
	
	
	/** Pathfinding algorithm */
	protected Pathfinding pathfinding;

	/** This class computes price of this vehicle going through a specified road. */
	protected RoadEvaluator roadEval;
	
	/** Owner of this class */
	protected DriverAgent owner;
	
	/** Data class of the owner */
	protected DriverAgentSimulationData ownerData;
	
	
	/** Waypoints - the route that the driver follows. */
	protected LinkedList<Waypoint> waypoints = new LinkedList<Waypoint>();  
	
	/** Position where the Driver is actually located or from which is now going. */
	protected String actualLoc;   
	
	/** The node where the vehicle is currently heading to. */
	protected String headingTo;
	
	/** Home node, as given by transporter. If the driver can not communicate with transporter, it goes home */
	protected String homeLoc = null;
	
	
	/** Transport orders this driver has to deliver */
	protected TransportOrder commitedTransports = new TransportOrder();
	
	/** Says whether the driver is currently retreating from an obstacle back to initial location */ 
	protected boolean retreating = false;
	
	/** When the counter gets to zero the barred nodes are refreshed */
	private int counterToRefresh = 50;
	
	/** Barred roads list */
	protected LinkedList<Arc> barredRoads = new LinkedList<Arc>();
	
	
	public DriverMovement(DriverAgent owner) {
		this.owner = owner;
	}
	
	/**
	 * Handles all topics concerning movement.
	 * @param content
	 * @param reason
	 */
	public void handleMovementTopic(Object content, String reason) {
		if(counterToRefresh==0) {
			counterToRefresh = 50;
			refreshBarred();
		} else {
			counterToRefresh--;
		}
	  if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_BAD_NODE)) {
  		  owner.gui.addToTextArea("Content: RESPONSE_FAILED");
  		  owner.gui.addToTextArea("Reason: " + reason);
  		  owner.logSevere("Bad node chosen!");
	  } else if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_FAILURE)) {
		  owner.gui.addToTextArea("Content: RESPONSE_FAILED");
		  owner.gui.addToTextArea("Reason: " + reason);
		  owner.logSevere("Movement failure!");
	  } else if(reason.equalsIgnoreCase(AcrossControlConstants.RESPONSE_STOPPED) && content instanceof String) {
		  
		  	// get the actual waypoint, check the requests to be loaded/unloaded
		  	if(!content.equals(actualLoc)) {
		  		double time = 0;
		  		for (Arc arc : mapArcs) {
		  			
					if((arc.getFrom() == actualLoc) && (arc.getTo() == content)){
						time = roadEval.getTime(arc);
					}else if ((arc.getTo() == actualLoc) && (arc.getFrom() == content)) {
						time = roadEval.getTime(arc);
					}
					
				}
		  		owner.totalTaskTime -= time;
		  	}
		  	
		  	actualLoc = (String) content;
		  	
		  	retreating = false;
		  	
		  	// if there are no waypoints scheduled.
		  	if(waypoints.size()==0) {
		  		return;
		  	}
		  	
		  	// if there are waypoints scheduled remove the first one
		  	Waypoint wp = waypoints.getFirst();
		  	if(wp.getLocationName().equals(actualLoc)) {
		  		waypoints.removeFirst();
		  	}
		  	
		  	// load and unload the cargo
			if (wp.getRequestList().size() > 0){
			 	owner.loadAndUnload(wp.getRequestList());  
				removeDoneWaypointFromTransportOrder(wp.getRequestList());
			}
			
		  	// XXX - gui
			owner.gui.jTextFieldPosition.setText("-> " + actualLoc);
			owner.gui.routeTabModel.fireTableDataChanged();
			if (owner.gui.routeTabSelectedRow == 0) { // First line in route table was selected, request table should be cleared
			    owner.gui.clearRequestTable();
			} else {
			    owner.gui.decreaseRouteTableSelection();
			}
			owner.gui.addToTextArea("Content: RESPONSE_ARRIVED");
			owner.gui.addToTextArea("Reason: " + reason);
			owner.gui.jTextFieldPosition.setText(actualLoc);
			owner.gui.jTextFieldTaskTime.setText(Double.toString((double)((int)(100*owner.totalTaskTime))/100));
			owner.gui.routeTabModel.fireTableDataChanged();
			if (owner.gui.routeTabSelectedRow == 0) {  // First line in route table was selected, request table should be cleared
				owner.gui.clearRequestTable();
			} else {
			  owner.gui.decreaseRouteTableSelection();
			}
			
			move();
      	}
	}
	
  /**
   * Move to the next waypoint (if there's any in the list)
   */
  protected void move(){
	if (!isOnRoute()) {
		// notify the transporter agent that we are idle...
		if (!owner.sendAvailable()) {
			// if inaccessible, go home...
	        goHome();
	    }
	    return;
	} else {
		
		Waypoint wp = waypoints.getFirst();
	    
	    String waypointName = wp.getLocationName();
	    
	    headingTo = waypointName;
	
	    owner.gisShell.submitTopic(ClientServerTopicConstants.TOPIC_ENTITY_CONTROL, waypointName, AcrossControlConstants.MOVE_TO);
	    
	 }
  }

	  
    /**
     * This method makes the agent move to it's home location as communicated in the subscribe it got from transporter.
     */
    protected void goHome() {
		if (homeLoc == null) {
			return;
		}
		if (!actualLoc.equalsIgnoreCase(homeLoc)) {
		    TransportOrder to = new TransportOrder();
		    to.getWaypoint().add(new Waypoint(homeLoc));
		    TransportRoute route = planRoute(to, false);
		    waypoints.addAll(route.getRoute().getWaypoints());
		    
		    // XXX - gui
		    double totalTaskTime = route.getRoute().getRouteInfo().routeTime;
		    owner.gui.jTextFieldTaskTime.setText(Double.toString((double)((int)(100*totalTaskTime))/100));
		    owner.gui.addToTextArea("Current plan: " + waypoints);
		    owner.gui.routeTabModel.fireTableDataChanged();
	    	move();
		} else {
		    // we are at home and can not communicate...
		    owner.logSevere(" ++++ Driver can not communicate locally. Driver: " + owner.getAddress().getName() + " loc: " + actualLoc + " home: " + homeLoc);
		}
	 }

	
	/**
	 * Handles configuration of the vehicle.
	 * @param ed
	 * @param velocity
	 * @param consumption
	 */
	public void handleConfiguration(EntityDescriptor ed, DriverAgentSimulationData data) {
		this.ownerData = data;
		this.actualLoc = ed.confParamsString.get(AcrossDataConstants.ENTITY_START_NODE);
		  
		mapNodes = ((AcrossMapNodes) ed.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_NODES)).getNode();
		mapArcs = ((AcrossMapArcs) ed.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_ARCS)).getArc();
		double[][] mapDistances = ((AcrossMapDistances) ed.typeDescriptor.userObjects.get(AcrossDataConstants.MAP_DISTANCES)).getDistance();
		  
		//Astar pathfinding
		RoadEvaluator re = new RoadEvaluator() {

			public double getPrice(Arc arc) {
				return (1/arc.getQuality()) * arc.getLength() * ownerData.consumption/100;
			}

			public double getTime(Arc arc) {
				return 10*arc.getLength()/ownerData.velocity;
			}
				  
		  };
		  this.roadEval = re;
			  
		  pathfinding = new AStarModPathfinding(mapNodes,mapArcs,mapDistances,re,barredRoads);
	}
	
	/**
	 * Sets the node where the vehicle goes where it does not have anything to do.
	 * @param node
	 */
	public void setHomeNode(String node) {
	   homeLoc = node;
	}
	 
	/**
	 * Tells whether the vehicle is currently following some route (carrying out a
	 * mission) or not.
	 * @return
	 */
	public boolean isOnRoute() {
	   return waypoints.size()!=0;
	}
	
	/**
	 * Tells whether the driver is home.
	 * @return
	 */
	public boolean isHome() {
		return actualLoc.equals(homeLoc) && !isOnRoute();
	}
	
	/** Tells whether the vehicle is going through the road specified by the start and end node */
	public boolean isVehicleGoingThroughRoad(String node1, String node2) {
		// we are going through the nodes in either direction
		if((node1.equals(actualLoc) && node2.equals(headingTo)) ||  (node1.equals(headingTo) && node2.equals(actualLoc))) {
			return true;
		}
		// we are standing at the specified location
		else if (!isOnRoute() && (node1.equals(actualLoc) && node1.equals(node2))) {
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * Tries to plan a route and returns information about it.
	 * @param to
	 * @param byPrice
	 * @return
	 */
	public RouteInfo getRouteInformation(TransportOrder to, boolean byPrice) {
		TransportRoute route = planRoute(to, byPrice);
		if(route!=null  && route.getPlanned().getWaypoint().size()!=0) {
			return route.getRoute().getRouteInfo();
		} else {
			return null;
		}
	}
	
	/**
	 * Carries out a route if it is possible.
	 * @param to
	 * @param byPrice - specifies what to take in consideration when planning the route
	 * @return
	 */
	public RouteInfo carryOutRoute(TransportOrder to, boolean byPrice) {
		TransportRoute route = planRoute(to, byPrice);
		if(route!=null && route.getPlanned().getWaypoint().size()!=0) {
			commitedTransports.getWaypoint().addAll(route.getPlanned().getWaypoint());
			waypoints.addAll(route.getRoute().getWaypoints());
			move();
			return route.getRoute().getRouteInfo();
		} else {
			return null;
		}
	}
	
	/**
	 * Called when an obstacle has been detected.
	 * 
	 */
	public void obstacleDetected(SimulationObjectDescription sod) {
		if(!retreating) {
			controlStop();
		}
		replanRouteRoadBarred(sod);
	}
	
	/**
	 * When there is an obstacle on the road this function replans the route from the
	 * last node reached.
	 *
	 */
	protected void replanRouteRoadBarred(SimulationObjectDescription sod) {
		// if currently retreating from the obstacle do nothing
		if(retreating) {
			return;
		}
		// else we have just started retreating
		else {
			retreating = true;
		}
		
		// clear the planned waypoints
		waypoints.clear();
		
		// retreat
		Waypoint actWaypoint = new Waypoint(actualLoc);
		waypoints.add(actWaypoint);
		
		// mark the road barred
		String barredNode = headingTo;
		for (Arc arc : mapArcs) {
			if((arc.getFrom().equals(actualLoc)) && (arc.getTo().equals(barredNode)) 
				|| (arc.getTo().equals(actualLoc)) && (arc.getFrom().equals(barredNode))) {
				barredRoads.add(arc);
				break;
			}
		}
		
		// replan the route
		TransportRoute r = planRoute(commitedTransports, true);
		if(r==null || r.getPlanned().getWaypoint().size()==0) {
			r = planRoute(commitedTransports,false);
		} 
		
		// set the new waypoints 
		waypoints.addAll(r.getRoute().getWaypoints());
		
		// process moving
		move();
	}
	

    /**
     * Plans the TransportOrder and calculates price and time.
     * @param to
     * @param byPrice
     * @return
     */
    protected TransportRoute planRoute(TransportOrder to, boolean byPrice){
      String startPoint;
      TransportRoute tr = new TransportRoute();
      tr.setNotPlanned(new TransportOrder());
      tr.setPlanned(new TransportOrder());
      Route route = new Route();
      
      if (waypoints.size() == 0) {
      	startPoint = new String(actualLoc);
      } else {
      	startPoint = new String(((Waypoint)waypoints.getLast()).getLocationName());
      }

      double taskTime = 0;
      double taskPrice = 0;

      for (Waypoint wp : to.getWaypoint()) {
        long lowestTime = getLowestDeliveryTime(wp);
        Route oneRoute;
        if (byPrice) {
      	  oneRoute = pathfinding.planRoute(startPoint, wp.getLocationName(), 0, 1);
        } else {
      	  oneRoute = pathfinding.planRoute(startPoint, wp.getLocationName(), 1, 0);
        }
        if (oneRoute == null) {
        	tr.getNotPlanned().getWaypoint().add(wp);
        	continue;
        }
        if ( (byPrice) && ((taskTime + oneRoute.getRouteInfo().routeTime) > lowestTime) ) {
        	tr.getNotPlanned().getWaypoint().add(wp);
        	continue;
        }
        taskTime += oneRoute.getRouteInfo().routeTime;
        taskPrice += oneRoute.getRouteInfo().routePrice;
        startPoint = wp.getLocationName();
        // add all intermediate waypoints (with empty request lists)
        route.addWaypoints(oneRoute.getWaypoints());
        route.addWaypoint(wp);  // add the actual waypoint
        tr.getPlanned().getWaypoint().add(wp);
      }
      route.setRouteInfo(new RouteInfo(taskTime, taskPrice));
      tr.setRoute(route);
      return tr;
    }
    
	
  /**
   * Removes the transports that have been already carried out.
   *
   */
  private void removeDoneWaypointFromTransportOrder(List<RequestList> requestList) {
	  for (Iterator<Waypoint> iter = commitedTransports.getWaypoint().iterator(); iter.hasNext();) {
		  Waypoint way = iter.next();
		  for (Iterator iterator = way.getRequestList().iterator(); iterator.hasNext();) {
			RequestList rl = (RequestList) iterator.next();
			if(requestList.contains(rl)) {
				iterator.remove();
			}
		  }
		  if(way.getRequestList()!=null && way.getRequestList().size()==0) {
			  iter.remove();
		  }
	  }
  }
    /**
     * Looks for the lowest batch delivery time in a given location
     * @param wp Waypoint
     * @return long Lowest batch delivery time in this location
     */
    protected long getLowestDeliveryTime(Waypoint wp){
      long lowestTime = Long.MAX_VALUE;
      for (RequestList rl : wp.getRequestList()) {
        for (Batch batch : rl.getBatch()) {
          if (batch.getDeliveryTime() < lowestTime) {
        	  lowestTime = batch.getDeliveryTime();
          }
        }
      }
      return lowestTime;
    }

    /**
     * Refreshes the barred roads so that now the driver thinks they are 
     * not barred anymore.
     *
     */
    private void refreshBarred() {
    	barredRoads.clear();
    }
    
    /** Sends control topic to simulation to stop the vehicle. */
    protected void controlStop() {
		owner.gisShell.submitTopicToServer(ClientServerTopicConstants.TOPIC_ENTITY_CONTROL, null, AcrossControlConstants.STOP);
	}
    
	    
}
